/***********************************************************
 * $Id: $
 * 
 * Utility code for dealing with odf files using odfdom etc.
 * http://www.clazzes.org
 *
 * Created: 09.04.2015
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.odf.util.factory;

import java.io.InputStream;
import java.util.Locale;

import org.clazzes.odf.util.core.OdfContext;
import org.odftoolkit.odfdom.doc.OdfDocument;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.odftoolkit.odfdom.doc.OdfTextDocument;
import org.odftoolkit.odfdom.dom.element.meta.MetaGeneratorElement;
import org.odftoolkit.odfdom.dom.element.office.OfficeBodyElement;
import org.odftoolkit.odfdom.dom.element.office.OfficeDocumentElement;
import org.odftoolkit.odfdom.dom.element.office.OfficeMetaElement;
import org.odftoolkit.odfdom.dom.element.office.OfficeStylesElement;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeAutomaticStyles;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.w3c.dom.Node;

public class DocumentFactory {
    
    
    public static <T> OdfTextDocument constructTextDocument(String resourcePath, Class<T> loaderClass, Locale locale) {
        try {
            ClassLoader classLoader = loaderClass.getClassLoader();
            InputStream stream = classLoader.getResourceAsStream(resourcePath);
            OdfTextDocument document = (OdfTextDocument) OdfDocument.loadDocument(stream);
            document.setLocale(locale);
            
            return document;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }        
    }
    
    public static <T> OdfSpreadsheetDocument constructSpreadsheetDocument(String resourcePath, Class<T> loaderClass, Locale locale) {
        try {
            Module module = loaderClass.getModule();
            InputStream stream = module.getResourceAsStream(resourcePath);
            OdfSpreadsheetDocument document = (OdfSpreadsheetDocument) OdfDocument.loadDocument(stream);
            document.setLocale(locale);
            
            return document;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }        
    }
    
    public static OdfContext constructDefaultOfficeDocumentElement(Node parentNode, String generatorString, String stylePrefix) {
        
        OfficeDocumentElement documentElement = DocumentFactory.constructOfficeDocumentElement(parentNode);
        OfficeMetaElement metaElement = DocumentFactory.constructOfficeMetaElement(documentElement);
        DocumentFactory.constructMetaGeneratorElement(metaElement, generatorString);
        
        OfficeStylesElement officeStylesElement = DocumentFactory.constructOfficeStylesElement(documentElement);
        OdfOfficeAutomaticStyles officeAutomaticStylesElement = DocumentFactory.constructOfficeAutomaticStylesElement(documentElement);
        OfficeBodyElement officeBodyElement = DocumentFactory.constructOfficeBodyElement(documentElement);

        return new OdfContext(documentElement, stylePrefix, officeStylesElement, officeAutomaticStylesElement, officeBodyElement);
    }
    
    public static OfficeDocumentElement constructOfficeDocumentElement(Node parentNode) {
        OfficeDocumentElement documentElement = (OfficeDocumentElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(OfficeDocumentElement.class);

        documentElement.setAttributeNS(null, "xmlns:office", "urn:oasis:names:tc:opendocument:xmlns:office:1.0");
        documentElement.setAttributeNS(null, "xmlns:style", "urn:oasis:names:tc:opendocument:xmlns:style:1.0");
        documentElement.setAttributeNS(null, "xmlns:text", "urn:oasis:names:tc:opendocument:xmlns:text:1.0");
        documentElement.setAttributeNS(null, "xmlns:table", "urn:oasis:names:tc:opendocument:xmlns:table:1.0");
        documentElement.setAttributeNS(null, "xmlns:draw", "urn:oasis:names:tc:opendocument:xmlns:drawing:1.0");
        documentElement.setAttributeNS(null, "xmlns:fo", "urn:oasis:names:tc:opendocument:xmlns:xsl-fo-compatible:1.0");
        documentElement.setAttributeNS(null, "xmlns:xlink", "http://www.w3.org/1999/xlink");
        documentElement.setAttributeNS(null, "xmlns:dc", "http://purl.org/dc/elements/1.1/");
        documentElement.setAttributeNS(null, "xmlns:meta", "urn:oasis:names:tc:opendocument:xmlns:meta:1.0");
        documentElement.setAttributeNS(null, "xmlns:number", "urn:oasis:names:tc:opendocument:xmlns:datastyle:1.0");
        documentElement.setAttributeNS(null, "xmlns:svg", "urn:oasis:names:tc:opendocument:xmlns:svg-compatible:1.0");
        documentElement.setAttributeNS(null, "xmlns:chart", "urn:oasis:names:tc:opendocument:xmlns:chart:1.0");
        documentElement.setAttributeNS(null, "xmlns:dr3d", "urn:oasis:names:tc:opendocument:xmlns:dr3d:1.0");
        documentElement.setAttributeNS(null, "xmlns:form", "urn:oasis:names:tc:opendocument:xmlns:form:1.0");
        documentElement.setAttributeNS(null, "xmlns:script", "urn:oasis:names:tc:opendocument:xmlns:script:1.0");
        documentElement.setAttributeNS(null, "xmlns:ooo", "http://openoffice.org/2004/office");
        documentElement.setAttributeNS(null, "xmlns:ooow", "http://openoffice.org/2004/writer");
        documentElement.setAttributeNS(null, "xmlns:oooc", "http://openoffice.org/2004/calc");
        documentElement.setAttributeNS(null, "xmlns:dom", "http://www.w3.org/2001/xml-events");
        documentElement.setAttributeNS(null, "xmlns:xforms", "http://www.w3.org/2002/xforms");
        documentElement.setAttributeNS(null, "xmlns:xsd", "http://www.w3.org/2001/XMLSchema");
        documentElement.setAttributeNS(null, "xmlns:xsi", "http://www.w3.org/2001/XMLSchema-instance");
        documentElement.setAttributeNS(null, "xmlns:rpt", "http://openoffice.org/2005/report");
        documentElement.setAttributeNS(null, "xmlns:of", "urn:oasis:names:tc:opendocument:xmlns:of:1.2");
        documentElement.setAttributeNS(null, "xmlns:xhtml", "http://www.w3.org/1999/xhtml");
        documentElement.setAttributeNS(null, "xmlns:grddl", "http://www.w3.org/2003/g/data-view#");
        documentElement.setAttributeNS(null, "xmlns:tableooo", "http://openoffice.org/2009/table");
        documentElement.setAttributeNS(null, "xmlns:chartooo", "http://openoffice.org/2010/chart");
        documentElement.setAttributeNS(null, "xmlns:drawooo", "http://openoffice.org/2010/draw");
        documentElement.setAttributeNS(null, "xmlns:calcext", "urn:org:documentfoundation:names:experimental:calc:xmlns:calcext:1.0");
        documentElement.setAttributeNS(null, "xmlns:loext", "urn:org:documentfoundation:names:experimental:office:xmlns:loext:1.0");
        documentElement.setAttributeNS(null, "xmlns:field", "urn:openoffice:names:experimental:ooo-ms-interop:xmlns:field:1.0");
        documentElement.setAttributeNS(null, "xmlns:formx", "urn:openoffice:names:experimental:ooxml-odf-interop:xmlns:form:1.0");
        documentElement.setAttributeNS(null, "xmlns:css3t", "http://www.w3.org/TR/css3-text/");

        documentElement.setOfficeVersionAttribute("1.2");
        documentElement.setOfficeMimetypeAttribute("application/vnd.oasis.opendocument.chart");
        
        parentNode.appendChild(documentElement);
        return documentElement;
    }
    
    public static OfficeMetaElement constructOfficeMetaElement(Node parentNode) {
        OfficeMetaElement metaElement = (OfficeMetaElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(OfficeMetaElement.class);
        parentNode.appendChild(metaElement);
        return metaElement;
    }
    
    public static MetaGeneratorElement constructMetaGeneratorElement(Node parentNode, String generatorString) {
        MetaGeneratorElement metaGeneratorElement = (MetaGeneratorElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(MetaGeneratorElement.class);
        metaGeneratorElement.setTextContent(generatorString);
        parentNode.appendChild(metaGeneratorElement);
        return metaGeneratorElement;
    }
    
    public static OfficeStylesElement constructOfficeStylesElement(Node parentNode) {
        OfficeStylesElement officeStylesElement = (OfficeStylesElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(OfficeStylesElement.class);        
        parentNode.appendChild(officeStylesElement);
        return officeStylesElement;
    }
    
    public static OdfOfficeAutomaticStyles constructOfficeAutomaticStylesElement(Node parentNode) {
        OdfOfficeAutomaticStyles officeAutomaticStylesElement = (OdfOfficeAutomaticStyles)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(OdfOfficeAutomaticStyles.class);        
        parentNode.appendChild(officeAutomaticStylesElement);
        return officeAutomaticStylesElement;
    }    
    
    public static OfficeBodyElement constructOfficeBodyElement(Node parentNode) {
        OfficeBodyElement officeBodyElement = (OfficeBodyElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(OfficeBodyElement.class);        
        parentNode.appendChild(officeBodyElement);
        return officeBodyElement;
    }
}
