/***********************************************************
 * $Id: $
 *
 * Utility code for dealing with odf files using odfdom etc.
 * http://www.clazzes.org
 *
 * Created: 03.08.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.odf.util.chart;

import java.util.ArrayList;
import java.util.List;

import org.clazzes.odf.util.style.NumberStyleFactory;
import org.clazzes.odf.util.style.Styles;
import org.clazzes.odf.util.table.FancyOdfTable;
import org.clazzes.odf.util.table.FancyOdfTableColumn;
import org.clazzes.odf.util.table.TableFactory;
import org.clazzes.odf.util.voc.StyleFamily;
import org.odftoolkit.odfdom.dom.element.chart.ChartAxisElement;
import org.odftoolkit.odfdom.dom.element.chart.ChartChartElement;
import org.odftoolkit.odfdom.dom.element.chart.ChartPlotAreaElement;
import org.odftoolkit.odfdom.dom.element.chart.ChartSeriesElement;
import org.odftoolkit.odfdom.dom.element.office.OfficeChartElement;
import org.odftoolkit.odfdom.dom.element.style.StyleChartPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleGraphicPropertiesElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowElement;
import org.w3c.dom.Node;

/** Note: The xml we construct here is based on the document referenced here:
 *  https://confluence.clazzes.org/display/OOOENGINE/Reference+documents
 *  (or directly to the attachment: https://confluence.clazzes.org/download/attachments/12320819/Test.xml?version=1&amp;modificationDate=1501836386000&amp;api=v2)
 *
 *  Things like style names ("ch5") refer to parts of that xml.
 */
public class OdfChart<T extends OdfChartDataPoint> {

    private List<OdfChartColumn<T>> columns = new ArrayList<OdfChartColumn<T>>();
    public void addColumn(OdfChartColumn<T> chartColumn) {
        columns.add(chartColumn);
    }

    private String width;
    public void setWidth(String width) {
        this.width = width;
    }

    private String height;
    public void setHeight(String height) {
        this.height = height;
    }

    private String chartClass;
    public void setChartClass(String chartClass) {
        this.chartClass = chartClass;
    }

    private String legendPosition;
    public void setLegendPosition(String legendPosition) {
        this.legendPosition = legendPosition;
    }

    private String legendX;
    public void setLegendX(String legendX) {
        this.legendX = legendX;
    }

    private String legendY;
    public void setLegendY(String legendY) {
        this.legendY = legendY;
    }

    private String legendExpansion;
    public void setLegendExpansion(String legendExpansion) {
        this.legendExpansion = legendExpansion;
    }

    private String plotAreaX;
    public void setPlotAreaX(String plotAreaX) {
        this.plotAreaX = plotAreaX;
    }

    private String plotAreaY;
    public void setPlotAreaY(String plotAreaY) {
        this.plotAreaY = plotAreaY;
    }

    private String plotAreaWidth;
    public void setPlotAreaWidth(String plotAreaWidth) {
        this.plotAreaWidth = plotAreaWidth;
    }

    private String plotAreaHeight;
    public void setPlotAreaHeight(String plotAreaHeight) {
        this.plotAreaHeight = plotAreaHeight;
    }

    private String coordRegionX;
    public void setCoordRegionX(String coordRegionX) {
        this.coordRegionX = coordRegionX;
    }

    private String coordRegionY;
    public void setCoordRegionY(String coordRegionY) {
        this.coordRegionY = coordRegionY;
    }

    private String coordRegionWidth;
    public void setCoordRegionWidth(String coordRegionWidth) {
        this.coordRegionWidth = coordRegionWidth;
    }

    private String coordRegionHeight;
    public void setCoordRegionHeight(String coordRegionHeight) {
        this.coordRegionHeight = coordRegionHeight;
    }

    private static final String TABLE_NAME = "local-table";

    public void render(Styles stylesStyles, Styles contentAutomaticStyles, Node parentNode, List<T> dataPoints) {
        // ch8 to ch11 are for chart:data-points; ch12 and ch13 are apparently unused
        String numberStyle = this.getChartNumberStyle(stylesStyles);              // N0
        String dateStyle = this.getChartDateStyle(stylesStyles);                  // N10036
        String chartStyle = this.getChartStyle(stylesStyles);                     // ch1
        String legendStyle = this.getLegendStyle(stylesStyles);                   // ch2
        String plotAreaStyle = this.getPlotAreaStyle(stylesStyles);               // ch3
        String xAxisStyle = this.getXAxisStyle(stylesStyles, dateStyle);          // ch4
        String yAxisStyle = this.getYAxisStyle(stylesStyles, numberStyle);        // ch5
        String yAxisGridStyle = this.getYAxisGridStyle(stylesStyles);             // ch6
        String seriesStyle = this.getChartSeriesStyle(stylesStyles, numberStyle); // ch7
        String wallStyle = this.getWallStyle(stylesStyles);                       // ch14
        String floorStyle = this.getFloorStyle(stylesStyles);                     // ch15

        OfficeChartElement officeChartElement = ChartFactory.constructOfficeChart(parentNode);
        ChartChartElement chartChartElement = ChartFactory.constructChartChart(officeChartElement, this.width, this.height, this.chartClass, chartStyle);
        ChartFactory.constructChartLegend(chartChartElement, this.legendPosition, this.legendX, this.legendY, this.legendExpansion, legendStyle);
        ChartPlotAreaElement plotAreaElement = ChartFactory.constructChartPlotArea(chartChartElement,
                this.plotAreaX, this.plotAreaY, this.plotAreaWidth, this.plotAreaHeight, "both", plotAreaStyle);
        // TODO: odfdom API for <chartooo:coordinate-region svg:x="2.785cm" svg:y="0.18cm" svg:width="8.64cm" svg:height="8.64cm"/> (should be child of plotArea) seems to be missing

        ChartAxisElement xAxis = ChartFactory.constructChartAxis(plotAreaElement, "x", "primary-x", xAxisStyle);
        ChartFactory.constructChartCategories(xAxis, TABLE_NAME, this.columns.size());
        ChartAxisElement yAxis = ChartFactory.constructChartAxis(plotAreaElement, "y", "primary-y", yAxisStyle);
        ChartFactory.constructChartGrid(yAxis, "major", yAxisGridStyle);
        ChartSeriesElement chartSeriesElement = ChartFactory.constructChartSeries(plotAreaElement, TABLE_NAME, this.columns.size(), "chart:circle", seriesStyle);

        for (T dataPoint : dataPoints) {
            String dataPointStyle = dataPoint.getDataPointStyle(contentAutomaticStyles);
            ChartFactory.constructChartDataPoint(chartSeriesElement, dataPointStyle);
        }

        ChartFactory.constructChartWall(plotAreaElement, wallStyle);
        ChartFactory.constructChartFloor(plotAreaElement, floorStyle);

        this.constructDataTable(chartChartElement, stylesStyles, dataPoints);
    }

    protected String getChartNumberStyle(Styles styles) {
        return styles.getStyle(StyleFamily.NUMBER,
                    NumberStyleFactory.constructNumberNumberStyleElement(styles),
                    NumberStyleFactory.constructNumberNumberElement(styles, 1));
    }

    protected String getChartDateStyle(Styles styles) {
        return styles.getStyle(StyleFamily.DATE,
                NumberStyleFactory.constructNumberDateStyleElement(styles, "de", "AT", true),
                NumberStyleFactory.constructNumberDayElement(styles, "long"),
                NumberStyleFactory.constructNumberTextElement(styles, "."),
                NumberStyleFactory.constructNumberMonthElement(styles, "long"),
                NumberStyleFactory.constructNumberTextElement(styles, "."),
                NumberStyleFactory.constructNumberYearElement(styles, "long"));
    }

    /** Constructs the chart:style-name assigned to the chart:chart element.
     */
    protected String getChartStyle(Styles styles) {
        return styles.getChartStyle(styles.constructGraphicPropertiesWithStrokeAndFill("none", "none"));
    }

    /** Constructs the chart:style-name assigned to the chart:legend element.
     */
    protected String getLegendStyle(Styles styles) {
        return styles.getChartStyle(
                styles.constructChartPropertiesWithAutoPosition(true),
                styles.constructGraphicPropertiesWithStrokeAndFillAndColor("none", "#b3b3b3", "none", "#e6e6e6"),
                styles.constructTextPropertiesWithFontSize("10pt"));

    }

    /** Constructs the chart:style-name assigned to the chart:plot-area element.
     */
    protected String getPlotAreaStyle(Styles styles) {
        StyleChartPropertiesElement chartProperties = styles.constructChartProperties();
        chartProperties.setChartIncludeHiddenCellsAttribute(false);
        chartProperties.setChartAutoPositionAttribute(true);
        chartProperties.setChartAutoSizeAttribute(true);
        chartProperties.setChartRightAngledAxesAttribute(true);
        return styles.getChartStyle(chartProperties);
    }

    /** Constructs the chart:style-name assigned to the chart:axis element for the x axis.
     */
    protected String getXAxisStyle(Styles styles, String dataStyleName) {
        StyleChartPropertiesElement chartProperties = styles.constructChartProperties();
        chartProperties.setChartDisplayLabelAttribute(true);
        chartProperties.setChartLogarithmicAttribute(false);
        chartProperties.setChartReverseDirectionAttribute(true);
        chartProperties.setTextLineBreakAttribute(false);
        chartProperties.setChartLinkDataStyleToSourceAttribute(true);;
        chartProperties.setChartAxisPositionAttribute(0d);

        return styles.getChartStyleWithDataStyle(dataStyleName,
                chartProperties,
                styles.constructGraphicPropertiesWithColor("#b3b3b3", null),
                styles.constructTextPropertiesWithFontSize("10pt"));
    }

    /** Constructs the chart:style-name assigned to the chart:axis element for the y axis.
     */
    protected String getYAxisStyle(Styles styles, String dataStyleName) {
        StyleChartPropertiesElement chartProperties = styles.constructChartProperties();
        chartProperties.setChartDisplayLabelAttribute(true);
        chartProperties.setChartLogarithmicAttribute(false);
        chartProperties.setChartReverseDirectionAttribute(false);
        chartProperties.setTextLineBreakAttribute(false);
        chartProperties.setChartLinkDataStyleToSourceAttribute(true);;
        chartProperties.setChartAxisPositionAttribute(0d);

        return styles.getChartStyleWithDataStyle(dataStyleName,
                chartProperties,
                styles.constructGraphicPropertiesWithColor("#b3b3b3", null),
                styles.constructTextPropertiesWithFontSize("10pt"));
    }

    /** Constructs the chart:style-name assigned to the chart:grid element inside the chart:axis element for the y axis.
     */
    protected String getYAxisGridStyle(Styles styles) {
        return styles.getChartStyle(styles.constructGraphicPropertiesWithColor("#b3b3b3", null));
    }

    /** Constructs the chart:style-name assigned to the chart:series element.
     */
    protected String getChartSeriesStyle(Styles styles, String dataStyleName) {
        StyleChartPropertiesElement chartProperties = styles.constructChartProperties();
        chartProperties.setChartLinkDataStyleToSourceAttribute(true);

        StyleGraphicPropertiesElement graphicProperties = styles.constructGraphicProperties();
        graphicProperties.setDrawStrokeAttribute("none");
        graphicProperties.setDrawFillColorAttribute("#004586");
        graphicProperties.setDr3dEdgeRoundingAttribute("5%");

        return styles.getChartStyleWithDataStyle(dataStyleName,
                chartProperties,
                graphicProperties,
                styles.constructTextPropertiesWithFontSize("10pt"));
    }

    /** Constructs the chart:style-name assigned to a chart:data-point inside a chart-series element.
     */
    protected String getDataPointStyle(Styles styles, String fillColor) {
        return styles.getChartStyle(styles.constructGraphicPropertiesWithColor(null, fillColor));
    }

    /** Constructs the chart:style-name assigned to the chart:wall element.
     */
    protected String getWallStyle(Styles styles) {
        return styles.getChartStyle(
                styles.constructGraphicPropertiesWithStrokeAndFillAndColor("solid", "#b3b3b3", "none", "#e6e6e6"));
    }

    /** Constructs the chart:style-name assigned to the chart:floor element.
     */
    protected String getFloorStyle(Styles styles) {
        return styles.getChartStyle(
                styles.constructGraphicPropertiesWithColor("#b3b3b3", "#cccccc"));
    }

    protected void constructDataTable(Node parentNode, Styles styles, List<T> data) {
        List<FancyOdfTableColumn<T>> fancyTableColumns = new ArrayList<FancyOdfTableColumn<T>>();

        for (OdfChartColumn<T> chartColumn : this.columns) {
            final OdfChartColumn<T> localChartColumn = chartColumn;
            fancyTableColumns.add(new FancyOdfTableColumn<T>(chartColumn.isHeaderColumn(), chartColumn.getName(), chartColumn.getLabel()) {

                public void appendCell(TableTableRowElement rowElement, T data) {
                    localChartColumn.appendDataTableCell(rowElement, data);
                }

                public int compare(T dataOne, T dataTwo, boolean descending) {
                    return 0;
                }
            });
        }

        FancyOdfTable<T> table = new FancyOdfTable<T>(TABLE_NAME, fancyTableColumns);
        table.render(parentNode, data, null);
    }
}
