/***********************************************************
 * $Id$
 *
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 06.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.jdbc2xml.tools;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import org.clazzes.jdbc2xml.schema.TableFilter;
import org.clazzes.jdbc2xml.sql.SqlIdentifierMapperFactory;

/**
 * This class holds all properties to customize dumping and restoring
 * database to/from XML files.
 *
 * @author lech
 */
public class ProcessRestrictionFilter implements TableFilter {

    protected String[] excludedTableNames = null;

    protected Map<String,String> primaryRestrictions;

    protected Map<String,String> tableClauses;

    protected boolean processSchema = true;
    protected boolean processData = true;
    protected boolean processConstraints = true;

    protected boolean processXmlSchemaCheck = false;

    protected boolean prettyPrintXml = true;
    private boolean keepInternalIndices = false;
    private boolean createFKIndices = false;
    private boolean transactional = false;
    private boolean checkForAutoIncrementColumns = true;

    private int batchSize = 1000;

    private String idMapper;

    Map<String,String> dialectProps;

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.tools.TableFilter#processTable(java.lang.String)
     */
    public boolean processTable(String tableName) {
        if (tableName==null || tableName.length()==0)
            return false;
        // excluded?
        if (this.excludedTableNames!=null) {
            for (String excludedTable : this.excludedTableNames) {
                if (tableName.equalsIgnoreCase(excludedTable))
                    return false;
            }
        }

        // Temporary backup table ?
        if (tableName.startsWith("JDBC2XML__"))
            return false;

        if (this.tableClauses==null)
            return true;

        if (this.tableClauses.containsKey(tableName.toLowerCase(Locale.ENGLISH))) {
            return true;
        }
        return false;
    }

    /**
     * @return the excludedTableNames
     */
    public String[] getExcludedTableNames() {
        return this.excludedTableNames;
    }
    /**
     * @param excludedTableNames the excludedTableNames to set
     */
    public void setExcludedTableNames(String[] excludedTableNames) {
        this.excludedTableNames = excludedTableNames;
    }
    /**
     * @return the processData
     */
    public boolean isProcessData() {
        return this.processData;
    }
    /**
     * @param processData the processData to set
     */
    public void setProcessData(boolean processData) {
        this.processData = processData;
    }
    /**
     * @return the processConstraints
     */
    public boolean isProcessConstraints() {
        return this.processConstraints;
    }
    /**
     * @param processConstraints the processConstraints to set
     */
    public void setProcessConstraints(boolean processConstraints) {
        this.processConstraints = processConstraints;
    }
    /**
     * @return the batchSize
     */
    public int getBatchSize() {
        return this.batchSize;
    }
    /**
     * @param batchSize the batchSize to set
     */
    public void setBatchSize(int batchSize) {
        this.batchSize = batchSize;
    }

    /**
     * @return the processXmlSchemaCheck
     */
    public boolean isProcessXmlSchemaCheck() {
        return this.processXmlSchemaCheck;
    }

    /**
     * @param processXmlSchemaCheck the processXmlSchemaCheck to set
     */
    public void setProcessXmlSchemaCheck(boolean processXmlSchemaCheck) {
        this.processXmlSchemaCheck = processXmlSchemaCheck;
    }

    /**
     * @return the prettyPrintXml
     */
    public boolean isPrettyPrintXml() {
        return this.prettyPrintXml;
    }

    /**
     * @param prettyPrintXml the prettyPrintXml to set
     */
    public void setPrettyPrintXml(boolean prettyPrintXml) {
        this.prettyPrintXml = prettyPrintXml;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.IndexFilter#isKeepInternalIndices()
     */
    public boolean isKeepInternalIndices() {
        return this.keepInternalIndices;
    }

    /**
     * @param keepInternalIndices The keepInternalIndices flag to set.
     */
    public void setKeepInternalIndices(boolean keepInternalIndices) {
        this.keepInternalIndices = keepInternalIndices;
    }


    /**
     * @return Whether to create an index on each foreign key column of an imported table.
     *         Some database engines like MySQL create such an index by default, others
     *         might show performance improvements, if such an index is manually created.
     */
    public boolean isCreateFKIndices() {
        return this.createFKIndices;
    }

    /**
     * @param createFKIndices Set whether to create an index on each foreign key column of
     *         an imported table. Some database engines like MySQL create such an index
     *         by default, others might show performance improvements, if such an index is
     *         manually created.
     */
    public void setCreateFKIndices(boolean createFKIndices) {
        this.createFKIndices = createFKIndices;
    }

    /**
     * @return The restrictions on the primary key of a table.
     *         The keys of the map are table names, whose data is restricted.
     *         The values are the primary key values to which the table data is restricted.
     *         If the primary key value is null, the table data should be omitted and
     *         foreign key columns in tables, which import the primary key of the given table
     *         should be set to null.
     */
    public Map<String, String> getPrimaryRestrictions() {
        return this.primaryRestrictions;
    }

    /**
     * @param primaryRestrictions the primaryRestrictions to set
     */
    public void setPrimaryRestrictions(Map<String, String> primaryRestrictions) {
        this.primaryRestrictions = primaryRestrictions;
    }

    /**
     * Add a data restriction to a given table.
     *
     * @param table The table which shall be restricted.
     * @param pkValue The primary key value which restricts the table data.
     *                If this value is null, the nullable foreign keys of the
     *                table are not used to restrict the table data, but are set
     *                to null, if the dependent dataset is not present in the target
     *                database.
     */
    public void addPrimaryRestriction(String table, String pkValue) {
        if (this.primaryRestrictions == null)
            this.primaryRestrictions = new HashMap<String, String>();

        this.primaryRestrictions.put(table,pkValue);
    }

    /**
     * @return The key of the mapper used to transform SQL identifier to
     *         their XML representation.
     *
     * @see SqlIdentifierMapperFactory#newMapper(String)
     */
    public String getIdMapper() {
        return this.idMapper;
    }

    /**
     * @param idMapper The of the mapper used to transform SQL identifier to
     *         their XML representation to set.
     *
     * @see SqlIdentifierMapperFactory#newMapper(String)
     */
    public void setIdMapper(String idMapper) {
        this.idMapper = idMapper;
    }

    /**
     * @return The transactional flag. If set to <code>true</code>, database dumps are
     *         generated inside a read-only transaction.
     */
    public boolean isTransactional() {
        return this.transactional;
    }

    /**
     * @param transactional The transactional flag to set.
     */
    public void setTransactional(boolean transactional) {
        this.transactional = transactional;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.IndexFilter#isCheckForAutoIncrementColumns()
     */
    public boolean isCheckForAutoIncrementColumns() {
        return this.checkForAutoIncrementColumns;
    }

    /**
     * @param checkForAutoIncrementColumns Whether to check for auto-increment columns.
     * @see #isCheckForAutoIncrementColumns()
     */
    public void setCheckForAutoIncrementColumns(boolean checkForAutoIncrementColumns) {
        this.checkForAutoIncrementColumns = checkForAutoIncrementColumns;
    }
    /**
     * @return Whether to process the schema or process data only.
     */
    public boolean isProcessSchema() {
        return this.processSchema;
    }
    /**
     * @param processSchema Whether to process the schema or process data only.
     */
    public void setProcessSchema(boolean processSchema) {
        this.processSchema = processSchema;
    }

    /**
     * Dialect-specific properties like <code>overflowTablespace</code> for
     * oracle.
     * @return The map of dialect properties.
     */
    public Map<String, String> getDialectProps() {
        return this.dialectProps;
    }

    public void setDialectProps(Map<String, String> dialectProps) {
        this.dialectProps = dialectProps;
    }

    public void addDialectProp(String key, String value) {

        if (this.dialectProps == null) {
            this.dialectProps = new HashMap<String,String>();
        }
        this.dialectProps.put(key,value);
    }

    /**
     * @return A map of lowercase table names to use. If the value of the
     *         the map under the table nams is not <code>null</code>,
     *         it is used as a where clauses used when dumping the
     *         data of the table.
     */
    public Map<String, String> getTableClauses() {
        return this.tableClauses;
    }

    /**
     * @param whereClauses A map of lowercase table names to where clauses used when
     *         dumping the data to set.
     */
    public void setTableClauses(Map<String, String> whereClauses) {
        this.tableClauses = whereClauses;
    }

    /**
     * @param table A table name.
     * @param whereClause a where clause used when dumping the
     *         data of the table.
     */
    public void addTableClause(String table, String whereClause) {

        if (this.tableClauses == null) {
            this.tableClauses = new HashMap<String,String>();
        }

        this.tableClauses.put(table.toLowerCase(Locale.ENGLISH),whereClause);
    }

    /**
     * @param table A table name.
     * @return a where clause used when dumping the
     *         data of the table or <code>null</code>, of no such
     *         where clause has been configured.
     */
    public String getTableClause(String table) {

        if (this.tableClauses == null) {
            return null;
        }

        return this.tableClauses.get(table.toLowerCase(Locale.ENGLISH));
    }
}
