/***********************************************************
 * $Id$
 *
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 27.11.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.jdbc2xml.tools;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.TimeZone;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerConfigurationException;

import org.clazzes.jdbc2xml.helper.StreamHelper;
import org.clazzes.jdbc2xml.schema.Dialect;
import org.clazzes.jdbc2xml.schema.DialectFactory;
import org.clazzes.jdbc2xml.schema.IDialectFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

/**
 * Usage example jdbc2xml mysql:
 * java -cp target/classes:/usr/share/java/mysql.jar org.clazzes.jdbc2xml.tools.ConnectionMain --jdbc2xml \
 *      --url jdbc:mysql://localhost:3306/foodb --user root --password secret --to-file foodb.xml
 *
 * Usage example jdbcping mysql:
 * java -cp target/classes:/usr/share/java/mysql.jar org.clazzes.jdbc2xml.tools.ConnectionMain --jdbcping \
 *      --url jdbc:mysql://localhost:3306/foodb --user root --password secret
 *
 * Usage example jdbcping ms sql server (note the " to protect the ; in the URL!):
 * java -cp target/classes:/opt/java/lib/sqljdbc.jar org.clazzes.jdbc2xml.tools.Main --jdbcping \
 *      --url "jdbc:sqlserver://localhost:1433;databaseName=foodb;user=Administrator;password=secret;"
 *
 * As we have not written a wrapper script yet, let's suggest a valid full classpath:
 * target/classes:/opt/java/lib/sqljdbc.jar:/usr/share/java/mysql.jar:/usr/share/java/commons-logging.jar:/usr/share/java/commons-logging-api.jar:~/.m2/repository/org/clazzes/util/0.3.4/util-0.3.4.jar
 *
 * @author wglas
 *
 */
public class Main {

    public static final String syntax=
        "  Syntax 1 (java):     java [opts] org.clazzes.jdbc2xml.tools.Main command options\n"+
        "  Syntax 2 (wrappers): command options\n"+
        "\n"+
        "    commands:\n"+
        "\n"+
        "      [--]jdbc2xml  dumps a DB to a XML file\n"+
        "                    needs --[from-]url jdbcurl --[to-]file file\n"+
        "\n"+
        "      [--]xml2jdbc  restors a DB from a XML file (empty DB expected)\n"+
        "                    needs --[from-]file file --[to-]url jdbcurl\n"+
        "\n"+
        "      [--]jdbc2jdbc copies a DB to another DB\n"+
        "                    needs --from-url jdbcurl --to-url jdbcurl\n"+
        "\n"+
        "      [--]xml2xml   copies a XML file (e.g. to extract a schema-only file)\n"+
        "                    needs --from-file file and --to-file\n"+
        "\n"+
        "      [--]jdbcping  tests a DB connection an prints out some server info\n"+
        "                    needs --[from-]url jdbcurl\n"+
        "\n"+
        "      [--]jdbcextr  extracts parts of a DB to a XML file\n"+
        "                    needs --[from-]url jdbcurl --[to-]file file\n"+
        "\n"+
        "      [--]help      prints this syntax description\n"+
        "                    not available as wrapper, call jdbc2xml --help\n"+
        "\n"+
        "    options prefixes:\n"+
        "      --from-*   specifies the data source\n"+
        "      --to-*     specifies the data destination\n"+
        "      from- resp. to- can be omitted expect where the command requires\n"+
        "      clear distinction, e.g. --to-url and --from-url for jdbc2jdbc\n"+
        "\n"+
        "    options for specifiying db access:\n"+
        "      JDBC URL to access the DB (like jdbc:servertype://host[:port]/db[?options]):\n" +
        "        --from-url \"jdbcurl\"\n"+
        "        --to-url   \"jdbcurl\"\n"+
        "        --url      \"jdbcurl\"\n"+
        "      User to access the DB (some drivers can parse that from the URL):\n"+
        "        --from-user user\n"+
        "        --to-user   user\n"+
        "        --user      user\n"+
        "      Password to access the DB (some drivers can parse that from the URL):\n"+
        "        --from-password password\n"+
        "        --to-password   password\n"+
        "        --password      password\n"+
        "\n"+
        "    options for specifiying an xml file (Win32 wrapper needs absolut paths!):\n"+
        "      --from-file file\n"+
        "      --to-file   file\n"+
        "      --file      file\n"+
        "    The given filenames may contain compressed input. The implementation instantiates\n"+
        "    an appropriate inflating/deflating stream if the filename ends with .gz or .bz2\n"+
        "\n"+
        "    options for restricting the extent of what is being processed:\n"+
        "      --tables table1,table2,...\n"+
        "        only processes on given tables, not on all\n"+
        "        If a table clause contains a colon like\n"+
        "            persons:ID>1234\n"+
        "        the part after the colon is used as where clause during data export.\n"+
        "      --exclude-tables table1,table2,...\n"+
        "        exclude the given tables from beeing processed\n"+
        "      --no-data\n"+
        "        do not process the data rows\n"+
        "      --no-schema\n"+
        "        do not create tables, only insert data\n"+
        "      --no-constraints\n"+
        "        do not create constraints\n"+
        "      --schema-only\n"+
        "        combines --no-data and --no-constraints\n"+
        "      --constraints-only\n"+
        "        only activate constraints\n"+
        "\n"+
        "    expert options\n"+
        "      --drop-tables\n" +
        "        Drop the affected tables before actually importing a dump.\n"+
        "        If --tables or --exclude-tables is given, only tables affected\n"+
        "        by the so-specified filter are dropped.\n"+
        "      --compression <n>\n"+
        "        Overrides the default bzip2 output compression, if --to-file ends with .bz2\n" +
        "      --timezone timezoneid\n"+
        "        forces a timezone\n" +
        "      --from-driver drivername\n"+
        "      --to-driver   drivername\n"+
        "      --driver      drivername\n"+
        "        allows to select a JDBC driver, overriding autoselection\n" +
        "      --jdbc-drivers jarfilename[;jarfilename...]\n"+
        "        tells the unix wrapper scripts to add these to the classpath\n" +
        "      --batch-size n\n" +
        "        for *2jdbc, changes the batch size for inserts (def.: 1000)\n"+
        "      --check-xml-schema\n" +
        "        for xml2* commands, this enables schema checking the xml file\n"+
        "      --pretty, --no-pretty\n" +
        "        for *2xml, forces resp. supresses pretty printing of xml data\n"+
        "      --to-lower\n" +
        "        for *2xml, forces the transformation of SQL identifiers to lower case" +
        "        in the generated XML files.\n"+
        "      --to-upper\n" +
        "        for *2xml, forces the transformation of SQL identifiers to upper case" +
        "        in the generated XML files.\n"+
        "      --keep-internal-indices\n" +
        "        for *2xml, write internal indices, which are internally generated\n"+
        "        by the RDMBS\n"+
        "      --no-auto-increment-check\n" +
        "        for *2xml, skip checks for auto-increment columns and assume all columns\n"+
        "        are not auto-incremented. This can speed up teh dump in some situations.\n"+
        "      --create-fk-indices\n" +
        "        for xml2*, create an extra index on the columns of each each foreign key.\n"+
        "      --transactional\n" +
        "        for jdbc2*, start a read-only transaction while fetching the contents of the database.\n"+
        "      --dialect-property key=value\n" +
        "        Set a dialect-specific property like\n"+
        "          overflowTablespace=MYTBS\n"+
        "        for oracle.\n"+
        "    expert properties of jdbc URLs\n"+
        "      mysql (?p1=v1&p2=v2&...):\n"+
        "        useCursorFetch=true\n"+
        "          forces the driver to use cursor based fetching saving RAM,\n"+
        "          allows fetching very large tables in jdbc2*\n"+
        "      MS SQL Server (;p1=v1;p2=v2;...):\n"+
        "        selectMethod=cursor\n"+
        "          forces the driver to use cursor based fetching saving RAM,\n"+
        "          allows fetching very large tables in jdbc2*\n"+
        // see http://technet.microsoft.com/de-de/library/ms378405.aspx
        //     http://technet.microsoft.com/de-de/library/ms378988.aspx
        "\n"+
        "    verbosity options:\n"+
        "      Usually Fatals, Errors, Warnings are printed, to change this use these:\n"+
        "      --quiet\n" +
        "        suppresses any output, only return code tells about success\n"+
        "      --verbose\n" +
        "        increases verbosity to Info\n"+
        "      --debug\n" +
        "        hides any eventual information between lots of spam lines\n"+
        "      --debug-startup\n" +
        "        invites wrapper scripts to log their startup processing\n"+
        "\n"
        ;

    private static Logger log = LoggerFactory.getLogger(Main.class);

    // direct parameters

    protected static int iarg = 0;

    protected static String command = null;

    protected static String fromFileName = null;
    protected static File fromFile = null;
    protected static InputStream fromIS = null;
    protected static String toFileName = null;
    protected static OutputStream toOS = null;
    protected static Integer compression = null;

    protected static String fromURL = null;
    protected static String fromUser = null;
    protected static String fromPasswd = null;
    protected static String fromDriverName = null;
    protected static Dialect fromDialect = null;
    protected static Connection fromConnection = null;

    protected static String toURL = null;
    protected static String toUser = null;
    protected static String toPasswd = null;
    protected static String toDriverName = null;
    protected static Dialect toDialect = null;
    protected static Connection toConnection = null;

    protected static TimeZone timeZone = TimeZone.getDefault();
    protected static String tempDB = null;

    // computed and combinational parameters

    protected static int verbosity = 0; // 0=quiet=fatal, 1=normal=warning, 2=verbose=info, 3=debug
    // in fact handeld by surrounding scripts by switching between src/test/resources/log4_*.properties

    protected static boolean fromFileNeeded = false;
    protected static boolean toFileNeeded = false;
    protected static boolean fromDBNeeded = false;
    protected static boolean toDBNeeded = false;
    protected static boolean tempDBNeeded = false;
    protected static boolean dropTables = false;

    protected static ProcessRestrictionFilter processRestrictionFilter;

    /**
     * @param args
     * @throws SAXException
     * @throws SQLException
     * @throws TransformerConfigurationException
     * @throws IOException
     * @throws ParserConfigurationException
     */
    public static void main(String[] args) throws TransformerConfigurationException, SQLException, SAXException, IOException, ParserConfigurationException {

        // read command

        if (args.length==0) {
            log.error("No parameters given.\n");
            log.info("The correct syntax is:"+syntax);
            System.exit(1);
        }

        // check, whether we've been called from fancylaunch, which sets -Dapp.name
        command = System.getProperty("app.name");

        if (command != null) {

            if ( "-?".equals(args[iarg])
                    || "--help".equals(args[iarg])
                    || "help".equals(args[iarg])) {
                command="help";
                ++iarg;
            }
        }
        else {
            if ( "--jdbc2xml".equals(args[iarg])
                    || "jdbc2xml".equals(args[iarg])) {
                command="jdbc2xml";
            } else if ( "--xml2jdbc".equals(args[iarg])
                    || "xml2jdbc".equals(args[iarg])) {
                command="xml2jdbc";
            } else if ( "--jdbc2jdbc".equals(args[iarg])
                    || "jdbc2jdbc".equals(args[iarg])) {
                command="jdbc2jdbc";
            } else if ( "--xml2xml".equals(args[iarg])
                    || "xml2xml".equals(args[iarg])) {
                command="xml2xml";
            } else if ( "--jdbcping".equals(args[iarg])
                    || "jdbcping".equals(args[iarg])) {
                command="jdbcping";
            } else if ( "--jdbcextr".equals(args[iarg])
                    || "jdbcextr".equals(args[iarg])) {
                command="jdbcextr";
            } else if ( "-?".equals(args[iarg])
                    || "--help".equals(args[iarg])
                    || "help".equals(args[iarg])) {
                command="help";
            }
            if (command==null) {
                log.error("Command unknown: "+command+"\n");
                log.info("The correct syntax is:"+syntax);
                System.exit(1);
            }
            iarg++;
        }

        if ( "jdbc2xml".equals(command)) {
            fromDBNeeded=true;
            toFileNeeded=true;
        } else if ("xml2jdbc".equals(command)) {
            fromFileNeeded=true;
            toDBNeeded=true;
        } else if ("jdbc2jdbc".equals(command)) {
            fromDBNeeded=true;
            toDBNeeded=true;
        } else if ("xml2xml".equals(command)) {
            fromFileNeeded=true;
            toFileNeeded=true;
        } else if ("jdbcping".equals(command)) {
            fromDBNeeded=true;
        } else if ("jdbcextr".equals(command)) {
            fromDBNeeded=true;
            tempDBNeeded=true;
        }

        if (log.isInfoEnabled()) log.info("Parsed command to: "+command);

        processRestrictionFilter=new ProcessRestrictionFilter();
        // read options

        while (iarg < args.length) {
            if (log.isDebugEnabled()) log.debug("Parsing option "+args[iarg]);
            if ("--batch-size".equals(args[iarg])) {
                processRestrictionFilter.setBatchSize(Integer.valueOf(args[++iarg]));
            } else if ("--check-xml-schema".equals(args[iarg])) {
                processRestrictionFilter.setProcessXmlSchemaCheck(true);
            } else if ("--debug".equals(args[iarg])) {
                if (verbosity < 2)
                    verbosity=2; // in fact handled by surrounding scripts
            } else if ("--debug-startup".equals(args[iarg])) {
                ; // it'd be difficult to filter out wrapper-only parameters in windows
            } else if ("--driver".equals(args[iarg])) {
                fromDriverName = toDriverName = args[++iarg];
            } else if ("--exclude-tables".equals(args[iarg])) {
                processRestrictionFilter.setExcludedTableNames(args[++iarg].split(","));
            } else if ("--from-driver".equals(args[iarg])) {
                fromDriverName = args[++iarg];
            } else if ("--file".equals(args[iarg])) {
                fromFileName = toFileName = args[++iarg];
            } else if ("--from-file".equals(args[iarg])) {
                fromFileName = args[++iarg];
            } else if ("--from-passwd".equals(args[iarg])
                    || "--from-password".equals(args[iarg])) {
                fromPasswd = args[++iarg];
            } else if ("--from-url".equals(args[iarg])) {
                fromURL = args[++iarg];
            } else if ("--from-user".equals(args[iarg])) {
                fromUser = args[++iarg];
            } else if ("--help".equals(args[iarg])
                    || "help".equals(args[iarg])
                    || "-?".equals(args[iarg]))  {
                command="help"; // this helps against wrappers forcing a non-help command
                fromFileNeeded = false;
                toFileNeeded = false;
                fromDBNeeded = false;
                toDBNeeded = false;
                tempDBNeeded = false;
            } else if ("--jdbc-drivers".equals(args[iarg])) {
                ++iarg; // it's difficult to filter out wrapper-only parameters
            } else if ("--keep-internal-indices".equals(args[iarg])) {
                processRestrictionFilter.setKeepInternalIndices(true);
            } else if ("--no-auto-increment-check".equals(args[iarg])) {
                processRestrictionFilter.setCheckForAutoIncrementColumns(false);
            } else if ("--create-fk-indices".equals(args[iarg])) {
                processRestrictionFilter.setCreateFKIndices(true);
            } else if ("--transactional".equals(args[iarg])) {
                processRestrictionFilter.setTransactional(true);
            } else if ("--no-data".equals(args[iarg])) {
                processRestrictionFilter.setProcessData(false);
            } else if ("--no-schema".equals(args[iarg])) {
                processRestrictionFilter.setProcessSchema(false);
                processRestrictionFilter.setProcessConstraints(false);
            } else if ("--no-constraints".equals(args[iarg])) {
                processRestrictionFilter.setProcessConstraints(false);
            } else if ("--constraints-only".equals(args[iarg])) {
                processRestrictionFilter.setProcessSchema(false);
                processRestrictionFilter.setProcessConstraints(true);
            } else if ("--no-pretty".equals(args[iarg])) {
                processRestrictionFilter.setPrettyPrintXml(false);
            } else if ("--to-lower".equals(args[iarg])) {
                processRestrictionFilter.setIdMapper("lower");
            } else if ("--to-upper".equals(args[iarg])) {
                processRestrictionFilter.setIdMapper("upper");
            } else if ("--password".equals(args[iarg])
                    || "--passwd".equals(args[iarg])) {
                fromPasswd = toPasswd = args[++iarg];
            } else if ("--pretty".equals(args[iarg])) {
                processRestrictionFilter.setPrettyPrintXml(true);
            } else if ("--quiet".equals(args[iarg])) {
                verbosity=0; // in fact handled by surrounding scripts
            } else if ("--schema-only".equals(args[iarg])) {
                processRestrictionFilter.setProcessData(false);
                processRestrictionFilter.setProcessConstraints(false);
            } else if ("--tables".equals(args[iarg])) {

                String[] tables = args[++iarg].split(",");

                for (String table:tables) {

                    int sep = table.indexOf(':');

                    if (sep < 0) {
                        processRestrictionFilter.addTableClause(table,null);
                    }
                    else {
                        processRestrictionFilter.addTableClause(
                                    table.substring(0,sep),
                                    table.substring(sep+1));
                    }
                }

            } else if ("--null-restrict".equals(args[iarg])) {
                processRestrictionFilter.addPrimaryRestriction(args[++iarg],null);
            } else if ("--restrict-table".equals(args[iarg])) {
                processRestrictionFilter.addPrimaryRestriction(args[++iarg],args[++iarg]);
            } else if ("--temp-db".equals(args[iarg])) {
                tempDB = args[++iarg];
            } else if ("--drop-tables".equals(args[iarg])) {
                dropTables = true;
            } else if ("--compression".equals(args[iarg])) {
                compression = Integer.valueOf(args[++iarg]);
            } else if ("--timezone".equals(args[iarg])) {
                timeZone = TimeZone.getTimeZone(args[++iarg]);
            } else if ("--to-driver".equals(args[iarg])) {
                toDriverName = args[++iarg];
            } else if ("--to-file".equals(args[iarg])) {
                toFileName = args[++iarg];
            } else if ("--to-passwd".equals(args[iarg])
                    || "--to-password".equals(args[iarg])) {
                toPasswd = args[++iarg];
            } else if ("--to-url".equals(args[iarg])) {
                toURL = args[++iarg];
            } else if ("--to-user".equals(args[iarg])) {
                toUser = args[++iarg];
            } else if ("--url".equals(args[iarg])) {
                fromURL = toURL = args[++iarg];
            } else if ("--user".equals(args[iarg])) {
                fromUser = toUser = args[++iarg];
            } else if ("--dialect-property".equals(args[iarg])) {

                String kv = args[++iarg];
                int pos = kv.indexOf('=');

                if (pos >= 0) {
                    processRestrictionFilter.addDialectProp(kv.substring(0,pos),kv.substring(pos+1));
                }
                else {
                    log.warn("Ignoring dialect property "+kv+" without a value, did your for the equals sign?");
                }

            } else if ("--verbose".equals(args[iarg])) {
                verbosity++; // in fact handled by surrounding scripts
            } else {
                log.warn("Ignoring unknown option "+args[iarg]);
            }
            ++iarg;
        }

        // check some basic requirements
        if (tempDBNeeded) {
            if (tempDB==null) {
                log.error("Which DB should I extract to? Use --temp-db or --help");
                System.exit(2);
            }
        }

        if (fromFileNeeded) {
            if (fromFileName==null) {
                log.info("Reading from standard input.");
            }
            if (!fromFileName.endsWith(".zip"))
                fromIS = StreamHelper.makeInputStream(fromFileName);
            else
                fromFile = new File(fromFileName);
        }

        if (toFileNeeded) {
            if (toFileName==null) {
                log.info("Writing to standard output.");
            }
            toOS = StreamHelper.makeOutpuStream(toFileName, compression);
        }

        if (timeZone == null)
                timeZone = TimeZone.getDefault();

        IDialectFactory dialectFactory = DialectFactory.newInstance();

        if (fromDBNeeded) {
            if (fromURL==null) {
                log.error("Which DB should I read from? Use --from-url or --help");
                System.exit(2);
            }
            if (fromDriverName==null) {
                try {
                    fromDialect=dialectFactory.newDialect(fromURL);
                    fromDialect.setProperties(processRestrictionFilter.getDialectProps());
                } catch (SQLException e) {
                    log.error("Could not determine DB dialect from JDBC URL "+fromURL+e.getMessage());
                    System.exit(2);
                }
                if (fromDialect==null) {
                    log.error("Could not determine DB dialect from JDBC URL "+fromURL);
                    System.exit(2);
                }
                fromDriverName=fromDialect.defaultDriverName();
                if (fromDriverName==null) {
                    log.error("Could not determine DB driver from JDBC URL "+fromURL);
                    System.exit(2);
                }
            }
            try {
                Class.forName(fromDriverName);
            } catch (ClassNotFoundException e) {
                log.error("Could not load from-driver "+fromDriverName);
                System.exit(2);
            }
            try {
                fromConnection = DriverManager.getConnection(fromURL, fromUser, fromPasswd);
            } catch (SQLException e) {
                log.error("Could not connect to source JDBC URL "+fromURL);
                System.exit(2);
            }
        }

        if (toDBNeeded) {
            if (toURL==null) {
                log.error("Which DB should I write to? Use --to-url or --help");
                System.exit(2);
            }
            if (toDriverName==null) {
                try {
                    toDialect=dialectFactory.newDialect(toURL);
                    toDialect.setProperties(processRestrictionFilter.getDialectProps());
                } catch (SQLException e) {
                    log.error("Could not determine DB dialect to JDBC URL "+toURL+e.getMessage());
                    System.exit(2);
                }
                if (toDialect==null) {
                    log.error("Could not determine DB dialect to JDBC URL "+toURL);
                    System.exit(2);
                }
                toDriverName=toDialect.defaultDriverName();
                if (toDriverName==null) {
                    log.error("Could not determine DB driver to JDBC URL "+toURL);
                    System.exit(2);
                }
                try {
                    Class.forName(toDriverName); // we don't mind doing this double
                } catch (ClassNotFoundException e) {
                    log.error("Could not load to-driver "+toDriverName);
                    System.exit(2);
                }
            }
            try {
                toConnection = DriverManager.getConnection(toURL, toUser, toPasswd);
            } catch (SQLException e) {
                log.error("Could not connect to destination JDBC URL "+toURL);
                System.exit(2);
            }
        }

        // Ass: no we have done all possible pre-checks, let's rock and roll

        // execute command

        if ( "jdbc2xml".equals(command)) {
            Commands.jdbc2xml(fromConnection, toOS, timeZone, processRestrictionFilter);
            System.exit(0);
        } else if ("xml2jdbc".equals(command)) {

            if (dropTables)
                Commands.dropTables(toConnection,processRestrictionFilter);

            Commands.xml2jdbc(((fromFile == null) ? fromIS : fromFile), toConnection, timeZone, processRestrictionFilter);
            System.exit(0);
        } else if ("jdbc2jdbc".equals(command)) {
            //jdbc2jdbc()
            log.error("Command not yet implemented: "+command);
            System.exit(99);
        } else if ("xml2xml".equals(command)) {
            Commands.xml2xml(((fromFile == null) ? fromIS : fromFile), toOS, processRestrictionFilter);
            System.exit(0);
        } else if ("jdbcping".equals(command)) {
            Commands.jdbcping(fromConnection);
            System.exit(0);
        } else if ("jdbcextr".equals(command)) {
            Commands.jdbcExtract(fromConnection, tempDB, timeZone, processRestrictionFilter);
            System.exit(0);
        } else if ("help".equals(command)) {
            log.info("The correct syntax is:"+syntax);
            System.exit(0);
        }

        throw new RuntimeException("You should not get here ...");
    }



}