/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 12.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.tools;

import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import org.clazzes.jdbc2xml.schema.ColumnInfo;
import org.clazzes.jdbc2xml.schema.ForeignKeyInfo;
import org.clazzes.jdbc2xml.schema.ISchemaEngine;
import org.clazzes.jdbc2xml.schema.SortableTableDescription;
import org.clazzes.jdbc2xml.schema.TableInfo;
import org.clazzes.jdbc2xml.schema.TableSorter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author rbreuss
 *
 */
public class DBDataExtractor {
    
    private static final Logger log = LoggerFactory.getLogger(DBDataExtractor.class);

    private ISchemaEngine schemaEngine;
    private ProcessRestrictionFilter processRestrictionFilter;
    private String exportDbName;
    private Hashtable<String, SortableTableDescription> tablesToExport;

    public void extract() throws SQLException 
    {
        if (this.schemaEngine == null)
        {
            log.error("schemaEngine is null.");
            return;
        }
        if (this.processRestrictionFilter == null)
        {
            log.error("processRestrictionFilter is null.");
            return;
        }
        
        if (this.exportDbName.equals(this.schemaEngine.getConnection().getCatalog()))
        {
            throw new SQLException("The export database must be different from the originating catalog.");
        }
        
        // fetch all requested TableInfo's from source database:
        log.info("begin fetching requested TableInfo's from source database ...");
        final List<TableInfo> tableInfos = this.schemaEngine.fetchTableInfos(this.processRestrictionFilter); 
        log.info("fetching TableInfo's from source database done.");

        String oldCatalog = this.schemaEngine.getConnection().getCatalog();
        this.schemaEngine.getConnection().setCatalog(this.exportDbName);
        
        final List<TableInfo> tempTableInfos = this.schemaEngine.fetchTableInfos(this.processRestrictionFilter);
        
        if (tempTableInfos != null && tempTableInfos.size() > 0)
        {
            log.info("Dropping [" + tempTableInfos.size() + "] old tables in destination database ...");
            this.schemaEngine.dropTables(tempTableInfos,true);
            this.schemaEngine.commit();
            log.info("Finished dropping [" + tempTableInfos.size() + "] old tables in destination database.");
        }

        log.info("Creating table structure in destination database ...");
        for (final TableInfo ti : tableInfos)
        {
            this.schemaEngine.createTable(ti, false);
        }
        this.schemaEngine.commit();
        
        log.info("Finished creating table structure in destination database.");
        
        this.schemaEngine.getConnection().setCatalog(oldCatalog);
        
        // sort tables by foreign key depth
        final List<SortableTableDescription> tableDescs = new ArrayList<SortableTableDescription>();
        for (final TableInfo ti : tableInfos)
        {
            tableDescs.add(new SortableTableDescription(ti));
        }
        TableSorter.sortTablesByFKDepth(tableDescs);
        
        // build up hash of tables to be exported
        this.tablesToExport = new Hashtable<String, SortableTableDescription>(tableDescs.size());
        for (final SortableTableDescription std : tableDescs)
        {
            this.tablesToExport.put(std.getTableInfo().getName(), std);
        }
        log.info("begin export to temporary database ...");
        // export to temporary database
        for (final SortableTableDescription std : tableDescs)
        {
            exportTableByForeignKeys(std, null);
        }
        log.info("export to temporary database done.");
        
        log.info("begin foreign key creation for temporary database ...");
        // create all foreign keys for destination database:
        for (final TableInfo ti : tableInfos)
        {
            final String tableName = ti.getName();
            ti.setName(this.exportDbName + "." + tableName);
            this.schemaEngine.createForeignKeys(ti);
            ti.setName(tableName);
        }
        log.info("foreign key creation for temporary database done.");
    }
    
    private boolean omitForeignKey(final ForeignKeyInfo fki, final SortableTableDescription std, final SortableTableDescription referencingTable) {
        if (fki == null)
            return true;
        if (fki.getForeignTable().equalsIgnoreCase(std.getTableInfo().getName()))
            return true;
        if ((referencingTable != null && referencingTable.getTableInfo().getName().equalsIgnoreCase(fki.getForeignTable())))
            return true;
        if (!this.processRestrictionFilter.processTable(fki.getForeignTable()))
            return true;

        return false;
    }
    
    private void exportTableByForeignKeys(final SortableTableDescription std,
            final SortableTableDescription referencingTable) throws SQLException
    {
        if (std == null || std.getTableInfo() == null)
            return;
        if (this.tablesToExport.get(std.getTableInfo().getName()) == null)
            return;
        
        final List<ForeignKeyInfo> fkis = std.getTableInfo().getForeignKeys();
        
        // test for dependencies of same depth which have to be exported first
        if (fkis != null && fkis.size() > 0)
        {
            for (final ForeignKeyInfo fki : fkis)
            {
                if (omitForeignKey(fki, std, referencingTable))
                    continue;
                final SortableTableDescription fkTable = this.tablesToExport.get(fki.getForeignTable());
                if (fkTable != null)
                {
                    log.info("try to export [" + std.getTableInfo().getName() + "], has to export [" + fki.getForeignTable() + "] first.");
                    exportTableByForeignKeys(fkTable, std);
                    exportTableByForeignKeys(std, null);
                    return;
                }
            }
        }
        final TableInfo ti = std.getTableInfo();
        final Map<String, String> restrictions = this.processRestrictionFilter.getPrimaryRestrictions();
        final StringBuffer sql = new StringBuffer();
        sql.append("INSERT INTO ");
        sql.append(this.exportDbName);
        sql.append(".");
        sql.append(ti.getName());
        sql.append(" SELECT DISTINCT ");
        
        StringBuffer joinExpr = new StringBuffer();
        joinExpr.append(" FROM ");
        
        joinExpr.append(ti.getName());
        joinExpr.append(" t1");
        
        int t = 2;
        if (fkis != null && fkis.size() > 0)
        {
            for (final ForeignKeyInfo fki : fkis)
            {
                if (omitForeignKey(fki, std, referencingTable))
                    continue;
                joinExpr.append(" LEFT JOIN ");
                joinExpr.append(this.exportDbName);
                joinExpr.append(".");
                joinExpr.append(fki.getForeignTable());
                joinExpr.append(" t");
                joinExpr.append(t);
                joinExpr.append(" ON (");
                int i = 0;
                for (final String col : fki.getColumns())
                {
                    if (i > 0)
                    {
                        joinExpr.append(" AND ");
                    }
                    joinExpr.append("t1.");
                    joinExpr.append(col);
                    joinExpr.append(" = t");
                    joinExpr.append(t);
                    joinExpr.append(".");
                    joinExpr.append(fki.getForeignColumns().get(i++));
                }
                joinExpr.append(")");
                t++;
            }
        }
        boolean hasWhere = false;
        Map<String,String> nullRestrictions=null;
        
        if (restrictions != null && restrictions.containsKey(ti.getName()))
        {
            final String restrictionValue = restrictions.get(ti.getName());
            
            if (restrictionValue == null)
                nullRestrictions=new HashMap<String, String>();
            else
            {
                log.info("restrcict table [" + ti.getName() + "] to id=[" + restrictionValue + "].");
                joinExpr.append(" WHERE t1.id=");
                joinExpr.append(restrictionValue);
                hasWhere = true;
            }
        }
        if (fkis != null && fkis.size() > 0)
        {
            t = 2;
            for (final ForeignKeyInfo fki : fkis)
            {
                if (omitForeignKey(fki, std, referencingTable))
                    continue;
                
                final boolean nullableFK = isFKNullable(ti,fki);
                
                if (nullRestrictions != null && nullableFK)
                {
                    for (int i = 0; i< fki.getColumns().size(); i++)
                    {
                        final String col = fki.getColumns().get(i);
                        final String foreignCol = fki.getForeignColumns().get(i);
                    
                        nullRestrictions.put(col,"t" + t + "." + foreignCol);
                    }
                }
                else
                {
                    for (int i = 0; i< fki.getColumns().size(); i++)
                    {
                        final String col = fki.getColumns().get(i);
                        final String foreignCol = fki.getForeignColumns().get(i);
                        
                        if (hasWhere)
                        {
                            joinExpr.append(" AND ");
                        }
                        else if (t == 2 && i == 0)
                        {
                            joinExpr.append(" WHERE ");
                            hasWhere = true;
                        }
                        if (nullableFK)
                        {
                            joinExpr.append("(t1.");
                            joinExpr.append(col);
                            joinExpr.append(" IS NULL OR ");
                        }
                        joinExpr.append("t");
                        joinExpr.append(t);
                        joinExpr.append(".");
                        joinExpr.append(foreignCol);
                        joinExpr.append(" IS NOT NULL");
                        if (nullableFK)
                        {
                            joinExpr.append(")");
                        }
                    }
                }
                ++t;
            }
        }
        
        boolean firstColumn = true;
        for (ColumnInfo ci : ti.getColumns())
        {
            if (firstColumn)
                firstColumn = false;
            else
                sql.append(", ");
                
            String replacement =
                nullRestrictions == null ? null : nullRestrictions.get(ci.getName());
            
            if (replacement != null)
                sql.append(replacement);
            else
            {
                sql.append("t1.");
                sql.append(ci.getName());
            }
        }
        
        sql.append(joinExpr);
        
        log.info("start executing: " + sql.toString());
        try(final Statement stm = this.schemaEngine.getConnection().createStatement()) {
            stm.execute(sql.toString());
        }
        this.tablesToExport.remove(ti.getName());
        
        log.info("execution ok, [" + this.tablesToExport.size() + "] tables remaining.");
    }
    
    private boolean isFKNullable(final TableInfo ti, final ForeignKeyInfo fki)
    {
        for (String columnName : fki.getColumns())
        {
            final ColumnInfo ci = ti.getColumnInfo(columnName);
            if (ci == null || !ci.isNullable())
                return false;
        }
        
        return true;
    }
    
    /**
     * @param schemaEngine the schemaEngine to set
     */
    public void setSchemaEngine(ISchemaEngine schemaEngine) {
        this.schemaEngine = schemaEngine;
    }

    /**
     * @param processRestrictionFilter the processRestrictionFilter to set
     */
    public void setProcessRestrictionFilter(
            ProcessRestrictionFilter processRestrictionFilter) {
        this.processRestrictionFilter = processRestrictionFilter;
    }

    /**
     * @param exportDbName the exportDbName to set
     */
    public void setExportDbName(String exportDbName) {
        this.exportDbName = exportDbName;
    }
    
    
    

}
