/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 14.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.sql;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Enumeration;
import java.util.Stack;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This is a queue of SQL commands, which may be rolled back or committed.
 * 
 * @author wglas
 */
public class SqlCommandQueue {

    private static final Logger log = LoggerFactory.getLogger(SqlCommandQueue.class);


    private final Stack<SqlCommand> commands;
    
    /**
     * How many command on the stack have been performed up to now.
     */
    private int watermark;

    /**
     * Default constructor.
     */
    public SqlCommandQueue()
    {
        this.commands = new Stack<SqlCommand>();
        this.watermark = 0;
    }
    
    /**
     * @param command The command to push to the queue.
     */
    public void pushCommand(SqlCommand command)
    {
        this.commands.push(command);
    }
    
    /**
     * Push an instance of {@link SimpleSqlCommand} to the queue.
     * 
     * @param performSql The sql statement to perform the change. 
     * @param rollbackSql The sql statement to roll back the change.
     * 
     * @see SimpleSqlCommand#SimpleSqlCommand(String, String)
     */
    public void pushCommand(String performSql, String rollbackSql)
    {
        this.commands.push(new SimpleSqlCommand(performSql,rollbackSql));
    }
    
    /**
     * Push an instance of {@link QuerieSqlCommand} to the queue.
     * 
     * @param performSql The sql statement to perform the change. 
     * @param rollbackSql The sql statement to roll back the change.
     * 
     * @see QuerieSqlCommand#QuerieSqlCommand(String, String)
     */
    public void pushQuerieCommand(String performSql, String rollbackSql)
    {
        this.commands.push(new QuerieSqlCommand(performSql,rollbackSql));
    }
    
      /**
     * Perform all commands currently pushed to the stack by {@link #pushCommand(SqlCommand)}
     * and not performed yet.
     * 
     * @param connection The JDBC connection on which to perform queries.
     * 
     * @throws SQLException Upon database errors.
     * @see SqlCommand#perform(Connection)
     */
    public void perform(Connection connection) throws SQLException
    {
        while (this.watermark < this.commands.size())
        {
            SqlCommand command = null;
            try
            {
                command = this.commands.get(this.watermark);
                command.perform(connection);
            }
            catch (Exception e)
            {
                String reason = "perform: Caught [" + e.getClass().getSimpleName() + "] performing command [" + command.toString() + "].";
                log.error(reason, e);
                throw new SQLException(reason, e);
            }
            ++this.watermark;
        }
    }
    
    /**
     * Perform all commands currently pushed to the stack by {@link #pushCommand(SqlCommand)}
     * and not performed yet. Afterwards discard the whole set of database commands.
     * @param connection The JDBC connection on which to perform queries.
     * 
     * @throws SQLException Upon database errors.
     */
    public void commit(Connection connection) throws SQLException
    {
        this.perform(connection);
        
        for (SqlCommand cmd : this.commands)
            cmd.cleanupOnCommit(connection);
            
        this.commands.clear();
        this.watermark = 0;
    }
    
    /**
     * Roll back all commands currently pushed to the stack by {@link #pushCommand(SqlCommand)}
     * and performed up to now. Afterwards discard the whole set of database commands.
     * @param connection The JDBC connection on which to perform the rollback.
     * 
     * @throws SQLException
     * @see SqlCommand#rollback(Connection)
     */
    public void rollback(Connection connection) throws SQLException
    {
        while (this.watermark > 0)
        {
            --this.watermark;
            SqlCommand command = null;
            try
            {
                command = this.commands.get(this.watermark);
                command.rollback(connection);
            }
            catch (Exception e)
            {
                String reason = "rollback: Caught [" + e.getClass().getSimpleName() + "] performing command [" + command.toString() + "].";
                log.error(reason);
                throw new SQLException(reason, e);
            }
        }
        
        this.commands.clear();
        this.watermark = 0;
    }
    
    public Enumeration<SqlCommand> commands()
    {
        return this.commands.elements();
    }
}
