/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 14.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.sql;

import java.sql.Connection;
import java.sql.SQLException;

/**
 * This interface defines a SQL command, which may be rolled back.
 * 
 * @author wglas
 */
public interface SqlCommand {

    /**
     * Perform an atomic SQL command.
     * 
     * @param connection A JDBC connection on which to perform command.
     * @throws SQLException Upon database errors.
     */
    public void perform(Connection connection) throws SQLException;
    
    /**
     * Roll back an atomic SQL command.
     * 
     * @param connection A JDBC connection on which to perform queries.
     * @throws SQLException Upon database errors or when a rollback is
     *                      not supported.
     */
    public void rollback(Connection connection) throws SQLException;
    
    /**
     * Cleanup resources kept by this command, if the transaction is committed.
     * This might incur deleting temporary backup tables or deleting data files,
     * which back up deleted table data.
     * 
     * @param connection
     * @throws SQLException
     */
    public void cleanupOnCommit(Connection connection) throws SQLException;
    
    /**
     * @return the tempTableName
     */
    public String getTempTableName();

    /**
     * @return the tempTableCreated
     */
    public boolean isTempTableCreated();
}
