/***********************************************************
 * $Id: SimpleSqlCommand.java 305 2011-02-24 08:57:46Z jdbc2xml $
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 14.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.sql;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;

import org.clazzes.jdbc2xml.helper.SQLHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * this is a simple SQL command, which performs SQL queries using
 * {@link Statement#executeQuery(String)}. 
 * It is just created for a special case in postgres, so it doesn't actually 
 * return the values form the ResultSet
 * 
 * @author wglas
 */
public class QuerieSqlCommand implements SqlCommand {

    private static final Logger log = LoggerFactory.getLogger(QuerieSqlCommand.class);
    
    private String performSql;
    private String rollbackSql;
    
    /**
     * @param performSql The sql statement to perform the change. 
     * @param rollbackSql The sql statement to roll back the change.
     */
    public QuerieSqlCommand(String performSql, String rollbackSql) {
        super();
        this.performSql = performSql;
        this.rollbackSql = rollbackSql;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.sql.SqlCommand#perfom(java.sql.Connection)
     */
    public void perform(Connection connection) throws SQLException {
        
        if (this.performSql == null)
            throw new SQLException("This statement has already been committed.");

        if (log.isDebugEnabled())
            log.debug("Performing statement ["+this.performSql+"].");
        
        try {
            SQLHelper.executeQuerie(connection,this.performSql);
        } catch (SQLException e) {
            log.error("SQL [" + this.performSql + "] threw an Exception: ", e);
            throw e;
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.sql.SqlCommand#rollback(java.sql.Connection)
     */
    public void rollback(Connection connection) throws SQLException {
        
        if (this.rollbackSql == null)
            throw new SQLException("Rollback of statement ["+this.performSql+"] is not supported.");
            
        if (log.isDebugEnabled())
            log.debug("Executing rollback statement ["+this.rollbackSql+"].");

        SQLHelper.executeUpdate(connection,this.rollbackSql);
    }

    /**
     * @return the performSql
     */
    public String getPerformSql() {
        return this.performSql;
    }

    /**
     * @return the rollbackSql
     */
    public String getRollbackSql() {
        return this.rollbackSql;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.sql.SqlCommand#cleanupOnCommit(java.sql.Connection)
     */
    public void cleanupOnCommit(Connection connection) throws SQLException {
        this.performSql = null;
        this.rollbackSql = null;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.sql.SqlCommand#getTempTableName()
     */
    public String getTempTableName() {
        return null;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.sql.SqlCommand#isTempTableCreated()
     */
    public boolean isTempTableCreated() {
        return false;
    }

}
