/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 28.11.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema.impl;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.SortedMap;
import java.util.TreeMap;

import org.clazzes.jdbc2xml.helper.JAVAHelper;
import org.clazzes.jdbc2xml.helper.SQLHelper;
import org.clazzes.jdbc2xml.helper.TypesHelper;
import org.clazzes.jdbc2xml.schema.ColumnInfo;
import org.clazzes.jdbc2xml.schema.ForeignKeyInfo;
import org.clazzes.jdbc2xml.schema.IDialectFactory;
import org.clazzes.jdbc2xml.schema.IndexFilter;
import org.clazzes.jdbc2xml.schema.IndexInfo;
import org.clazzes.jdbc2xml.schema.IndexInfo.Order;
import org.clazzes.jdbc2xml.schema.PrimaryKeyInfo;
import org.clazzes.jdbc2xml.schema.SchemaEngine;
import org.clazzes.jdbc2xml.schema.TableFilter;
import org.clazzes.jdbc2xml.schema.TableInfo;
import org.clazzes.jdbc2xml.sql.SqlCommand;
import org.clazzes.jdbc2xml.sql.SqlCommandQueue;
import org.clazzes.util.lang.Util;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The default implementation of the schema engine.
 * 
 * @author wglas
 */
public class SchemaEngineImpl extends SchemaEngine {

    private static final Logger log = LoggerFactory.getLogger(SchemaEngineImpl.class);

    private final SqlCommandQueue queue;

    private int maxTableNameLength;

    /**
     * Default constructor.
     */
    public SchemaEngineImpl() {
        super();
        this.queue = new SqlCommandQueue();
    }

    /**
     * Default constructor.
     */
    public SchemaEngineImpl(IDialectFactory dialectFactory) {
        super(dialectFactory);
        this.queue = new SqlCommandQueue();
    }


    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#setConnection(java.sql.Connection)
     */
    @Override
    public void setConnection(Connection connection) throws SQLException {

        super.setConnection(connection);

        if (this.getConnection() != null) {
            DatabaseMetaData dbMeta = this.getConnection().getMetaData();
            this.maxTableNameLength = dbMeta.getMaxTableNameLength();
        }
    }

    private void addColumnsToTableInfo(DatabaseMetaData dbMeta,TableInfo ti, boolean doCheckForAutoIncrement) throws SQLException
    {
        try(ResultSet rs = dbMeta.getColumns(this.getConnection().getCatalog(),toStoredIdentifier(dbMeta,this.getSchema()),toStoredIdentifier(dbMeta,ti.getName()),null)) {
        

            /*
              1. TABLE_CAT String => table catalog (may be null)
              2. TABLE_SCHEM String => table schema (may be null)
              3. TABLE_NAME String => table name
              4. COLUMN_NAME String => column name
              5. DATA_TYPE int => SQL type from java.sql.Types
              6. TYPE_NAME String => Data source dependent type name, for a UDT the type name is fully qualified
              7. COLUMN_SIZE int => column size. For char or date types this is the maximum number of characters, for numeric or decimal types this is precision.
              8. BUFFER_LENGTH is not used.
              9. DECIMAL_DIGITS int => the number of fractional digits
             10. NUM_PREC_RADIX int => Radix (typically either 10 or 2)
             11. NULLABLE int => is NULL allowed.
                 * columnNoNulls - might not allow NULL values
                 * columnNullable - definitely allows NULL values
                 * columnNullableUnknown - nullability unknown 
             12. REMARKS String => comment describing column (may be null)
             13. COLUMN_DEF String => default value (may be null)
             14. SQL_DATA_TYPE int => unused
             15. SQL_DATETIME_SUB int => unused
             16. CHAR_OCTET_LENGTH int => for char types the maximum number of bytes in the column
             17. ORDINAL_POSITION int => index of column in table (starting at 1)
             18. IS_NULLABLE String => "NO" means column definitely does not allow NULL values; "YES" means the column might allow NULL values. An empty string means nobody knows.
             19. SCOPE_CATLOG String => catalog of table that is the scope of a reference attribute (null if DATA_TYPE isn't REF)
             20. SCOPE_SCHEMA String => schema of table that is the scope of a reference attribute (null if the DATA_TYPE isn't REF)
             21. SCOPE_TABLE String => table name that this the scope of a reference attribure (null if the DATA_TYPE isn't REF)
             22. SOURCE_DATA_TYPE short => source type of a distinct type or user-generated Ref type, SQL type from java.sql.Types (null if DATA_TYPE isn't DISTINCT or user-generated REF) 
             */

            while (rs.next())
            {
                String name = rs.getString("COLUMN_NAME");
                int type = rs.getInt("DATA_TYPE");
                String typeName = rs.getString("TYPE_NAME");
                /* 
                    SQL types for timestamp_with_timezone:

                    PostgreSQL: timestamptz with type 93 (timestamp), which is timestamp
                    MariaDB:
                    MSSQL: DATETIMEOFFSET int value -155
                    Oracle: TIMESTAMP(6) WITH TIME ZONE int value -101
                */ 
                log.info("addColumnsToTableInfo: Column [" + name + "] type=[" + type + "] -> [" + rs.getString("TYPE_NAME") + "].");
                
                // work around a postgres idiosyncrasy
                if (type == Types.BIT && "bool".equals(typeName)) {
                    type = Types.BOOLEAN;
                }
                
                // -101 and -155 are reserved enums for TIMESTAMP_WITH_TIMEZONE in OracleDB and MSSQL respectively.
                if (type == Types.OTHER || type == Types.TIMESTAMP || type == Types.TIMESTAMP_WITH_TIMEZONE || type == -101 || type == -155)
                {
                    type = this.getDialect().getMappedSqlType(typeName);
                    log.debug("Got mapping [" + type + "] for typeName [" + typeName + "]");
                }
                int prec_i = rs.getInt("COLUMN_SIZE");
                Integer precision = rs.wasNull() ? null : prec_i;
                Integer scale;

                if (    type == Types.NUMERIC ||
                        type == Types.DECIMAL ||
                        type == Types.REAL      )
                {
                    int scale_i = rs.getInt("DECIMAL_DIGITS");
                    scale = rs.wasNull() ? null : scale_i;
                }
                else
                    scale = null;

                if (type == Types.NUMERIC && precision != null && precision.intValue() > 65535) {
                    // Postgtres SQL's JDBC driver reports a precision of 131089
                    // for columns, which are bare 'NUMERIC', so filter this out :-/
                    precision = null;
                    scale = null;
                }

                if ((type == Types.NUMERIC || type == Types.DECIMAL) && scale != null && scale.intValue() < 0) {
                    // Oracle's JDBC driver reports a scale of -127
                    // for columns, which are bare 'NUMERIC', so filter this out :-/
                    precision = null;
                    scale = null;
                }

                if (type == Types.BINARY && precision!=null && precision.intValue() >= 2147483647) {
                    // Postgtres SQL's JDBC driver reports a precision of 2147483647
                    // for columns, which are bare 'bytea', so filter this out :-/
                    type = Types.LONGVARBINARY;
                    precision = null;
                }

                int nullable_i = rs.getInt("NULLABLE");
                boolean nullable = nullable_i != DatabaseMetaData.columnNoNulls;

                // Well, some database engines write database-specific default values to
                // date/time columns, which cause more headache than profit.
                // An example is MySQL reporting 'CURRENT_TIMESTAMP' here or
                // timestamps in the format '0000-00-00 00:00:00', which are
                // definitely not accepted by each RDBMS.

                //
                // Well, default values for BLOB's and binaries do not make much sense
                // either, so drop these, too.
                // Moreover, some RDMBS engines do not cope with default values for
                // CLOBs or LONGVARCHARs, so do not generate default values for these, too.
                //

                boolean autoIncrement = false;
                // FIXME someday cleanup:
                // for MSSQL this is detected ~10 lines below
                // for Oracle this is detected in this.getDialect().fetchAdditionalColumnInfo()
                // for MySQL, Postgres, ... it is not detected at all
                // However, for DB export it is detected in ColumnInfo(ResultSetMetaData md, int column) called by JDBCToSAXWriter  

                String defaultValue = null;

                if (doCheckForAutoIncrement && this.getDialect() instanceof MSSQLServerDialect)
                {
                    if (typeName.contains(" identity"))
                        autoIncrement = true;
                }
                // Finally we write default values only for numeric and
                // normal string types.
                if (TypesHelper.isNumeric(type)) {

                    String def = rs.getString("COLUMN_DEF");
                    if (this.getDialect() != null)
                        def = this.getDialect().normalizeDefaultValue(type,def);

                    if (def != null && def.length() > 0)
                        defaultValue = def;
                }
                else if (TypesHelper.isString(type)) {
                    defaultValue = rs.getString("COLUMN_DEF");

                    if (precision != null && precision.intValue() >= 2147483647) {

                        // Postgtres SQL's JDBC driver reports a precision of 2147483647
                        // for columns, which are bare 'TEXT', so filter this out :-/
                        type = Types.LONGVARCHAR;
                        precision = null;
                    }


                    if  (this.getDialect() != null)
                        defaultValue =
                        this.getDialect().normalizeDefaultValue(type,defaultValue);
                } else {
                    // nothing here, because other type often contain unportable default values
                    // like MySQL's CURRENT_TIMESTAMP
                }

                ColumnInfo ci = new ColumnInfo(name,type,precision,scale,nullable,defaultValue, autoIncrement);

                ti.addColumn(ci);
            }
        }
    }

    private void checkAutoIncrementColumn(TableInfo ti) throws SQLException
    {
        ResultSet rs = null;
        String autoIncrementColumn = null;
        ColumnInfo ci;
        try
        {
            Statement st = this.getConnection().createStatement();
            st.setMaxRows(1);
            if (st.execute("SELECT * FROM " + (this.getSchema()==null?ti.getName():this.getSchema()+"."+ti.getName())))
            {
                rs = st.getResultSet();
                ResultSetMetaData md = rs.getMetaData();
                int cols = md.getColumnCount();
                for (int col = 1; col<cols; col++)
                {
                    if (md.isAutoIncrement(col))
                    {
                        autoIncrementColumn = md.getColumnName(col);
                        break;
                    }
                }
                if (autoIncrementColumn != null)
                {
                    ci = ti.getColumnInfo(autoIncrementColumn);
                    if (ci != null)
                        ci.setAutoIncrement(true);
                }
            }
        }
        catch (SQLException e)
        {
            log.error("Error in checkAutoIncrementColumn table=[" + ti.getName() + "]: ", e);
            throw e;
        }
        finally
        {
            if (rs != null)
                rs.close();
        }
    }

    private void addPrimaryKeyToTableInfo(DatabaseMetaData dbMeta,TableInfo ti) throws SQLException
    {
        /* 1. TABLE_CAT String => table catalog (may be null)
           2. TABLE_SCHEM String => table schema (may be null)
           3. TABLE_NAME String => table name
           4. COLUMN_NAME String => column name
           5. KEY_SEQ short => sequence number within primary key
           6. PK_NAME String => primary key name (may be null)
         */
        try (ResultSet rs = dbMeta.getPrimaryKeys(this.getConnection().getCatalog(),toStoredIdentifier(dbMeta,this.getSchema()),toStoredIdentifier(dbMeta,ti.getName()))) {

            if (!rs.next()) return;
            
            String tableName = rs.getString("TABLE_NAME");
            String tableCat = rs.getString("TABLE_CAT");
            String tableSchema = rs.getString("TABLE_SCHEM");
            Integer seq = Integer.parseInt(rs.getString("KEY_SEQ"));
            
            String name = rs.getString("PK_NAME");
            
            SortedMap<Integer,String> sortedColumns = new TreeMap<Integer, String>();
            
            sortedColumns.put(seq,rs.getString("COLUMN_NAME"));
            
            while (rs.next())
            {
                if (!Util.equalsNullAware(tableName,rs.getString("TABLE_NAME")))
                    break;
                
                if (!Util.equalsNullAware(tableCat,rs.getString("TABLE_CAT")))
                    break;
                
                if (!Util.equalsNullAware(tableSchema,rs.getString("TABLE_SCHEM")))
                    break;
                
                if (!Util.equalsNullAware(name,rs.getString("PK_NAME")))
                    break;
                
                seq = Integer.parseInt(rs.getString("KEY_SEQ"));
                
                sortedColumns.put(seq,rs.getString("COLUMN_NAME"));
            }
            
            if ("PRIMARY".equals(name)) name=null; // workaround a bug parsing mysql meta data
            
            List<String> columns = new ArrayList<String>(sortedColumns.size());
            columns.addAll(sortedColumns.values());
            
            PrimaryKeyInfo pki = new PrimaryKeyInfo();
            pki.setName(name);
            pki.setColumns(columns);
            ti.setPrimaryKey(pki);
        }
    }
    
    private void addIndicesToTableInfo(DatabaseMetaData dbMeta,TableInfo ti,
            boolean keepInternalIndices) throws SQLException
    {
        try(ResultSet rs =
                dbMeta.getIndexInfo(this.getConnection().getCatalog(),toStoredIdentifier(dbMeta,this.getSchema()),toStoredIdentifier(dbMeta,ti.getName()),
                        false, false)) {

            boolean hasNext = rs.next();

            while (hasNext)
            {
                /* 
                  1. TABLE_CAT String => table catalog (may be null)
                  2. TABLE_SCHEM String => table schema (may be null)
                  3. TABLE_NAME String => table name
                  4. NON_UNIQUE boolean => Can index values be non-unique. false when TYPE is tableIndexStatistic
                  5. INDEX_QUALIFIER String => index catalog (may be null); null when TYPE is tableIndexStatistic
                  6. INDEX_NAME String => index name; null when TYPE is tableIndexStatistic
                  7. TYPE short => index type:
                       * tableIndexStatistic - this identifies table statistics that are returned in conjuction with a table's index descriptions
                       * tableIndexClustered - this is a clustered index
                       * tableIndexHashed - this is a hashed index
                       * tableIndexOther - this is some other style of index 
                  8. ORDINAL_POSITION short => column sequence number within index; zero when TYPE is tableIndexStatistic
                  9. COLUMN_NAME String => column name; null when TYPE is tableIndexStatistic
                 10. ASC_OR_DESC String => column sort sequence, "A" => ascending, "D" => descending, may be null if sort sequence is not supported; null when TYPE is tableIndexStatistic
                 11. CARDINALITY int => When TYPE is tableIndexStatistic, then this is the number of rows in the table; otherwise, it is the number of unique values in the index.
                 12. PAGES int => When TYPE is tableIndexStatisic then this is the number of pages used for the table, otherwise it is the number of pages used for the current index.
                 13. FILTER_CONDITION String => Filter condition, if any. (may be null) 
                 */
                String tableName = rs.getString("TABLE_NAME");
                String tableCat = rs.getString("TABLE_CAT");
                String tableSchema = rs.getString("TABLE_SCHEM");

                if (rs.getShort("TYPE") == DatabaseMetaData.tableIndexStatistic)
                {
                    hasNext = rs.next();
                    continue;
                }

                String order_s = rs.getString("ASC_OR_DESC");
                IndexInfo.Order order = (order_s == null ? null : (order_s.startsWith("D") ? Order.DESC : Order.ASC));

                boolean unique = !rs.getBoolean("NON_UNIQUE");

                String filterCondition = rs.getString("FILTER_CONDITION");

                Integer seq = Integer.parseInt(rs.getString("ORDINAL_POSITION"));

                String name = rs.getString("INDEX_NAME");

                SortedMap<Integer,String> sortedColumns = new TreeMap<Integer, String>();

                sortedColumns.put(seq,rs.getString("COLUMN_NAME"));

                while ((hasNext = rs.next()) == true)
                {
                    if (!Util.equalsNullAware(tableName,rs.getString("TABLE_NAME")))
                        break;

                    if (!Util.equalsNullAware(tableCat,rs.getString("TABLE_CAT")))
                        break;

                    if (!Util.equalsNullAware(tableSchema,rs.getString("TABLE_SCHEM")))
                        break;

                    if (!Util.equalsNullAware(name,rs.getString("INDEX_NAME")))
                        break;

                    seq = Integer.parseInt(rs.getString("ORDINAL_POSITION"));

                    sortedColumns.put(seq,rs.getString("COLUMN_NAME"));
                }

                List<String> columns = new ArrayList<String>(sortedColumns.size());
                columns.addAll(sortedColumns.values());

                boolean doInsert = true;

                if (!keepInternalIndices)
                {
                    if (ti.getPrimaryKey() != null && ti.getPrimaryKey().getColumns().equals(columns))
                    {
                        if (log.isDebugEnabled())
                            log.debug("Skipping internal index ["+name+
                                    "] because it covers the primary key columns ["+
                                    JAVAHelper.joinStrings(ti.getPrimaryKey().getColumns())+"].");

                        doInsert =false;
                    }
                    if (doInsert && ti.getForeignKeys() != null )  
                    {
                        for (ForeignKeyInfo fkInfo : ti.getForeignKeys())
                        {
                            if (fkInfo.getColumns().equals(columns))
                            {
                                if (log.isDebugEnabled())
                                    log.debug("Skipping internal index ["+name+
                                            "] because it covers the columns ["+
                                            JAVAHelper.joinStrings(fkInfo.getColumns())+
                                            "] of foreign key [" + fkInfo.getName() + "].");

                                doInsert=false;
                                break;
                            }
                        }
                    }
                }

                if (doInsert)
                {
                    IndexInfo ii = new IndexInfo();
                    ii.setName(name);
                    ii.setColumns(columns);
                    ii.setUnique(unique);
                    ii.setFilterCondition(filterCondition);
                    ii.setOrder(order);
                    ti.addIndex(ii);
                }
            }
        }
    }

    private void addForeignKeysToTableInfo(DatabaseMetaData dbMeta,TableInfo ti) throws SQLException
    {
        try (ResultSet rs = dbMeta.getImportedKeys(this.getConnection().getCatalog(),toStoredIdentifier(dbMeta,this.getSchema()),toStoredIdentifier(dbMeta,ti.getName()))) {

            /* 1. PKTABLE_CAT String => primary key table catalog being imported (may be null)
               2. PKTABLE_SCHEM String => primary key table schema being imported (may be null)
               3. PKTABLE_NAME String => primary key table name being imported
               4. PKCOLUMN_NAME String => primary key column name being imported
               5. FKTABLE_CAT String => foreign key table catalog (may be null)
               6. FKTABLE_SCHEM String => foreign key table schema (may be null)
               7. FKTABLE_NAME String => foreign key table name
               8. FKCOLUMN_NAME String => foreign key column name
               9. KEY_SEQ short => sequence number within a foreign key
              10. UPDATE_RULE short => What happens to a foreign key when the primary key is updated:
                  * importedNoAction - do not allow update of primary key if it has been imported
                  * importedKeyCascade - change imported key to agree with primary key update
                  * importedKeySetNull - change imported key to NULL if its primary key has been updated
                  * importedKeySetDefault - change imported key to default values if its primary key has been updated
                  * importedKeyRestrict - same as importedKeyNoAction (for ODBC 2.x compatibility) 
              11. DELETE_RULE short => What happens to the foreign key when primary is deleted.
                  * importedKeyNoAction - do not allow delete of primary key if it has been imported
                  * importedKeyCascade - delete rows that import a deleted key
                  * importedKeySetNull - change imported key to NULL if its primary key has been deleted
                  * importedKeyRestrict - same as importedKeyNoAction (for ODBC 2.x compatibility)
                  * importedKeySetDefault - change imported key to default if its primary key has been deleted 
              12. FK_NAME String => foreign key name (may be null)
              13. PK_NAME String => primary key name (may be null)
              14. DEFERRABILITY short => can the evaluation of foreign key constraints be deferred until commit
                  * importedKeyInitiallyDeferred - see SQL92 for definition
                  * importedKeyInitiallyImmediate - see SQL92 for definition
                  * importedKeyNotDeferrable - see SQL92 for definition 
             */
            boolean hasNext = rs.next();

            while (hasNext)
            {
                String tableName = rs.getString("FKTABLE_NAME");
                String tableCat = rs.getString("FKTABLE_CAT");
                String tableSchema = rs.getString("FKTABLE_SCHEM");

                if (!Util.equalsNullAware(tableCat, rs.getString("PKTABLE_CAT"))||
                        !Util.equalsNullAware(tableSchema, rs.getString("PKTABLE_SCHEM")))
                {
                    hasNext = rs.next();
                    continue;
                }

                String foreignTable = rs.getString("PKTABLE_NAME");

                Integer seq = Integer.parseInt(rs.getString("KEY_SEQ"));

                String name = rs.getString("FK_NAME");
                String pkName = rs.getString("PK_NAME");
                short updateRule = rs.getShort("UPDATE_RULE");
                if (updateRule == DatabaseMetaData.importedKeyRestrict)
                    updateRule = DatabaseMetaData.importedKeyNoAction;
                short deleteRule = rs.getShort("DELETE_RULE");
                if (deleteRule == DatabaseMetaData.importedKeyRestrict)
                    deleteRule = DatabaseMetaData.importedKeyNoAction;
                short deferrability = rs.getShort("DEFERRABILITY");

                SortedMap<Integer,String> sortedColumns = new TreeMap<Integer, String>();
                SortedMap<Integer,String> sortedForeignColumns = new TreeMap<Integer, String>();

                sortedColumns.put(seq,rs.getString("FKCOLUMN_NAME"));
                sortedForeignColumns.put(seq,rs.getString("PKCOLUMN_NAME"));

                while ((hasNext = rs.next()) == true)
                {
                    if (!Util.equalsNullAware(tableName,rs.getString("FKTABLE_NAME")))
                        break;

                    if (!Util.equalsNullAware(foreignTable,rs.getString("PKTABLE_NAME")))
                        break;

                    if (!Util.equalsNullAware(tableCat,rs.getString("FKTABLE_CAT")))
                        break;

                    if (!Util.equalsNullAware(tableCat,rs.getString("PKTABLE_CAT")))
                        break;

                    if (!Util.equalsNullAware(tableSchema,rs.getString("FKTABLE_SCHEM")))
                        break;

                    if (!Util.equalsNullAware(tableSchema,rs.getString("PKTABLE_SCHEM")))
                        break;

                    if (!Util.equalsNullAware(name,rs.getString("FK_NAME")))
                        break;

                    if (!Util.equalsNullAware(pkName,rs.getString("PK_NAME")))
                        break;

                    seq = Integer.parseInt(rs.getString("KEY_SEQ"));

                    sortedColumns.put(seq,rs.getString("FKCOLUMN_NAME"));
                    sortedForeignColumns.put(seq,rs.getString("PKCOLUMN_NAME"));
                }

                List<String> columns = new ArrayList<String>(sortedColumns.size());
                columns.addAll(sortedColumns.values());

                List<String> foreignColumns = new ArrayList<String>(sortedForeignColumns.size());
                foreignColumns.addAll(sortedForeignColumns.values());

                ForeignKeyInfo fki = new ForeignKeyInfo();

                fki.setName(name);
                fki.setForeignTable(foreignTable);
                fki.setPkName(pkName);
                fki.setColumns(columns);
                fki.setForeignColumns(foreignColumns);
                fki.setDeleteRule(deleteRule);
                fki.setUpdateRule(updateRule);
                fki.setDeferrability(deferrability);
                ti.addForeignKey(fki);
            }
        }
    }


    private String toStoredIdentifier(DatabaseMetaData dbMeta, String id) throws SQLException {
        if (id == null) {
            return null;
        } else if (dbMeta.storesLowerCaseIdentifiers()) {
            return id.toLowerCase(Locale.ENGLISH);
        } else if (dbMeta.storesUpperCaseIdentifiers()) {
            return id.toUpperCase(Locale.ENGLISH);
        } else {
            return id;
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#fetchTableInfo(java.lang.String, org.clazzes.jdbc2xml.schema.IndexFilter)
     */
    public TableInfo fetchTableInfo(String tableName, IndexFilter filter) throws SQLException {

        DatabaseMetaData dbMeta = this.getConnection().getMetaData();
        this.maxTableNameLength = dbMeta.getMaxTableNameLength();
        log.info("fetch table info [" + dbMeta.getDatabaseProductName() + ", version: " + dbMeta.getDatabaseProductVersion() + "], table=[" + tableName + "].");

        boolean doCheckForAutoIncrementColumns = isDoCheckAutoIncrementColumns(filter);

        try(ResultSet rs = dbMeta.getTables(this.getConnection().getCatalog(),
                toStoredIdentifier(dbMeta,this.getSchema()),toStoredIdentifier(dbMeta, tableName), new String[]{"TABLE"})) {

            if (!rs.next())
                throw new SQLException("Table ["+tableName+"] has not been found.");
            
            String tableComment = rs.getString("REMARKS");
            
            TableInfo ti = new TableInfo(tableName);
            ti.setComment(tableComment);
            
            this.addColumnsToTableInfo(dbMeta,ti,doCheckForAutoIncrementColumns);
            this.addPrimaryKeyToTableInfo(dbMeta,ti);
            this.addForeignKeysToTableInfo(dbMeta,ti);
            this.addIndicesToTableInfo(dbMeta,ti,filter == null ? false : filter.isKeepInternalIndices());
            if (doCheckForAutoIncrementColumns) {
                this.checkAutoIncrementColumn(ti);
            }
            for(ColumnInfo ci : ti.getColumns()){
                // fetch any additional, dialect specific column information:
                this.getDialect().fetchAdditionalColumnInfo(this, ti, ci);
            }
            return ti;
        }
    }

    private static boolean isDoCheckAutoIncrementColumns(IndexFilter indexFilter) {

        try {
            if (indexFilter == null) {
                return true;
            }
            else {
                return indexFilter.isCheckForAutoIncrementColumns();
            }
        }
        catch(AbstractMethodError e) {
            // old implementation of IndexFilter w/o isCheckForAutoIncrementColumns() called
            // -> The default behavior is to check for autoincrement columns.
            return true;
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#fetchTableInfos(org.clazzes.jdbc2xml.schema.TableFilter)
     */
    public List<TableInfo> fetchTableInfos(TableFilter filter)
            throws SQLException {

        log.info("Fetching all tables from catalog ["+this.getConnection().getCatalog()+"], schema ["+this.getSchema()+"].");
        DatabaseMetaData dbMeta = this.getConnection().getMetaData();
        this.maxTableNameLength = dbMeta.getMaxTableNameLength();
        log.info("fetch table infos [" + dbMeta.getDatabaseProductName() + ", version: " + dbMeta.getDatabaseProductVersion() + "].");

        boolean doCheckForAutoIncrementColumns = isDoCheckAutoIncrementColumns(filter);

        try (ResultSet rs = dbMeta.getProcedures(null, toStoredIdentifier(dbMeta,this.getSchema()), "*")) {
            
            List<String> func = new ArrayList<String>();
            while (rs.next())
            {
                func.add(rs.getString(1));
            }
            String[] functions = new String[func.size()];
            func.toArray(functions);
            if (log.isDebugEnabled())
            {
                String str = "";
                for (String s : functions)
                    str += s + ", ";
                log.debug("Functions: [" + str + "]");
            }
        } catch (Throwable e) {
            log.error("Error fetching table types.", e);
        }
        
        String[] tableTypes = null;
        try (ResultSet rs = dbMeta.getTableTypes();) {
            
            List<String> tt = new ArrayList<String>();
            while (rs.next())
            {
                tt.add(rs.getString(1));
            }
            tableTypes = new String[tt.size()];
            tt.toArray(tableTypes);
            if (log.isDebugEnabled())
            {
                String str = "";
                for (String s : tt)
                    str += s + ", ";
                log.debug("TableType: [" + str + "]");
            }
        } catch (Exception e) {
            log.error("Error fetching table types.");
        }

        try (ResultSet rs = dbMeta.getTables(this.getConnection().getCatalog(),
                toStoredIdentifier(dbMeta,this.getSchema()),null,new String[]{"TABLE"}))
        {
            List<TableInfo> ret = new LinkedList<TableInfo>();
            final String rex = "\\w*";
            while (rs.next())
            {
                String tableName = rs.getString("TABLE_NAME");
                if (!tableName.matches(rex))
                {
                    log.info("ignqoring Table [" + tableName + "], doesn't match regular erpression: [" + rex + "].");
                    continue;
                }
                if (log.isDebugEnabled())
                {
                    int cols = rs.getMetaData().getColumnCount();
                    for (int i=1; i<=cols;i++)
                    {
                        Object result = null;
                        int type = rs.getMetaData().getColumnType(i);
                        result = rs.getObject(i);                    
                        log.debug("\tcol[" + i + "] of type[" + type + "]=[" + result  + "].");
                    }
                }
                if (filter != null && !filter.processTable(tableName)) {
                    log.info("Skipping table ["+tableName+"].");
                    continue;
                }

                if (log.isDebugEnabled())
                    log.info("Parsing structure of table ["+tableName+"].");

                String tableComment = rs.getString("REMARKS");

                TableInfo ti = new TableInfo(tableName);
                ti.setComment(tableComment);

                try
                {
                    this.addColumnsToTableInfo(dbMeta,ti,doCheckForAutoIncrementColumns);
                    this.addPrimaryKeyToTableInfo(dbMeta,ti);
                    this.addForeignKeysToTableInfo(dbMeta,ti);
                    this.addIndicesToTableInfo(dbMeta,ti,filter == null ? false : filter.isKeepInternalIndices());
                    if (doCheckForAutoIncrementColumns) {
                        this.checkAutoIncrementColumn(ti);
                    }
                    for(ColumnInfo ci : ti.getColumns()){
                        // fetch any additional, dialect specific column information:
                        this.getDialect().fetchAdditionalColumnInfo(this, ti, ci);
                    }
                    ret.add(ti);
                }
                catch (Exception e)
                {
                    log.error("Error [" + e.getMessage() + "] in fetchTableInfos tableName=[" + tableName + "]");
                    continue;
                }
            }
            return ret;
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#createInsertStatement(org.clazzes.jdbc2xml.schema.TableInfo, boolean)
     */
    public PreparedStatement createInsertStatement(TableInfo ti, boolean setAutoValues)
            throws SQLException {

        StringBuffer insertSql = new StringBuffer();
        String autoIncrementColumn = null;
        insertSql.append("insert into ");
        insertSql.append(ti.getName());
        insertSql.append(" (");

        List<ColumnInfo> columns = ti.getColumns();
        int i = 0;
        for (ColumnInfo ci : columns) {
            if (ci.isAutoIncrement())
            {
                autoIncrementColumn = ci.getName();
                if (!setAutoValues)
                    continue;
            }
            if (i>0)
                insertSql.append(',');

            insertSql.append(ci.getName());
            i++;
        }

        insertSql.append(") values (");

        i = 0;
        for (ColumnInfo ci : columns) {
            if (ci.isAutoIncrement() && !setAutoValues)
                continue;

            if (i>0)
                insertSql.append(',');

            insertSql.append('?');
            i++;
        }

        insertSql.append(")");

        if (autoIncrementColumn != null && !setAutoValues && this.getDialect() instanceof OracleDialect)
        {
            String generatedColumns[] = {autoIncrementColumn};
            return this.getConnection().prepareStatement(insertSql.toString(), generatedColumns);
        }
        if (autoIncrementColumn != null && setAutoValues && this.getDialect() instanceof MSSQLServerDialect)
        {
            return this.getConnection().prepareStatement("set identity_insert " + ti.getName() + " on " + insertSql.toString() + "set identity_insert " + ti.getName() + " off", Statement.RETURN_GENERATED_KEYS);
        }
        if (setAutoValues || autoIncrementColumn == null)
        {
            return this.getConnection().prepareStatement(insertSql.toString());
        }
        return this.getConnection().prepareStatement(insertSql.toString(), Statement.RETURN_GENERATED_KEYS);
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#createTable(org.clazzes.jdbc2xml.schema.TableInfo, boolean, boolean)
     */
    public void createTable(TableInfo ti, boolean addForeignKeys) throws SQLException {

        this.getDialect().pushCreateTable(this, this.queue,ti);

        if (ti.getIndices()!=null)
            createIndices(ti);

        if (addForeignKeys)
            createForeignKeys(ti);

        this.queue.perform(this.getConnection());
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#renameTable(org.clazzes.jdbc2xml.schema.TableInfo, java.lang.String)
     */
    public TableInfo renameTable(TableInfo ti, String newTableName)
            throws SQLException {

        if (ti==null) {
            throw new SQLException("renameTable(TableInfo) called with tableInfo==null.");
        }

        this.getDialect().pushRenameTable(this, this.queue,ti,newTableName);

        this.queue.perform(this.getConnection());

        ti.setName(newTableName);

        return ti;
    }

    /**
     * Calls addIndex for each index
     * @param ti
     * @throws SQLException
     */
    public void createIndices(TableInfo ti) throws SQLException {
        if (ti==null) {
            log.warn("createIndices(TableInfo) called with tableInfo==null.");
            return;
        }
        List<IndexInfo>indices=ti.getIndices();
        if (indices == null) {
            log.warn("createIndices(TableInfo) called with tableInfo.getIndices()==null.");
            return;
        }
        if (indices.size() == 0) {
            log.warn("createIndices(TableInfo) called with tableInfo.getIndices() empty.");
            return;
        }
        for (IndexInfo indexInfo : indices) {
            this.getDialect().pushAddIndex(this.queue,ti,indexInfo);
        }
        this.queue.perform(this.getConnection());
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#createForeignKeys(org.clazzes.jdbc2xml.schema.TableInfo)
     */
    public void createForeignKeys(TableInfo ti) throws SQLException {
        if (ti==null) {
            throw new SQLException("createForeignKeys(TableInfo) called with tableInfo==null.");
        }
        List<ForeignKeyInfo> foreignKeys=ti.getForeignKeys();
        if (foreignKeys==null || foreignKeys.size()==0)
            return;

        for (ForeignKeyInfo foreignKeyInfo : foreignKeys) {
            this.getDialect().pushAddForeignKey(this.queue,ti,foreignKeyInfo);
        }
        this.queue.perform(this.getConnection());
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#dropForeignKeys(org.clazzes.jdbc2xml.schema.TableInfo)
     */
    public TableInfo dropForeignKeys(TableInfo ti) throws SQLException {

        if (ti.getForeignKeys() == null) return ti;

        for (ForeignKeyInfo fki : ti.getForeignKeys())
            this.getDialect().pushDropForeignKey(this.queue,ti,fki);

        this.queue.perform(this.getConnection());

        ti.setForeignKeys(null);

        return ti;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#dropTable(org.clazzes.jdbc2xml.schema.TableInfo, boolean)
     */
    public void dropTable(TableInfo ti, boolean force) throws SQLException {

        if (ti.getForeignKeys() != null)
            for (ForeignKeyInfo fki : ti.getForeignKeys())
                this.getDialect().pushDropForeignKey(this.queue,ti,fki);

        this.getDialect().pushDropTable(this, this.queue,ti,force);
        this.queue.perform(this.getConnection());
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#dropTables(java.util.List, boolean)
     */
    public void dropTables(List<TableInfo> tables, boolean force) throws SQLException {

        if (tables == null) return;

        for (TableInfo ti : tables)
        {
            if (ti.getForeignKeys() != null)
            {
                for (ForeignKeyInfo fki : ti.getForeignKeys())
                    this.getDialect().pushDropForeignKey(this.queue,ti,fki);
            }  
        }

        for (TableInfo ti : tables)
        {
            this.getDialect().pushDropTable(this, this.queue,ti,force);
        }
        this.queue.perform(this.getConnection());
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#addColumn(org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public TableInfo addColumn(final TableInfo ti, final ColumnInfo columnInfo)
            throws SQLException 
    {
        if (ti == null)
            throw new SQLException("Can't add column [" + ((columnInfo == null) ? "null" : columnInfo.getName()) + "], TableInfo is null.");
        if (columnInfo == null)
            throw new SQLException("Can't add column to table [" + ti.getName() + "], ColumnInfo is null.");


        this.getDialect().pushAddColumn(this.queue,ti,columnInfo);
        this.queue.perform(this.getConnection());

        ti.addColumn(columnInfo);

        return ti;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#modifyColumn(org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public TableInfo modifyColumn(final TableInfo ti, final ColumnInfo columnInfo)
            throws SQLException 
    {
        if (ti == null)
            throw new SQLException("Can't modify column [" + ((columnInfo == null) ? "null" : columnInfo.getName()) + "], TableInfo is null.");
        if (columnInfo == null)
            throw new SQLException("Can't modify column of table [" + ti.getName() + "], ColumnInfo is null.");

        ColumnInfo oldColumnInfo = ti.getColumnInfo(columnInfo.getName());

        if (oldColumnInfo == null)
            throw new SQLException("Can't modify column [" + columnInfo.getName() + "] of table [" + ti.getName() + "], column [" + columnInfo.getName() + "] does not exist.");

        this.getDialect().pushModifyColumn(this, this.queue,ti,oldColumnInfo,columnInfo);
        this.queue.perform(this.getConnection());

        ti.replaceColumnInfo(columnInfo.getName(), columnInfo);

        return ti;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#changeColumn(org.clazzes.jdbc2xml.schema.TableInfo, java.lang.String, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public TableInfo changeColumn(final TableInfo ti, String oldColumnName, final ColumnInfo columnInfo)
            throws SQLException 
    {
        if (ti == null)
            throw new SQLException("Can't change column [" + ((columnInfo == null) ? "null" : columnInfo.getName()) + "], TableInfo is null.");
        if (columnInfo == null)
            throw new SQLException("Can't change column of table [" + ti.getName() + "], ColumnInfo is null.");

        // optimize the non-rename case here, because
        // Dialect.changeColumn() assumes two different column values.
        if (oldColumnName.equals(columnInfo.getName()))
            return this.modifyColumn(ti, columnInfo);                

        ColumnInfo oldColumnInfo = ti.getColumnInfo(oldColumnName);

        if (oldColumnInfo == null)
            throw new SQLException("Can't change column [" + columnInfo.getName() + "] of table [" + ti.getName() + "], column [" + columnInfo.getName() + "] does not exist.");

        this.getDialect().pushChangeColumn(this, this.queue,ti,oldColumnInfo,columnInfo);
        this.queue.perform(this.getConnection());

        ti.replaceColumnInfo(oldColumnName, columnInfo);

        return ti;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#dropColumn(org.clazzes.jdbc2xml.schema.TableInfo, java.lang.String, boolean)
     */
    public TableInfo dropColumn(final TableInfo ti, final String columnName, boolean force)
            throws SQLException 
    {
        if (ti == null)
            throw new SQLException("Can't drop column [" + ((columnName == null) ? "null" : columnName) + "], TableInfo is null.");
        if (columnName == null)
            throw new SQLException("Can't drop column of table [" + ti.getName() + "], columnName is null.");

        ColumnInfo ci = ti.getColumnInfo(columnName); 

        if (ci == null)
            throw new SQLException("Can't drop column [" + columnName + "] of table [" + ti.getName() + "], column [" + columnName + "] does not exist.");

        this.getDialect().pushDropColumn(this, this.queue,ti,ci,force);
        this.queue.perform(this.getConnection());

        ti.removeColumnInfo(columnName);

        return ti;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#addForeignKey(org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ForeignKeyInfo)
     */
    public TableInfo addForeignKey(final TableInfo ti, final ForeignKeyInfo foreignKeyInfo)
            throws SQLException 
    {
        if (ti == null)
            throw new SQLException("Can't add foreign key [" + ((foreignKeyInfo == null) ? "null" : foreignKeyInfo.getName()) + "], TableInfo is null.");
        if (foreignKeyInfo == null)
            throw new SQLException("Can't add foreign key to table [" + ti.getName() + "], ForeignKeyInfo is null.");
        if (ti.getForeignKeyInfo(foreignKeyInfo.getName()) != null)
        {
            log.warn("Can't add foreign key [" + foreignKeyInfo.getName() + "] to table [" + ti.getName() + "], [" + foreignKeyInfo.getName() + "] already exists.");
            return ti;
        }

        this.getDialect().pushAddForeignKey(this.queue,ti,foreignKeyInfo);
        this.queue.perform(this.getConnection());

        ti.addForeignKey(foreignKeyInfo);

        return ti;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#dropForeignKey(org.clazzes.jdbc2xml.schema.TableInfo, java.lang.String)
     */
    public TableInfo dropForeignKey(TableInfo ti, String fkName)
            throws SQLException 
    {
        if (ti == null)
            throw new SQLException("Can't drop foreign key [" + ((fkName == null) ? "null" : fkName) + "], TableInfo is null.");
        if (fkName == null)
            throw new SQLException("Can't drop foreign key of table [" + ti.getName() + "], name of foreign key is null.");

        ForeignKeyInfo fki = ti.getForeignKeyInfo(fkName);
        if (fki == null)
            throw new SQLException("Can't drop foreign key [" + fkName + "] from table [" + ti.getName() + "], [" + fkName + "] does not exists.");

        this.getDialect().pushDropForeignKey(this.queue,ti,fki);
        this.queue.perform(this.getConnection());

        ti.removeForeignKey(fkName);

        return ti;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#addIndex(org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.IndexInfo)
     */
    public TableInfo addIndex(TableInfo ti, IndexInfo indexInfo)
            throws SQLException {
        if (ti == null)
            throw new SQLException("Can't add index [" + ((indexInfo == null) ? "null" : indexInfo.getName()) + "], TableInfo is null.");
        if (indexInfo == null)
            throw new SQLException("Can't add index to table [" + ti.getName() + "], indexInfo is null.");
        if (ti.getIndexInfo(indexInfo.getName()) != null)
        {
            log.warn("Can't add index [" + indexInfo.getName() + "] to table [" + ti.getName() + "], [" + indexInfo.getName() + "] already exists.");
            return ti;
        }

        this.getDialect().pushAddIndex(this.queue,ti,indexInfo);
        this.queue.perform(this.getConnection());

        ti.addIndex(indexInfo);

        return ti;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#dropIndex(org.clazzes.jdbc2xml.schema.TableInfo, java.lang.String)
     */
    public TableInfo dropIndex(TableInfo ti, String indexName)
            throws SQLException {

        IndexInfo indexInfo = ti.getIndexInfo(indexName);

        if (indexInfo==null)
            throw new SQLException("dropIndex: Index ["+indexName+"] does not exit in table ["+ti.getName()+"].");

        this.getDialect().pushDropIndex(this.queue,ti,indexInfo);
        this.queue.perform(this.getConnection());

        ti.removeIndex(indexName);

        return ti;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#dropStaleBackupTables()
     */
    public void dropStaleBackupTables() throws SQLException {

        log.info("Fetching all stale backup tables from catalog ["+this.getConnection().getCatalog()+"], schema ["+this.getSchema()+"].");

        DatabaseMetaData dbMeta = this.getConnection().getMetaData();

        StringBuffer tableFilter = new StringBuffer("JDBC2XML");

        tableFilter.append(dbMeta.getSearchStringEscape());
        tableFilter.append('_');
        tableFilter.append(dbMeta.getSearchStringEscape());
        tableFilter.append('_');
        tableFilter.append('%');

        try (ResultSet rs = dbMeta.getTables(this.getConnection().getCatalog(),
                toStoredIdentifier(dbMeta,this.getSchema()),tableFilter.toString(),
                new String[] {"TABLE"})) {

            while (rs.next())
            {
                String tableName = rs.getString("TABLE_NAME");
                log.info("Dropping stale backup table ["+tableName+"].");
                
                SQLHelper.executeUpdate(this.getConnection(),DDLHelper.buildDropTable(tableName));
            }
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#commit()
     */
    public void commit() throws SQLException {

        this.queue.commit(this.getConnection());
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#rollback()
     */
    public void rollback() throws SQLException {

        this.queue.rollback(this.getConnection());
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.ISchemaEngine#getTempTableNames()
     */
    public List<String> getTempTableNames() {
        final List<String> res = new ArrayList<String>();
        final Enumeration<SqlCommand> e = this.queue.commands();
        while (e.hasMoreElements())
        {
            final SqlCommand c = e.nextElement();
            if (c.isTempTableCreated())
                res.add(c.getTempTableName());
        }
        return res;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.SchemaEngine#getMaxTableNameLength()
     */
    public int getMaxTableNameLength() {
        return this.maxTableNameLength;
    }

    @Override
    public SqlCommandQueue getQueue() {
        return this.queue;
    }

}
