/***********************************************************
 * $Id$
 *
 * Topic Map API implementation for clazzes.org
 * http://www.tmapi.org
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Created: 29 Sep 2009
 *
 ***********************************************************/
package org.clazzes.jdbc2xml.schema.impl;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.Map;

import org.clazzes.jdbc2xml.schema.ISchemaEngine;
import org.clazzes.jdbc2xml.schema.ISchemaUpdateSnippet;
import org.clazzes.jdbc2xml.schema.SchemaEngine;
import org.clazzes.jdbc2xml.schema.TableInfo;
import org.clazzes.jdbc2xml.sql.SqlCommandQueue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This is the workhorse for executing schema update snippets.
 */
public class SchemaCheckerBean {

    private static final Logger log = LoggerFactory.getLogger(SchemaCheckerBean.class);

    private final Map<String, ISchemaUpdateSnippet> updateSnippets;

    private final ISchemaEngine engine;

    private String schemaVersion;

    private final TableInfo versionHistory;

    private final String insertSql;

    private final ISchemaUpdateSnippet baseMigration;

    public static String buildInsertSql(TableInfo versionHistory) {
        return "INSERT INTO " +
                versionHistory.getName() + " (" +
                versionHistory.getColumns().get(0).getName() + ", " +
                versionHistory.getColumns().get(1).getName() + ", " +
                versionHistory.getColumns().get(2).getName() + ", " +
                versionHistory.getColumns().get(3).getName() +
                ") VALUES (?,?,?,?)";
    }

    public SchemaCheckerBean(ISchemaEngine engine,  Map<String, ISchemaUpdateSnippet> updateSnippets, TableInfo versionHistory, ISchemaUpdateSnippet baseVersion) {
        super();

        this.engine = engine;
        this.updateSnippets = updateSnippets;
        this.versionHistory = versionHistory;
        this.baseMigration = baseVersion;

        this.insertSql = buildInsertSql(versionHistory);
    }

    public synchronized void checkSchema() throws SQLException, InstantiationException, IllegalAccessException {
        if (this.engine == null)
            throw new RuntimeException("No database connection");

        if ((this.updateSnippets == null || this.updateSnippets.size() < 1)
            && this.baseMigration == null) {
            return;
        }

        fetchSchemaVersion();

        updateDatabaseSchema();
    }

    private void fetchSchemaVersion() throws SQLException {
        this.schemaVersion = null;

        try(Statement stmt = this.engine.getConnection().createStatement()) {
            try(ResultSet rs = stmt.executeQuery("SELECT VERSION FROM "+this.versionHistory.getName()+" ORDER BY SERIALNR DESC"))
            {
                if (rs.next()) {
                    this.schemaVersion = rs.getString(1);
                } else {
                    this.schemaVersion = null;
                }

            } catch (SQLException e)
            {
                if (log.isDebugEnabled())
                    log.debug("Problem detecting old DB schema. Suggesting empty DB.");

                this.schemaVersion = null;
            }
        }

    }

    // Keep an error in finally from entirely replacing the inner exception.
    private static final class EngineClosable implements AutoCloseable {
        private boolean done=false;
        private final ISchemaEngine engine;
        public EngineClosable(ISchemaEngine engine) {
            this.engine = engine;
        }
        public void commit() throws SQLException {
            if (done) {
                throw new RuntimeException();
            }
            engine.commit();
            done = true;
        }
        @Override
        public void close() throws SQLException {
            if (!done) {
                engine.rollback();
                done = true;
            }
        }
    }

    private void updateDatabaseSchema() throws SQLException, InstantiationException, IllegalAccessException
    {
        // now we have a "clean" schema 0.00
        String newVersion;
        Integer serialNr;
        String sql = "SELECT " + this.versionHistory.getColumns().get(3).getName() +
                " FROM " + this.versionHistory.getName() +
                " ORDER BY " + this.versionHistory.getColumns().get(3).getName() +
                " DESC";

        if (log.isDebugEnabled())
            log.debug("Executing query [" + sql + "]");

        try(PreparedStatement getSerialNr =
                    this.engine.getConnection().prepareStatement(sql)) {

            try (ResultSet rs = getSerialNr.executeQuery()) {
                if (rs.next())
                    serialNr = rs.getInt(1);
                else
                    serialNr = -1;
            }

            // new-style SchemaUpdateSnippet updates.
            try(PreparedStatement insertSchemaLog =
                    this.engine.getConnection().prepareStatement(this.insertSql))
            {
                while (this.schemaVersion == null
                       || (this.updateSnippets != null
                           && this.updateSnippets.containsKey(this.schemaVersion)))
                {
                    boolean success = false;

                    try (EngineClosable _e = new EngineClosable(this.engine))
                    {
                        ISchemaUpdateSnippet snippet = this.schemaVersion == null
                            ? this.baseMigration
                            : this.updateSnippets.get(this.schemaVersion);

                        newVersion = snippet.getTargetVersion();

                        log.info("Going to update DB schema from [" + this.schemaVersion + "] to [" + newVersion +"].");

                        snippet.performUpdate(this.engine);

                        insertSchemaLog.setString(1, newVersion);
                        insertSchemaLog.setString(2, snippet.getUpdateComment());
                        insertSchemaLog.setTimestamp(3, new Timestamp(System.currentTimeMillis()));
                        insertSchemaLog.setInt(4, ++serialNr);
                        insertSchemaLog.execute();

                        _e.commit();

                        log.info("Successfully updated schema from [" + this.schemaVersion
                                    + "] to [" + newVersion + "].");

                        this.schemaVersion = newVersion;
                        success = true;
                    }

                }

                // *********************************************************************************
                // end of updates
                log.info("schema is up to date (Version " + this.schemaVersion + ")");
            }
        }
    }
 }
