/* **********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 24.03.2009
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema.impl;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.Enumeration;
import java.util.Locale;
import java.util.Objects;
import java.util.Properties;

import org.clazzes.jdbc2xml.helper.SQLHelper;
import org.clazzes.jdbc2xml.helper.TypesHelper;
import org.clazzes.jdbc2xml.schema.ColumnInfo;
import org.clazzes.jdbc2xml.schema.DataTypeNotSupportedException;
import org.clazzes.jdbc2xml.schema.ForeignKeyInfo;
import org.clazzes.jdbc2xml.schema.ISchemaEngine;
import org.clazzes.jdbc2xml.schema.IndexInfo;
import org.clazzes.jdbc2xml.schema.TableInfo;
import org.clazzes.jdbc2xml.sql.SqlCommandQueue;

/**
 * This class implements Dialect for PostgreSQL 8.3
 * Indexe having INCLUDE only after versions 9.5 or 11
 * Backend SHOULD always be B-Tree (for multi column indexe, unique indexe, ...)
 */
public class PostgreSQLDialect extends AbstrDialectSupport {

    public static final String defaultDriverName = "org.postgresql.Driver";

    private static final String addColumnCommand = "ADD COLUMN";

    private static final String dropConstraintCommand = "DROP CONSTRAINT";

    private static final String renameTableCommand = "ALTER TABLE %s RENAME TO %s";

    private static final String renameColumnCommand = "ALTER TABLE %s RENAME COLUMN %s TO %s";

    private static final String changeColumnDataTypeCommand = "ALTER TABLE %s ALTER COLUMN %s TYPE %s ";

    /**
     * Normally the statement would include DEFAULT and therefore look like this: 
     * "ALTER TABLE foo ALTER COLUMN bar SET DEFAULT 'baz'". But we add this part already when 
     * we create the column default value. 
     */
    private static final String changeColumnDefaultValueCommand = "ALTER TABLE %s ALTER COLUMN %s SET %s";
    
    private static final String dropColumnDefaultValueCommand = "ALTER TABLE %s ALTER COLUMN %s DROP DEFAULT";

    private static final String setColumnNotNullCommand = "ALTER TABLE %s ALTER COLUMN %s SET NOT NULL";
    private static final String dropColumnNotNullCommand = "ALTER TABLE %s ALTER COLUMN %s DROP NOT NULL";

    public String createColumnSpec(ColumnInfo columnInfo) {
        StringBuffer columnSpec = new StringBuffer();

        columnSpec.append(columnInfo.getName());
        columnSpec.append(" ");
        try {
            columnSpec.append(this.buildColumnDataType(columnInfo));
            columnSpec.append(this.buildColumnNullValues(columnInfo));
        } catch (IllegalArgumentException e) {
            // this means that the used data type isn't supported/recoginsed.
            return null;
        }
        String defValue = this.buildColumnDefaultValue(columnInfo);
        if (defValue != null && !defValue.equals(""))
            columnSpec.append(defValue);

        return columnSpec.toString();
    }

    private String buildColumnNullValues(ColumnInfo columnInfo)
    {
        if (!columnInfo.isNullable())
            return " NOT NULL";
        return "";
    }

    private String buildColumnDefaultValue(ColumnInfo columnInfo) {
        StringBuffer spec = new StringBuffer();
        if (columnInfo.getDefaultValue() != null) {
            spec.append(" DEFAULT ");

            if (columnInfo.getType() == Types.BOOLEAN) {
                spec.append('\'');
                spec.append(columnInfo.getDefaultValue());
                spec.append('\'');
            }
            else if (TypesHelper.isNumeric(columnInfo.getType()))
                spec.append(columnInfo.getDefaultValue());
            else {
                spec.append('\'');
                this.quoteString(spec, columnInfo.getDefaultValue());
                spec.append('\'');
            }
        }
        return spec.toString();
    }

    private String buildColumnDataType(ColumnInfo columnInfo)
            throws IllegalArgumentException {
        StringBuffer spec = new StringBuffer();

        switch (columnInfo.getType()) {
        case Types.ARRAY: {
            if (columnInfo.getPrecision() == null) {
                throw new IllegalArgumentException();
            }
            spec.append("ARRAY[" + String.valueOf(columnInfo.getPrecision())
                    + "]");
        }
        break;
        case Types.BIGINT: {
            //postgres doesn't allow any precision for bigint
            if(columnInfo.isAutoIncrement()){
                spec.append("SERIAL");
            }else{
                spec.append("BIGINT");
            }
        }
        break;
        case Types.BINARY: {
            spec.append("BYTEA");
        }
        break;
        case Types.BIT: {
            if (columnInfo.getPrecision() == null) {
                spec.append("BIT");
            } else {
                spec.append("BIT(" + String.valueOf(columnInfo.getPrecision())
                        + ")");
            }
        }
        break;
        case Types.BOOLEAN: {
            spec.append("BOOLEAN");
        }
        break;
        case Types.NCHAR:
        case Types.CHAR: {
            if (columnInfo.getPrecision() == null) {
                spec.append("CHARACTER");
            } else {
                spec.append("CHARACTER("
                        + String.valueOf(columnInfo.getPrecision()) + ")");
            }
        }
        break;
        case Types.DATALINK: {
            throw new IllegalArgumentException();
        }
        // break;
        case Types.DATE: {
            spec.append("DATE");
        }
        break;
        case Types.DECIMAL: {
            if (columnInfo.getPrecision() == null) {
                spec.append("DECIMAL");
            } else {
                if (columnInfo.getScale() == null)
                    spec.append("DECIMAL("
                            + String.valueOf(columnInfo.getPrecision()) + ")");
                else
                    spec.append("DECIMAL("
                            + String.valueOf(columnInfo.getPrecision()) + ","
                            + String.valueOf(columnInfo.getScale()) + ")");
            }
        }
        break;
        case Types.DISTINCT: {
            throw new IllegalArgumentException();
        }
        // break;
        case Types.DOUBLE: {
            spec.append("DOUBLE PRECISION");
        }
        break;
        case Types.FLOAT: {
            spec.append("FLOAT");
        }
        break;
        case Types.INTEGER:
            if(columnInfo.isAutoIncrement()){
                spec.append("SERIAL");
            }else{
                spec.append("INTEGER");
            }
            break;
        case Types.JAVA_OBJECT: {
            throw new IllegalArgumentException();
        }
        // break;
        case Types.BLOB:
        case Types.LONGVARBINARY: {
            spec.append("BYTEA");
        }
        break;
        case Types.CLOB:
        case Types.LONGNVARCHAR:
        case Types.LONGVARCHAR: {
            spec.append("TEXT");
        }
        break;
        case Types.NULL: {
            throw new IllegalArgumentException();
        }
        // break;
        case Types.NUMERIC: {
            if (columnInfo.getPrecision() == null) {
                spec.append("NUMERIC");
            } else {
                if (columnInfo.getScale() == null)
                    spec.append("NUMERIC("
                            + String.valueOf(columnInfo.getPrecision()) + ")");
                else
                    spec.append("NUMERIC("
                            + String.valueOf(columnInfo.getPrecision()) + ","
                            + String.valueOf(columnInfo.getScale()) + ")");
            }
        }
        break;
        case Types.OTHER: {
            throw new IllegalArgumentException();
        }
        // break;
        case Types.REAL: {
            spec.append("REAL");
        }
        break;
        case Types.REF: {
            throw new IllegalArgumentException();
        }
        // break;
        case Types.SMALLINT: {
            spec.append("SMALLINT");
        }
        break;
        case Types.STRUCT: {
            throw new IllegalArgumentException();
        }
        // break;
        case Types.TIME: {
            spec.append("TIME");
        }
        break;
        case Types.TIMESTAMP: {
            spec.append("TIMESTAMP");
        }
        break;
        case Types.TIMESTAMP_WITH_TIMEZONE: {
            spec.append("TIMESTAMPTZ");
        }
        break;
        case Types.TINYINT: {
            spec.append("SMALLINT");
        }
        break;
        case Types.VARBINARY: {
            spec.append("BYTEA");
        }
        break;
        case Types.NVARCHAR:
        case Types.VARCHAR: {
            if (columnInfo.getPrecision() == null) {
                spec.append("CHARACTER VARYING");
            } else {
                spec.append("CHARACTER VARYING("
                        + String.valueOf(columnInfo.getPrecision()) + ")");
            }
        }
        break;

        default:
            throw new DataTypeNotSupportedException(columnInfo.getType());
        }

        return spec.toString();
    }

    public String defaultDriverName() {
        return defaultDriverName;
    }

    public String getID() {
        return "POSTGRESQL_9";
    }

    public String normalizeDefaultValue(int type, String s) {

        if (s == null) {
            return null;
        }

        if (s.startsWith("nextval(") && s.endsWith(")")) {
            // filter out interal sequences of autoincrement columns
            return null;
        }

        if (s.startsWith("'") && (
                s.endsWith("'::character varying") ||
                s.endsWith("'::bpchar") ||
                s.endsWith("'::text") )
                ) {

            // unquote ISO SQL '' -> '
            return s.substring(1,s.lastIndexOf('\'')).replace("''","'");
        }
        
        if (type == Types.BIT) {
            
            if ("true".equals(s)) {
                return "1::bit";
            }
            if ("false".equals(s)) {
                return "0::bit";
            }
        }

        return s;
    }

    public void pushAddColumn(SqlCommandQueue queue, TableInfo ti, ColumnInfo ci)
            throws SQLException {
        queue.pushCommand(DDLHelper.buildAddColumn(ti.getName(), ci, this,
                addColumnCommand, false), DDLHelper.buildDropColumn(ti.getName(), ci
                        .getName()));
    }

    public void pushAddForeignKey(SqlCommandQueue queue, TableInfo ti,
            ForeignKeyInfo fki) throws SQLException {
        queue.pushCommand(DDLHelper.buildAddForeignKey(ti.getName(), fki, false),
                DDLHelper.buildDropForeignKey(ti.getName(), fki.getName(),
                        dropConstraintCommand));
    }

    public void pushAddIndex(SqlCommandQueue queue, TableInfo ti,
            IndexInfo indexInfo) throws SQLException {
        queue.pushCommand(buildAddIndex(ti, indexInfo), buildDropIndex(ti,
                indexInfo));
    }

    private static String buildAddIndex(TableInfo ti, IndexInfo indexInfo)
            throws SQLException {
        // https://postgrespro.com/docs/postgresql/11/sql-createindex

        // multiColumn-Indexe only in some backends

        // v13: only B-Tree supports unique indexe

        // allowsIncludeColumns:
        // AFTER 11 for all versions (OpenSource; 18.10.2018)
        // AFTER 9.6 for professional versions (29.09.2016)
        // for 13: B-Tree and GIST support this

        // filterCondition: pre 9.4

        return DDLHelper.buildAddIndex(ti, indexInfo, false, true);
    }

    private static String buildDropIndex(TableInfo ti, IndexInfo indexInfo)
            throws SQLException {
        return DDLHelper.buildDropIndex(ti.getName(), indexInfo.getName(),
                false);
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushChangeColumn(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void pushChangeColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo)
                    throws SQLException {

        // PostgreSQL doesn't have a method that does all this operations at
        // once (like MySQLs modify).
        // Rename
        queue.pushCommand(String.format(renameColumnCommand, ti.getName(),
                oldColumnInfo.getName(), newColumnInfo.getName()), String
                .format(renameColumnCommand, ti.getName(), newColumnInfo
                        .getName(), oldColumnInfo.getName()));

        ColumnInfo intermediateColumnInfo = new ColumnInfo(
                newColumnInfo.getName(),
                oldColumnInfo.getType(),
                oldColumnInfo.getPrecision(),
                oldColumnInfo.getScale(),
                oldColumnInfo.isNullable(),
                oldColumnInfo.getDefaultValue(),
                oldColumnInfo.isAutoIncrement());
        
        this.pushModifyColumn(schemaEngine, queue, ti, intermediateColumnInfo, newColumnInfo);
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushCreateTable(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo)
     */
    public void pushCreateTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti)
            throws SQLException {
        queue.pushCommand(DDLHelper.buildCreateTable(ti, this, null), DDLHelper
                .buildDropTable(ti.getName()));
        String autoIncrementValueCommand = this.createAddAutoincrementValue(ti);
        if(autoIncrementValueCommand != null){
            queue.pushQuerieCommand(autoIncrementValueCommand, createAddAutoincrementValueDefault(ti));
        }
    }

    private String createAddAutoincrementValue(TableInfo ti)
            throws SQLException {
        String sql = null;
        for(ColumnInfo ci : ti.getColumns()){
            if(ci.isAutoIncrement() && ci.getNextValue() != null){
                StringBuffer spec = new StringBuffer();
                spec.append("SELECT SETVAL('");
                spec.append(ti.getName()+"_");
                spec.append(ci.getName()+"_seq'");
                spec.append(", ");
                spec.append(ci.getNextValue()-1+")");
                sql = spec.toString();
                break;
            }
        }

        return sql;
    }

    private String createAddAutoincrementValueDefault(TableInfo ti)
            throws SQLException {
        String sql = null;
        for(ColumnInfo ci : ti.getColumns()){
            if(ci.isAutoIncrement() && ci.getNextValue() != null){
                StringBuffer spec = new StringBuffer();
                spec.append("SELECT SETVAL('");
                spec.append(ti.getName()+"_");
                spec.append(ci.getName()+"_seq'");
                spec.append(", ");
                spec.append(1+")");
                sql = spec.toString();
                break;
            }
        }

        return sql;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropColumn(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo, boolean)
     */
    public void pushDropColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo ci, boolean force) throws SQLException {

        if (force)
            queue.pushCommand(DDLHelper.buildDropColumn(ti.getName(), ci
                    .getName()), DDLHelper.buildAddColumn(ti.getName(), ci,
                            this, addColumnCommand, false));
        else {
            if (!ci.isNullable()) {
                // assure, that the column is made nullable before backing up
                // the data in
                // DropColumnCommand Otherwise, the rollback will fail, because
                // creating a not-nullable column on an non-empty table fails.
                ColumnInfo ciNull = ColumnHelper.adaptNullability(ci, true);
                pushModifyColumn(schemaEngine, queue, ti, ci, ciNull);
                queue.pushCommand(new DropColumnCommand(schemaEngine, ti, ciNull, this, null,
                        addColumnCommand, false));
            } else

                queue.pushCommand(new DropColumnCommand(schemaEngine, ti, ci, this, null,
                        addColumnCommand, false));
        }
    }

    public void pushDropForeignKey(SqlCommandQueue queue, TableInfo ti,
            ForeignKeyInfo fki) throws SQLException {
        queue.pushCommand(DDLHelper.buildDropForeignKey(ti.getName(), fki
                .getName(), dropConstraintCommand), DDLHelper
                .buildAddForeignKey(ti.getName(), fki, false));
    }

    public void pushDropIndex(SqlCommandQueue queue, TableInfo ti,
            IndexInfo indexInfo) throws SQLException {
        queue.pushCommand(buildDropIndex(ti, indexInfo), buildAddIndex(ti,
                indexInfo));
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushRenameTable(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, java.lang.String)
     */
    public void pushRenameTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            String newTableName) throws SQLException {

        queue.pushCommand(DDLHelper.buildRenameTable(renameTableCommand,ti.getName(),newTableName));
    }


    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropTable(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, boolean)
     */
    public void pushDropTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti, boolean force)
            throws SQLException {
        if (force)
            queue.pushCommand(DDLHelper.buildDropTable(ti.getName()), null);
        else
            queue.pushCommand(new DropTableCommand(schemaEngine, ti, renameTableCommand, this));
    }

    
    private static String getNullChangeCmd(ColumnInfo ci) {
        
        if (ci.isNullable()) {
            return dropColumnNotNullCommand;
        }
        else {
            return setColumnNotNullCommand;
        }
    }
    
    private String buildChangeDefaultValueSql(String table, ColumnInfo ci) {
        
        if (ci.getDefaultValue() != null && ! ci.getDefaultValue().equals("")) {
        
            return String.format(Locale.ENGLISH,changeColumnDefaultValueCommand,
                    table,ci.getName(),
                    this.buildColumnDefaultValue(ci));
        }
        else {
            
            return String.format(Locale.ENGLISH,dropColumnDefaultValueCommand,
                    table,ci.getName());
        }
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushModifyColumn(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void pushModifyColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo)
                    throws SQLException {
        // Change data type
        queue.pushCommand(String.format(changeColumnDataTypeCommand, ti
                .getName(), newColumnInfo.getName(), this
                .buildColumnDataType(newColumnInfo)), String.format(
                        changeColumnDataTypeCommand, ti.getName(), oldColumnInfo
                        .getName(), this.buildColumnDataType(oldColumnInfo)));
        // Set column NULL values
        queue.pushCommand(String.format(getNullChangeCmd(newColumnInfo), ti
                .getName(), newColumnInfo.getName(), this
                .buildColumnNullValues(newColumnInfo)), String
                .format(getNullChangeCmd(oldColumnInfo), ti.getName(),
                        oldColumnInfo.getName()));
        // Change default value
        
        if (!Objects.equals(oldColumnInfo.getDefaultValue(),newColumnInfo.getDefaultValue())) {
           
            queue.pushCommand(
                    buildChangeDefaultValueSql(ti.getName(),newColumnInfo),
                    buildChangeDefaultValueSql(ti.getName(),oldColumnInfo)
                    );
        }
    }

    public void quoteString(StringBuffer sb, String s) {
        SQLHelper.quoteISOSqlString(sb, s);
    }

    public String constructJDBCURL(String hostname, Integer port,
            String databaseName, Properties properties) {
        // jdbc:postgresql://localhost:5432/foodb
        StringBuffer url = new StringBuffer("jdbc:postgresql://");
        if (hostname != null && hostname.length() > 0)
            url.append(hostname);
        else
            url.append("localhost");
        url.append(':');
        if (port != null && port > 0)
            url.append(port);
        else
            url.append("5432");
        if (databaseName != null && databaseName.length() > 0)
            url.append(databaseName);
        if (properties != null && properties.size() > 0) {
            char paramSep = '?';
            Enumeration<?> propertyNames = properties.propertyNames();
            while (propertyNames.hasMoreElements()) {
                String propertyName = propertyNames.toString();
                String propertyValue = properties.getProperty(propertyName);
                url.append(paramSep);
                paramSep = '&'; // firs ?, then &
                url.append(propertyName);
                url.append('=');
                url.append(propertyValue);
            }
        }
        return url.toString();
    }
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#getMappedSqlType(java.lang.String)
     */
    public int getMappedSqlType(String dialectDataType) {
        //TODO: Add potential other types as well.
        dialectDataType = dialectDataType.toUpperCase().replace(" ", "_").trim();
        switch (dialectDataType) {
            case "TIMESTAMPTZ": return Types.TIMESTAMP_WITH_TIMEZONE;
            case "TIMESTAMP": return Types.TIMESTAMP;
            default: throw new DataTypeNotSupportedException(dialectDataType);
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#fetchAdditionalColumnInfo(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void fetchAdditionalColumnInfo(ISchemaEngine schemaEngine, TableInfo ti, 
            ColumnInfo ci) {
        if(ci.isAutoIncrement()){
            Connection con = schemaEngine.getConnection();
            int nextValue;
            try(Statement nextVal = con.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_UPDATABLE)) {
              
                nextVal.setFetchSize(1);
                try (ResultSet result = nextVal.executeQuery("SELECT last_value FROM "+ti.getName()+"_"+ci.getName()+"_seq"))
                {
                    if(result != null && result.next()){
                        nextValue = result.getInt(1);
                        nextValue++;
                        ci.setNextValue(nextValue);
                    }   
                }
            }catch(SQLException e){
                ci.setNextValue(1);
            }
        }
    }
}
