/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 2008-06-24
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema.impl;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.Enumeration;
import java.util.Locale;
import java.util.Properties;

import org.clazzes.jdbc2xml.helper.SQLHelper;
import org.clazzes.jdbc2xml.helper.TypesHelper;
import org.clazzes.jdbc2xml.schema.ColumnInfo;
import org.clazzes.jdbc2xml.schema.DataTypeNotSupportedException;
import org.clazzes.jdbc2xml.schema.ForeignKeyInfo;
import org.clazzes.jdbc2xml.schema.ISchemaEngine;
import org.clazzes.jdbc2xml.schema.IndexInfo;
import org.clazzes.jdbc2xml.schema.TableInfo;
import org.clazzes.jdbc2xml.sql.SimpleSqlCommand;
import org.clazzes.jdbc2xml.sql.SqlCommand;
import org.clazzes.jdbc2xml.sql.SqlCommandQueue;
import org.clazzes.util.lang.Util;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class implements Dialect for Oracle 10.0.x
 * 
 * Oracle has a long list of key- and reserved words, which is a common reason for failures while creating
 * tables etc. a enumeration of these can be found here:
 * http://www.camden.rutgers.edu/HELP/Documentation/Oracle/server.815/a42525/apb.htm
 * 
 * Warning: The status of this implementation is beta.
 */
public class OracleDialect extends AbstrDialectSupport {
    
    private static final Logger log = LoggerFactory.getLogger(OracleDialect.class);

    public static final String defaultDriverName="oracle.jdbc.OracleDriver";
    
    /**
     * This property may be set via {@link #setProperty(String, String)} in
     * order to configure an overflow table space for generated tables.
     * If the {@value #OVERFLOW_TABLES_PROPERTY} is set, the overflow
     * tablespace is only applied to tables who's name match the given property.
     */
    public static final String OVERFLOW_TABLESPACE_PROPERTY="overflowTablespace";
    
    /**
     * This property may be set via {@link #setProperty(String, String)} in
     * order to configure a pattern for tabel names to which the 
     * overflow tablespace applied.
     */
    public static final String OVERFLOW_TABLES_PROPERTY="overflowTables";

    /**
     * This property may be set via {@link #setProperty(String, String)} in
     * order to configure the creation of backing indices on foreign key columns.
     * If the {@value #CREATE_BACKING_INDICES_PROPERTY} is set to
     * <code>false</code>, the default behavior of creating extra backing indices on
     * foreign key columns is turned off.
     */
    public static final String CREATE_BACKING_INDICES_PROPERTY="createBackingIndices";

    private static final String addColumnCommand="ADD";
    
    private static final String renameTableCommand = "RENAME %s TO %s";
    private static final String createTableSuffix = "ORGANIZATION INDEX";
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#getID()
     */
    public String getID() {
        
        return "Oracle_10";
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushCreateTable(java.lang.String, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo)
     */
    public void pushCreateTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti) {
     
        String overflowTbs = this.getProperty(OVERFLOW_TABLESPACE_PROPERTY);
        
        String suffix;

        if (log.isDebugEnabled()) {
            log.debug("Oracle: tableName=[{}], overflowTbs=[{}].",ti.getName(),overflowTbs);
        }
        
        if (overflowTbs == null) {
            
            suffix = createTableSuffix;
        }
        else {
            
            String overflowTables = this.getProperty(OVERFLOW_TABLES_PROPERTY);
            
            if (overflowTables == null || ti.getName().matches(overflowTables)) {
            
                suffix = createTableSuffix + " OVERFLOW TABLESPACE " + overflowTbs;
            }
            else {
                suffix = createTableSuffix;
            }
        }
        
        queue.pushCommand(
                DDLHelper.buildCreateTable(ti,this,suffix),
                DDLHelper.buildDropTable(ti.getName())
        );
        String createAutoIncrement = buildCreateAutoIncrementSequence(schemaEngine, ti, ti.getName());
        if (createAutoIncrement != null)
        {
            String dropAutoIncrement = buildDropAutoIncrementSequence(schemaEngine, ti, ti.getName());
            if (createAutoIncrement != null && dropAutoIncrement != null)
            {
                queue.pushCommand(createAutoIncrement, dropAutoIncrement);
                queue.pushCommand(buildCreateAutoIncrementTrigger(schemaEngine, ti, ti.getName()), buildDropAutoIncrementTrigger(schemaEngine, ti, ti.getName()));
            }
        }
    }
    
    private String buildDropAutoIncrementTrigger(ISchemaEngine schemaEngine, TableInfo ti, String tableName) {
        if (ti == null)
            return null;
        String autoIncrementCol = null;
        for (ColumnInfo ci : ti.getColumns())
        {
            if (ci.isAutoIncrement())
            {
                autoIncrementCol = ci.getName();
                break;
            }
        }
        if (autoIncrementCol != null)
        {
            String triggerName = this.createTriggerName(schemaEngine, tableName, autoIncrementCol);
            String sql = "drop trigger " + triggerName;
            return sql;
        }
        return null;
    }
    
    private String buildCreateAutoIncrementTrigger(ISchemaEngine schemaEngine, TableInfo ti, String tableName)
    {
        if (ti == null)
            return null;
        String autoIncrementCol = null;
        for (ColumnInfo ci : ti.getColumns())
        {
            if (ci.isAutoIncrement())
            {
                autoIncrementCol = ci.getName();
                break;
            }
        }
        if (autoIncrementCol != null)
        {
            String seqName = createSequenceName(schemaEngine, ti.getName(), autoIncrementCol);   
            String triggerName = createTriggerName(schemaEngine, ti.getName(), autoIncrementCol);

            String sql = "CREATE OR REPLACE TRIGGER " + triggerName + 
            " BEFORE INSERT ON " + ti.getName() + 
            " REFERENCING NEW AS NEW" +
            " FOR EACH ROW BEGIN" +
            "   IF :NEW." + autoIncrementCol + " IS NULL THEN" +
            "     SELECT " + seqName + ".NEXTVAL INTO :NEW." + autoIncrementCol + " FROM DUAL;" +
            "   END IF;" +
            " END;";
            return sql;
        }

        return null;
    }

    private String buildDropAutoIncrementSequence(ISchemaEngine schemaEngine, TableInfo ti, String tableName) {
        if (ti == null)
            return null;
        String autoIncrementCol = null;
        for (ColumnInfo ci : ti.getColumns())
        {
            if (ci.isAutoIncrement())
            {
                autoIncrementCol = ci.getName();
                break;
            }
        }
        if (autoIncrementCol != null)
        {
            String seqName = createSequenceName(schemaEngine, tableName, autoIncrementCol);
            String sql = "DROP SEQUENCE " + seqName;
            return sql;
        }
        return null;
    }
    
    private String createSequenceName(ISchemaEngine schemaEngine, String tableName, String autoIncrementCol)
    {
        return NameHelper.buildTempTableName(schemaEngine, tableName, autoIncrementCol, "sq").toUpperCase();
    }

    private String createTriggerName(ISchemaEngine schemaEngine, String tableName, String autoIncrementCol)
    {
        return NameHelper.buildTempTableName(schemaEngine, tableName, autoIncrementCol, "tr").toUpperCase();
    }

    private String buildCreateAutoIncrementSequence(ISchemaEngine schemaEngine, TableInfo ti, String tableName) {
        if (ti == null)
            return null;
        String autoIncrementCol = null;
        for (ColumnInfo ci : ti.getColumns())
        {
            if (ci.isAutoIncrement())
            {
                autoIncrementCol = ci.getName();
                break;
            }
        }
        if (autoIncrementCol != null)
        {
            String seqName = createSequenceName(schemaEngine, tableName, autoIncrementCol);
            String sql = "CREATE SEQUENCE " + seqName  + " START WITH 1 INCREMENT BY 1 NOMAXVALUE";
            return sql;
        }
        return null;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushRenameTable(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, java.lang.String)
     */
    public void pushRenameTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            String newTableName) throws SQLException 
    {
        String dropAutoIncrementSequence = this.buildDropAutoIncrementSequence(schemaEngine, ti, ti.getName());
        if (dropAutoIncrementSequence != null)
        {
            queue.pushCommand(this.buildDropAutoIncrementTrigger(schemaEngine, ti, ti.getName()), this.buildCreateAutoIncrementTrigger(schemaEngine, ti, ti.getName()));
            queue.pushCommand(dropAutoIncrementSequence, this.buildCreateAutoIncrementSequence(schemaEngine, ti, ti.getName()));
        }
        
        queue.pushCommand(DDLHelper.buildRenameTable(renameTableCommand,ti.getName(),newTableName));

        dropAutoIncrementSequence = this.buildDropAutoIncrementSequence(schemaEngine, ti, newTableName);
        if (dropAutoIncrementSequence != null)
        {
            queue.pushCommand(this.buildCreateAutoIncrementTrigger(schemaEngine, ti, newTableName), this.buildDropAutoIncrementTrigger(schemaEngine, ti, newTableName));
            queue.pushCommand(this.buildCreateAutoIncrementSequence(schemaEngine, ti, newTableName), dropAutoIncrementSequence);
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropTable(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, boolean)
     */
    public void pushDropTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti, boolean force) throws SQLException {
        

        String dropAutoIncrementSequence = this.buildDropAutoIncrementSequence(schemaEngine, ti, ti.getName());
        if (dropAutoIncrementSequence != null)
        {
            queue.pushCommand(this.buildDropAutoIncrementTrigger(schemaEngine, ti, ti.getName()), this.buildCreateAutoIncrementTrigger(schemaEngine, ti, ti.getName()));
            queue.pushCommand(dropAutoIncrementSequence, this.buildCreateAutoIncrementSequence(schemaEngine, ti, ti.getName()));
        }
        if (force)
            queue.pushCommand(DDLHelper.buildDropTable(ti.getName()), null);
        else
            queue.pushCommand(new DropTableCommand(schemaEngine, ti,renameTableCommand, this));
    }

    private static StringBuffer appendColumnType(StringBuffer columnSpec, ColumnInfo columnInfo)
    {
        switch (columnInfo.getType()) {
        case Types.BIGINT: {
            columnSpec.append("NUMBER");
        }
        break;
        case Types.BINARY: {
            if (columnInfo.getPrecision() == null)
                return null;
            columnSpec.append("BLOB("
                    + String.valueOf(columnInfo.getPrecision()) + ")");
        }
        break;
        case Types.BIT: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("NUMBER(1)");
            } else {
                columnSpec.append("NUMBER("
                        + String.valueOf(columnInfo.getPrecision()) + ")");
            }
        }
        break;
        case Types.BLOB: {
            columnSpec.append("BLOB");
        }
        break;
        case Types.BOOLEAN: {
            columnSpec.append("NUMBER(1)");
        }
        break;
        case Types.NCHAR:
        case Types.CHAR: {
            if (columnInfo.getPrecision() == null)
                return null;
            columnSpec.append("CHAR("
                    + String.valueOf(columnInfo.getPrecision()) + ")");
        }
        break;
        case Types.CLOB: {
            columnSpec.append("CLOB");
        }
        break;
        case Types.DATALINK: {
            return columnSpec.append("FILE");
        }
        //break;
        case Types.DATE: {
            columnSpec.append("DATE");
        }
        break;
        case Types.DECIMAL: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("NUMBER");
            } else {
                if (columnInfo.getScale() == null)
                    columnSpec.append("NUMBER("
                            + String.valueOf(columnInfo.getPrecision()) + ")");
                else
                    columnSpec.append("NUMBER("
                            + String.valueOf(columnInfo.getPrecision()) + ","
                            + String.valueOf(columnInfo.getScale()) + ")");
            }
        }
        break;
        case Types.DOUBLE: 
            columnSpec.append("BINARY_DOUBLE");
        break;
        case Types.FLOAT: 
            columnSpec.append("BINARY_FLOAT");
        break;
        case Types.INTEGER:
            columnSpec.append("NUMBER(11)");
        break;
        case Types.LONGVARBINARY: {
            columnSpec.append("BLOB");
        }
        break;
        case Types.LONGNVARCHAR:
        case Types.LONGVARCHAR: {
            columnSpec.append("CLOB");
        }
        break;
        case Types.NUMERIC: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("NUMBER");
            } else {
                if (columnInfo.getScale() == null)
                    columnSpec.append("NUMBER("
                            + String.valueOf(columnInfo.getPrecision()) + ")");
                else
                    columnSpec.append("NUMBER("
                            + String.valueOf(columnInfo.getPrecision()) + ","
                            + String.valueOf(columnInfo.getScale()) + ")");
            }
        }
        break;
        case Types.REAL: {
            columnSpec.append("NUMBER(*,7)");
        }
        break;
        case Types.SMALLINT: {
            columnSpec.append("NUMBER(6)");
        }
        break;
        case Types.TIME: {
            columnSpec.append("TIMESTAMP");
        }
        break;
        case Types.TIMESTAMP: {
            columnSpec.append("TIMESTAMP");
        }
        break;
        case Types.TIMESTAMP_WITH_TIMEZONE: {
            columnSpec.append("TIMESTAMP WITH TIME ZONE");
        }
        break;
        case Types.TINYINT: {
            columnSpec.append("NUMBER(3)");
        }
        break;
        case Types.VARBINARY: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("RAW(255)");
            } else {
                columnSpec.append("RAW("
                        + String.valueOf(columnInfo.getPrecision()) + ")");
            }
        }
        break;
        case Types.NVARCHAR:
        case Types.VARCHAR: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("NVARCHAR2(255)");
            } else {
                columnSpec.append("NVARCHAR2("
                        + String.valueOf(columnInfo.getPrecision()) + ")");
            }
        }
        break;

        default:
            throw new DataTypeNotSupportedException(columnInfo.getType());
        }
        return columnSpec;
    }
    
    private void appendDefaultValue(StringBuffer columnSpec, ColumnInfo columnInfo, boolean query)
    {
        if (columnInfo.getDefaultValue() != null)
        {
            if (!query)
                columnSpec.append(" default ");
            
            if (TypesHelper.isNumeric(columnInfo.getType()))
                columnSpec.append(columnInfo.getDefaultValue());
            else
            {
                columnSpec.append('\'');
                this.quoteString(columnSpec,columnInfo.getDefaultValue());
                columnSpec.append('\'');
            }
        }
    }
    
    /*
     * (non-Javadoc)
     * 
     * @see org.clazzes.jdbc2xml.schema.Dialect#createColumnSpec(org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public String createColumnSpec(ColumnInfo columnInfo) {

        StringBuffer columnSpec = new StringBuffer();
        
        columnSpec.append(columnInfo.getName());
        columnSpec.append(" ");
        
        if (appendColumnType(columnSpec,columnInfo) == null) return null;
        
        this.appendDefaultValue(columnSpec, columnInfo, false);

        if (!columnInfo.isNullable())
            columnSpec.append(" NOT NULL");
        
        return columnSpec.toString();
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#defaultDriverName()
     */
    public String defaultDriverName() {
        return defaultDriverName;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#quoteString(java.lang.StringBuffer, java.lang.String)
     */
    public void quoteString(StringBuffer sb, String s) {
        SQLHelper.quoteISOSqlString(sb,s);
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#normalizeDefaultValue(int, java.lang.String)
     */
    public String normalizeDefaultValue(int type, String s) {
        
        if (s==null)
            return null;
        
        s = s.substring(0, s.length() - 1);
        if (TypesHelper.isNumeric(type))
        {
            if (s.startsWith("((") && s.endsWith("))")) {
                return s.substring(2,s.length()-2);
            }
            return s.trim();
        }
        else
        {
            if ("''".equals(s))
                return null;
            if (s.startsWith("''") && s.endsWith("''")) {
                s = s.substring(2,s.length()-2);
            }
            s = SQLHelper.unquoteISOSqlString(s);
        }
            
        return s;
    }

    /**
     * Create a backing index name for a a corresponding foreign key.
     * This is her in order to mitigate "TM Contention" errors with concurrent
     * access to tables on both sides of a foreign key.
     * 
     * See http://logicalread.solarwinds.com/solving-oracle-enq-tm-contention-waits-dr01
     * for details.
     * 
     * @param fk The foreign key to back by the returned index.
     * @return A name of the corresponding backing index.
     */
    private static final IndexInfo makeBackingIndexInfo(ForeignKeyInfo fk) {
        
        IndexInfo ret = new IndexInfo();
        
        if (fk.getName() != null) {
            String fkNameUpper = fk.getName().toUpperCase(Locale.ENGLISH);
            
            if (fkNameUpper.endsWith("_FK")) {
                ret.setName(fkNameUpper.substring(0,fk.getName().length()-3) + "_BK");
            }
            else if (fkNameUpper.startsWith("FK_")) {
                ret.setName("BK_" + fkNameUpper.substring(3));
            }
            else {
                ret.setName(fkNameUpper + "_BK");
            }
        }
        
        ret.setColumns(fk.getColumns());
        ret.setUnique(false);
        
        return ret;
    }
    
    private static String buildDropForeignKey(TableInfo ti,
            ForeignKeyInfo fki)
    {
        return DDLHelper.buildDropForeignKey(ti.getName(),fki.getName(),"DROP CONSTRAINT");
    }
    
    private static class AddForeignKeyCommand extends SimpleSqlCommand
    {
        private final String addKey;
        private final String dropKey;
        
        public AddForeignKeyCommand(TableInfo ti,
                                    ForeignKeyInfo fki) throws SQLException {
            super(DDLHelper.buildAddForeignKey(ti.getName(), fki, false),buildDropForeignKey(ti, fki));
            
            IndexInfo bkInfo = makeBackingIndexInfo(fki);
           
            this.addKey = DDLHelper.buildAddIndex(ti,bkInfo,false,false);
            this.dropKey = DDLHelper.buildDropIndex(ti.getName(),bkInfo.getName(),false);
        }

        @Override
        public void perform(Connection connection) throws SQLException {
            
            if (log.isDebugEnabled()) {
                log.debug("Running backing key create statement ["+this.addKey+"].");
            }
            SQLHelper.executeUpdate(connection,this.addKey);
            
            super.perform(connection);
        }

        /* (non-Javadoc)
         * @see org.clazzes.jdbc2xml.sql.SimpleSqlCommand#rollback(java.sql.Connection)
         */
        @Override
        public void rollback(Connection connection) throws SQLException {
            super.rollback(connection);
            
            if (log.isDebugEnabled())
                log.debug("Running rollback statement ["+this.dropKey+"].");
                
            try {
                SQLHelper.executeUpdate(connection,this.dropKey);
            } catch(SQLException e) {
                
                // Ignore drop errors, because the backing index is not
                // there anymore. This may be the case, when the backing index
                // has been created by the user himself.
                //
                // ORA-01418: specified index does not exist
                //
                if (e.getErrorCode() == 1418) {
                    if (log.isDebugEnabled())
                        log.debug("SQL error 1418 dropping backing index through command ["+this.dropKey+
                                "] during rollback of add foreign key.");
                }
                else {
                    throw e;
                }
            }
       }
    }
    
    private static class DropForeignKeyCommand extends SimpleSqlCommand
    {
        private final String addKey;
        private final String dropKey;
        
        public DropForeignKeyCommand(TableInfo ti,
                                    ForeignKeyInfo fki) throws SQLException {
            super(buildDropForeignKey(ti, fki),DDLHelper.buildAddForeignKey(ti.getName(), fki, false));
            IndexInfo bkInfo = makeBackingIndexInfo(fki);
            
            this.addKey = DDLHelper.buildAddIndex(ti,bkInfo,false,false);
            this.dropKey = DDLHelper.buildDropIndex(ti.getName(),bkInfo.getName(),false);
        }

        /* (non-Javadoc)
         * @see org.clazzes.jdbc2xml.sql.SimpleSqlCommand#rollback(java.sql.Connection)
         */
        @Override
        public void perform(Connection connection) throws SQLException {
            
            super.perform(connection);
            
            if (log.isDebugEnabled())
                log.debug("Running drop backing key statement ["+this.dropKey+"].");
                
            try {
                SQLHelper.executeUpdate(connection,this.dropKey);
            } catch(SQLException e) {
                if (e.getErrorCode() == 1418) {
                    if (log.isDebugEnabled())
                        log.debug("SQL error 1418 dropping backing index through command ["+this.dropKey+"].");
                }
                else {
                    throw e;
                }
            }
        }

        @Override
        public void rollback(Connection connection) throws SQLException {
            
            if (log.isDebugEnabled()) {
                log.debug("Running backing key create statement ["+this.addKey+"] during rollback.");
            }
            
            try {
                SQLHelper.executeUpdate(connection,this.addKey);
            }
            catch (SQLException e) {
                
                //
                //  ORA-01408: such column list already indexed
                //
                if (e.getErrorCode() == 1408) {
                    if (log.isDebugEnabled())
                        log.debug("SQL error 1408 dropping backing index through command ["+this.addKey+"].");
                }
                else {
                    throw e;
                }
            }
            super.rollback(connection);
        }
    }

    protected boolean isCreateBackingIndices() {
        
        String createBackingIndices_s = this.getProperty(CREATE_BACKING_INDICES_PROPERTY);
        
        return createBackingIndices_s == null ? true :
                Boolean.parseBoolean(createBackingIndices_s);
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushAddForeignKey(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ForeignKeyInfo)
     */
    public void pushAddForeignKey(SqlCommandQueue queue, TableInfo ti,
            ForeignKeyInfo fki) throws SQLException {
        
        if (this.isCreateBackingIndices()) {
            queue.pushCommand(new AddForeignKeyCommand(ti,fki));
        }
        else {
            queue.pushCommand(new SimpleSqlCommand(DDLHelper.buildAddForeignKey(ti.getName(), fki, false),buildDropForeignKey(ti, fki)));
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropForeignKey(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ForeignKeyInfo)
     */
    public void pushDropForeignKey(SqlCommandQueue queue, TableInfo ti,
            ForeignKeyInfo fki) throws SQLException {

        if (this.isCreateBackingIndices()) {
            queue.pushCommand(new DropForeignKeyCommand(ti,fki));
        }
        else {
            queue.pushCommand(new SimpleSqlCommand(buildDropForeignKey(ti,fki),DDLHelper.buildAddForeignKey(ti.getName(),fki,false)));            
        }
    }

    private static String buildAddIndex(TableInfo ti, IndexInfo indexInfo) throws SQLException
    {
        return DDLHelper.buildAddIndex(ti,indexInfo,false,false);
    }
    
    private static String buildDropIndex(TableInfo ti, IndexInfo indexInfo) throws SQLException
    {
        return DDLHelper.buildDropIndex(ti.getName(),indexInfo.getName(),false);
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushAddIndex(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.IndexInfo)
     */
    public void pushAddIndex(SqlCommandQueue queue, TableInfo ti,
            IndexInfo indexInfo) throws SQLException {
        queue.pushCommand(
                buildAddIndex(ti,indexInfo),
                buildDropIndex(ti,indexInfo));
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropIndex(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.IndexInfo)
     */
    public void pushDropIndex(SqlCommandQueue queue, TableInfo ti,
            IndexInfo indexInfo) throws SQLException {
        queue.pushCommand(
                buildDropIndex(ti,indexInfo),
                buildAddIndex(ti,indexInfo));
    }

    private String buildAddColumn(TableInfo ti, ColumnInfo ci)
    {
        return DDLHelper.buildAddColumn(ti.getName(),ci,this,addColumnCommand, true);
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushAddColumn(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void pushAddColumn(SqlCommandQueue queue, TableInfo ti, ColumnInfo ci)
            throws SQLException {
        
        if (ci.getDefaultValue() == null)
        {    
            queue.pushCommand(
                    this.buildAddColumn(ti, ci),
                    DDLHelper.buildDropColumn(ti.getName(),ci.getName()));
        }
        else
        {
            StringBuffer sql = new StringBuffer();
            sql.append("ALTER TABLE ");
            sql.append(ti.getName());
            sql.append(" ADD ( ");
            sql.append(this.createColumnSpec(ci));
            sql.append(")");
            
            StringBuffer rsql = new StringBuffer();
            rsql.append("ALTER TABLE ");
            rsql.append(ti.getName());
            rsql.append(" DROP COLUMN ");
            rsql.append(ci.getName());
            
            queue.pushCommand(sql.toString(), rsql.toString());
            
/*            // first, create the column with no default value.
            ColumnInfo ciNull = ColumnHelper.adaptNullability(ci,true);
            ColumnInfo ciNoDefNull = ColumnHelper.adaptDefault(ciNull,null);
           
            // first, create the column with no default value and with nullable true.
            queue.pushCommand(
                    this.buildAddColumn(ti, ciNoDefNull),
                    DDLHelper.buildDropColumn(ti.getName(),ci.getName()));
            
            // add default value.
            queue.pushCommand(new ModifyDefaultValueCommand(ti.getName(),ciNull,false,this));
            
            // Fill in the default value.
            StringBuffer sql = new StringBuffer();
            sql.append("UPDATE ");
            sql.append(ti.getName());
            sql.append(" SET ");
            sql.append(ci.getName());
            sql.append(" = ");
            this.appendDefaultValue(sql,ci,true);
           
            StringBuffer rsql = new StringBuffer();
            rsql.append("UPDATE ");
            rsql.append(ti.getName());
            rsql.append(" SET ");
            rsql.append(ci.getName());
            rsql.append(" = NULL");
           
            queue.pushCommand(sql.toString(),rsql.toString());
            
            // finally set nullability to false.
            if (!ci.isNullable())
                queue.pushCommand(
                        buildAlterColumn(ti.getName(),ci),
                        buildAlterColumn(ti.getName(),ciNull));*/
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropColumn(java.lang.String, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo, boolean)
     */
    public void pushDropColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo ci, boolean force) throws SQLException {
        
        if (ci.getDefaultValue() != null)
            queue.pushCommand(new ModifyDefaultValueCommand(ti.getName(),ci,true,this));
            
        ColumnInfo ciNoDef = ColumnHelper.adaptDefault(ci,null);
        
        if (force)
        {   
            SqlCommand command = new SimpleSqlCommand(DDLHelper.buildDropColumn(ti.getName(),ci.getName()), this.buildAddColumn(ti,ciNoDef));
            queue.pushCommand(command);
        }
        else
        {
            if (!ci.isNullable())
            {
                // assure, that the column is made nullable before backing up the data in
                // DropColumnCommand Otherwise, the rollback will fail, because
                // creating a not-nullable column on an non-empty table fails.
                ColumnInfo ciNull = ColumnHelper.adaptNullability(ciNoDef,true);
                pushModifyColumnType(queue,ti,ciNoDef,ciNull);
                SqlCommand command = new DropColumnCommand(schemaEngine, ti,ciNull,this,null,addColumnCommand, true);
                queue.pushCommand(command);
            }
            else
            {
                SqlCommand command = new DropColumnCommand(schemaEngine, ti,ciNoDef,this,null,addColumnCommand, true);
                queue.pushCommand(command);
            }
        }
    }

    private static class ModifyDefaultValueCommand implements SqlCommand
    {
        private String table;
        private String column;
        private String performSql;
        private String rollbackSql;
        
        private void dropDefaultValue(Connection connection) throws SQLException
        {
            if (this.table == null || this.column == null)
                throw new SQLException("This statement has already been committed.");
            
            Statement statement = connection.createStatement();
            
            StringBuffer sql = new StringBuffer();
            sql.append("SELECT dl.name FROM sys.tables AS tl,sys.columns AS cl,sys.default_constraints AS dl WHERE tl.name = '");
            sql.append(this.table);
            sql.append("' AND cl.object_id = tl.object_id AND cl.name = '");
            sql.append(this.column);
            sql.append("' AND dl.parent_object_id = tl.object_id AND dl.parent_column_id = cl.column_id");
            
            ResultSet rs = statement.executeQuery(sql.toString());
            
            if (!rs.next())
                throw new SQLException("Deafult value contraint for column ["+this.column+
                        "] on table ["+this.table+"] could not be found.");
            
            sql.delete(0,sql.length());
            
            sql.append("ALTER TABLE ");
            sql.append(this.table);
            sql.append(" DROP CONSTRAINT ");
            sql.append(rs.getString(1));
            
            rs.close();
            
            if (log.isDebugEnabled())
                log.debug("Executing drop default value statement ["+sql+"].");
                
            statement.executeUpdate(sql.toString());
            statement.close();
        }
        
        public ModifyDefaultValueCommand(String table, ColumnInfo column,
                boolean drop, OracleDialect dialect) {
            super();
            this.table = table;
            this.column = column.getName();
            
            StringBuffer sql = new StringBuffer();
            sql.append("ALTER TABLE ");
            sql.append(table);
            sql.append(" MODIFY ( ");
            sql.append(column.getName());
            dialect.appendDefaultValue(sql,column, false);
            sql.append(" )");

            if (drop)
            {
                this.performSql = null;
                this.rollbackSql = sql.toString();
            }
            else
            {
                this.performSql = sql.toString();
                this.rollbackSql = null;
            }
        }
        
        public void perform(Connection connection) throws SQLException {
            
            if (this.performSql != null) {
                
                if (log.isDebugEnabled())
                    log.debug("Executing add default value statement ["+this.performSql+"].");
                SQLHelper.executeUpdate(connection,this.performSql);
            }
            else if (this.rollbackSql != null)
            {
                if (log.isDebugEnabled())
                    log.debug("Executing drop default value statement ["+this.rollbackSql+"].");
                SQLHelper.executeUpdate(connection,this.rollbackSql);
            }
            else
                this.dropDefaultValue(connection);
        }
        public void rollback(Connection connection) throws SQLException {
            
            if (this.rollbackSql != null) {
                
                if (log.isDebugEnabled())
                    log.debug("Executing add default value statement ["+this.rollbackSql+"] during rollback.");
                
                SQLHelper.executeUpdate(connection,this.rollbackSql);
            }
            else if (this.performSql != null)
            {
                if (log.isDebugEnabled())
                    log.debug("Executing add default value statement ["+this.performSql+"] during rollback.");
                
                SQLHelper.executeUpdate(connection,this.performSql);
            }
            else
                this.dropDefaultValue(connection);
        }

        public void cleanupOnCommit(Connection connection) throws SQLException {
            this.column = null;
            this.table = null;
            this.performSql = null;
            this.rollbackSql = null;
        }

        public String getTempTableName() {
            return null;
        }

        public boolean isTempTableCreated() {
            return false;
        }
    }
    
    private String buildAlterColumn(String tableName, ColumnInfo newColumnInfo, ColumnInfo oldColumnInfo) throws SQLException
    {
        StringBuffer sql = new StringBuffer();
        
        sql.append("ALTER TABLE ");
        sql.append(tableName);
        sql.append(" MODIFY ( ");
        sql.append(newColumnInfo.getName());
        sql.append(' ');
        if (appendColumnType(sql,newColumnInfo) == null)
            throw new SQLException("Cannot modify column ["+newColumnInfo.getName()+"] to type ["+TypesHelper.typeToString(newColumnInfo.getType())+"]: Type is not supported.");

        appendDefaultValue(sql, newColumnInfo, false);

        if (newColumnInfo.isNullable() != oldColumnInfo.isNullable())
        {
            if (newColumnInfo.isNullable())
                sql.append(" NULL");
            else
                sql.append(" NOT NULL");
        }
        sql.append(" )");
        return sql.toString();
    }
    
    private void pushModifyColumnType(SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo) throws SQLException
    {
        // assume, that the default value and the column name is equals.
        
        // check for NOOP.
        if (Util.equalsNullAware(oldColumnInfo.getPrecision(),newColumnInfo.getPrecision()) &&
                Util.equalsNullAware(oldColumnInfo.getScale(),newColumnInfo.getScale())&&
                oldColumnInfo.getType() == newColumnInfo.getType() &&
                oldColumnInfo.isNullable() == newColumnInfo.isNullable() )
            return;
        
        queue.pushCommand(
                buildAlterColumn(ti.getName(),newColumnInfo, oldColumnInfo),
                buildAlterColumn(ti.getName(),oldColumnInfo, newColumnInfo));
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushChangeColumn(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void pushChangeColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo)
            throws SQLException {
        
        // first, append a new column with the new type and NULL values allowed.
        ColumnInfo tmpInfo = ColumnHelper.adaptNullability(newColumnInfo,true);
        
        this.pushAddColumn(queue,ti,tmpInfo);
             
        // Then copy the values over from the old column
        StringBuffer sql = new StringBuffer();
        sql.append("UPDATE ");
        sql.append(ti.getName());
        sql.append(" SET ");
        sql.append(newColumnInfo.getName());
        sql.append(" = ");
        sql.append(oldColumnInfo.getName());
       
        StringBuffer rsql = new StringBuffer();
        rsql.append("UPDATE ");
        rsql.append(ti.getName());
        rsql.append(" SET ");
        rsql.append(newColumnInfo.getName());
        rsql.append(" = NULL");
       
        queue.pushCommand(sql.toString(),rsql.toString());
        
        sql.delete(0,sql.length());
        rsql.delete(0,rsql.length());
        
        // change the type of the new column to the destination type,
        // because we cannot set not null flag before copying the actual data.
        if (!newColumnInfo.isNullable())
        {
            pushModifyColumnType(queue,ti,tmpInfo,newColumnInfo);
        }
        
        // drop the column.
        this.pushDropColumn(schemaEngine, queue,ti,oldColumnInfo,true);
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushModifyColumn(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void pushModifyColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo)
            throws SQLException {

        boolean defaultValueChanged = 
            !Util.equalsNullAware(oldColumnInfo.getDefaultValue(),newColumnInfo.getDefaultValue());
        
        if (defaultValueChanged)
        {
            if (oldColumnInfo.getDefaultValue() != null)
                queue.pushCommand(new ModifyDefaultValueCommand(ti.getName(),oldColumnInfo,true,this));
            
            if (newColumnInfo.getDefaultValue() != null)
                queue.pushCommand(new ModifyDefaultValueCommand(ti.getName(),newColumnInfo,false,this));
        }
        
        pushModifyColumnType(queue,ti,oldColumnInfo,newColumnInfo);
    }
    
    /**
     * The behaviour of this method varies slightly from that described in the interface:<br/>
     * <br/>
     * The parameter <code>hostname</code> is interpreted either as a TNS-name, which is resolved
     * in the local <code>hostname.ora</code> file. This is the classic Oracle approach, which results in a URL
     * of the type <code>jdbc:oracle:thin:@&lt;TNS&gt;:&lt;PORT&gt;</code>. <br/>
     * If the hostname contains a "/", it is interpreted as <code>&lt;HOSTNAME&gt;/&lt;SID&gt;</code>. This causes
     * generation of a Oracle10-style URL with the pattern 
     * <code>jdbc:oracle:thin:@&lt;IP-HOST&gt;:&lt;PORT&gt;:&lt;SID&gt;</code><br/>
     * <br/>
     * The parameter <code>databaseName</code> is ignored, as until now Oracle does not support connecting to a
     * database specified via the JDBC-URL. To emulate this behaviour, you must either address the tables with
     * <code>&lt;SCHEMA&gt;&lt;TABLE&gt;</code> or submit the command <br/>
     * <code>alter session set current_schema = &lt;SCHEMA&gt;</code><br/>
     * after establishing a connection.
     */
    public String constructJDBCURL(String hostname, Integer port,
            String databaseName, Properties properties) {
        // jdbc:oracle:thin:@<TNS>:<PORT>[/<DBNAME>]
        // jdbc:oracle:thin:@<IP-HOST>:<PORT>:<SID>[/<DBNAME>]
        StringBuffer url=new StringBuffer("jdbc:oracle:thin:@");
        
        String iphost = null;
        String sid = null;
        
        if (hostname != null && hostname.length()>0) {
            if (hostname.contains("/")) {         // expect hostname string to be of type <IP-host>/<SID>
                iphost = hostname.substring(0, hostname.indexOf('/'));
                sid = hostname.substring(hostname.indexOf('/')+1, hostname.length());
                url.append(iphost);
            } else {
                url.append(hostname);
            }
        } else
            url.append("localhost");
        url.append(':');
        if (port!=null && port>0)
            url.append(port);
        else
            url.append("1521");
        
        if (sid != null) {
            url.append(':');
            url.append(sid);
        }
        
//        if (databaseName!=null && databaseName.length()>0) {
//            url.append("/");
//            url.append(databaseName);
//        }
        if (properties!=null && properties.size()>0) {

            url.append(';');

            Enumeration<?> propertyNames=properties.propertyNames();
            while (propertyNames.hasMoreElements()) {
                String propertyName=propertyNames.toString();
                String propertyValue=properties.getProperty(propertyName);
                url.append(propertyName);
                url.append('=');
                url.append(propertyValue);
                url.append(';');
            }
        }
        return url.toString();
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#getMappedSqlType(java.lang.String)
     */
    public int getMappedSqlType(final String dialectDataType) {
        String effectiveType = dialectDataType.replace(' ', '_')
            .replaceAll("\\(.*?\\)", "")
            .trim();
        switch (OracleDataType.valueOf(effectiveType))
        {
            case CHAR: return Types.CHAR;
            case VARCHAR2: return Types.VARCHAR;
            case VARCHAR: return Types.VARCHAR;
            case NCHAR: return Types.VARCHAR;
            case NVARCHAR2: return Types.VARCHAR;
            case DATE: return Types.DATE;
            case BLOB: return Types.BLOB;
            case CLOB: return Types.CLOB;
            case NCLOB: return Types.NCLOB;
            case BFILE: throw new DataTypeNotSupportedException(dialectDataType);
            case LONG: throw new DataTypeNotSupportedException(dialectDataType);
            case LONG_RAW: throw new DataTypeNotSupportedException(dialectDataType);
            case LONG_VARCHAR: throw new DataTypeNotSupportedException(dialectDataType);
            case RAW: return Types.BLOB;
            case NUMBER: return Types.NUMERIC;
            case DECIMAL: return Types.DECIMAL;
            case FLOAT: return Types.FLOAT;
            case INTEGER: return Types.INTEGER;
            case MLSLABEL: throw new DataTypeNotSupportedException(dialectDataType);
            case SMALLINT: return Types.SMALLINT;
            case RAW_MLSLABEL: throw new DataTypeNotSupportedException(dialectDataType);
            case ROWID: throw new DataTypeNotSupportedException(dialectDataType);
            case TIMESTAMP: return Types.TIMESTAMP;
            case TIMESTAMP_WITH_TIME_ZONE: return Types.TIMESTAMP_WITH_TIMEZONE;
            default: throw new DataTypeNotSupportedException(dialectDataType);
        }
    }

    enum OracleDataType
    {
        CHAR,
        VARCHAR2,
        VARCHAR,
        NCHAR,
        NVARCHAR2,
        DATE,
        BLOB,
        CLOB,
        NCLOB,
        BFILE,
        LONG,
        LONG_RAW,
        LONG_VARCHAR,
        RAW,
        NUMBER,
        DECIMAL,
        FLOAT,
        INTEGER,
        MLSLABEL,
        SMALLINT,
        RAW_MLSLABEL,
        ROWID,
        TIMESTAMP,
        TIMESTAMP_WITH_TIME_ZONE
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#fetchAdditionalColumnInfo(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void fetchAdditionalColumnInfo(ISchemaEngine schemaEngine, TableInfo ti, 
            ColumnInfo ci) {
        // test whether there is a autoincrement sequence for this column ...
        String seqName = createSequenceName(schemaEngine, ti.getName(), ci.getName());
        ResultSet rs = null;
        try {
            PreparedStatement ps = schemaEngine.getConnection().prepareStatement("select * from USER_SEQUENCES");
            rs = ps.executeQuery();
            while (rs.next())
            {
                String res = rs.getString(1);
                if (seqName.equalsIgnoreCase(res))
                {
                    log.debug("found sequence [" + res + "] for table=[" + ti.getName() + "], column=[" + ci.getName() + "]");
                    ci.setAutoIncrement(true);
                }
            }
        } catch (SQLException e) {
            if (log.isDebugEnabled())
                log.debug("fetchAdditionalColumnInfo: ", e);
            ; // no error, we just wanted to test if there is a sequence for this column ...
        }
        finally
        {
            if (rs != null)
                try {
                    rs.close();
                } catch (SQLException e) {
                    ; // no error
                }
        }
    }
}
