/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 27.11.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema.impl;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Enumeration;
import java.util.Properties;

import org.clazzes.jdbc2xml.helper.SQLHelper;
import org.clazzes.jdbc2xml.helper.TypesHelper;
import org.clazzes.jdbc2xml.schema.ColumnInfo;
import org.clazzes.jdbc2xml.schema.DataTypeNotSupportedException;
import org.clazzes.jdbc2xml.schema.ForeignKeyInfo;
import org.clazzes.jdbc2xml.schema.ISchemaEngine;
import org.clazzes.jdbc2xml.schema.IndexInfo;
import org.clazzes.jdbc2xml.schema.TableInfo;
import org.clazzes.jdbc2xml.sql.SimpleSqlCommand;
import org.clazzes.jdbc2xml.sql.SqlCommandQueue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class implements Dialect for MySQL 5.0.x
 * 
 * @author lech
 */
public class MySQLDialect extends AbstrDialectSupport {
    
    private static final Logger log = LoggerFactory.getLogger(MySQLDialect.class);
    
    public static final String defaultDriverName="com.mysql.jdbc.Driver";

    private static final String createTableSuffix = "ENGINE=InnoDB DEFAULT CHARSET=utf8";
    private static final String addColumnCommand="ADD COLUMN";
 
    /**
     * This property may be set via {@link #setProperty(String, String)} in
     * order to unquote default values like with newer
     * MariaDB drivers.
     */
    public static final String UNQUOTE_DEFAULT_VALUES_PROPERTY="unquoteDefaultValues";

    /**
     * @return Whether we need to unquote default values like with newer
     *         MariaDB drivers.
     */
    protected boolean isUnquoteDefaultValues() {
        
        String unquoteDefaultValues_s = this.getProperty(UNQUOTE_DEFAULT_VALUES_PROPERTY);
        
        return unquoteDefaultValues_s == null ? false :
                Boolean.parseBoolean(unquoteDefaultValues_s);
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#getID()
     */
    public String getID() {
        
        return "MYSQL_5";
    }

    private String buildCreateTable(TableInfo ti) {
        return DDLHelper.buildCreateTable(ti,this,createTableSuffix);
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushCreateTable(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo)
     */
    public void pushCreateTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti) {
     
        queue.pushCommand(
                this.buildCreateTable(ti),
                DDLHelper.buildDropTable(ti.getName())
        );
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushRenameTable(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, java.lang.String)
     */
    public void pushRenameTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            String newTableName) throws SQLException {
        
        queue.pushCommand(DDLHelper.buildRenameTable(null,ti.getName(),newTableName));
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropTable(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, boolean)
     */
    public void pushDropTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti, boolean force) {

        if (force)
            queue.pushCommand(
                    DDLHelper.buildDropTable(ti.getName()),
                    null
            );
        else
            queue.pushCommand(new DropTableCommand(schemaEngine, ti,null, this));
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.clazzes.jdbc2xml.schema.Dialect#createColumnSpec(org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public String createColumnSpec(ColumnInfo columnInfo) {

        StringBuffer columnSpec = new StringBuffer();
        
        columnSpec.append(columnInfo.getName());
        columnSpec.append(" ");
        
        switch (columnInfo.getType()) {
        case Types.BIGINT: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("BIGINT");
            } else {
                columnSpec.append("BIGINT("
                        + String.valueOf(columnInfo.getPrecision()) + ")");
            }
        }
        break;
        case Types.BINARY: {
            if (columnInfo.getPrecision() == null)
                return null;
            columnSpec.append("BINARY("
                    + String.valueOf(columnInfo.getPrecision()) + ")");
        }
        break;
        case Types.BIT: {
            if (columnInfo.getPrecision() == null || columnInfo.getPrecision() == 1) {
                columnSpec.append("BOOL");
            } else {
                columnSpec.append("BIT("
                        + String.valueOf(columnInfo.getPrecision()) + ")");
            }
        }
        break;
        case Types.BOOLEAN: {
            columnSpec.append("BOOL");
        }
        break;
        case Types.NCHAR:
        case Types.CHAR: {
            if (columnInfo.getPrecision() == null)
                return null;
            columnSpec.append("CHAR("
                    + String.valueOf(columnInfo.getPrecision()) + ")");
        }
        break;
        case Types.DATE: {
            columnSpec.append("DATE");
        }
        break;
        case Types.DECIMAL: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("DECIMAL");
            } else {
                if (columnInfo.getScale() == null)
                    columnSpec.append("DECIMAL("
                            + String.valueOf(columnInfo.getPrecision()) + ")");
                else
                    columnSpec.append("DECIMAL("
                            + String.valueOf(columnInfo.getPrecision()) + ","
                            + String.valueOf(columnInfo.getScale()) + ")");
            }
        }
        break;
        case Types.DOUBLE: {
            columnSpec.append("DOUBLE");
        }
        break;
        case Types.FLOAT: {
            columnSpec.append("FLOAT");
        }
        break;
        case Types.INTEGER:
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("INT");
            } else {
                columnSpec.append("INT("
                        + String.valueOf(columnInfo.getPrecision()) + ")");
            }
        break;
        case Types.BLOB:
        case Types.LONGVARBINARY: {
            if (columnInfo.getPrecision() == null ||
                    columnInfo.getPrecision().intValue() > 16277215)
                columnSpec.append("LONGBLOB");
            else if (columnInfo.getPrecision().intValue() > 65535)
                columnSpec.append("MEDIUMBLOB");
            else
                columnSpec.append("BLOB");
        }
        break;
        case Types.CLOB:
        case Types.LONGNVARCHAR:
        case Types.LONGVARCHAR: {
            if (columnInfo.getPrecision() == null ||
                    columnInfo.getPrecision().intValue() > 16277215)
                columnSpec.append("LONGTEXT");
            else if (columnInfo.getPrecision().intValue() > 65535)
                columnSpec.append("MEDIUMTEXT");
            else
                columnSpec.append("TEXT");
        }
        break;
        case Types.NUMERIC: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("NUMERIC");
            } else {
                if (columnInfo.getScale() == null)
                    columnSpec.append("NUMERIC("
                            + String.valueOf(columnInfo.getPrecision()) + ")");
                else
                    columnSpec.append("NUMERIC("
                            + String.valueOf(columnInfo.getPrecision()) + ","
                            + String.valueOf(columnInfo.getScale()) + ")");
            }
        }
        break;
        case Types.REAL: {
            columnSpec.append("REAL");
        }
        break;
        case Types.SMALLINT: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("SMALLINT");
            } else {
                columnSpec.append("SMALLINT("
                        + String.valueOf(columnInfo.getPrecision()) + ")");
            }
        }
        break;
        case Types.TIME: {
            columnSpec.append("TIME");
        }
        break;
        case Types.TIMESTAMP: {
            columnSpec.append("TIMESTAMP");
        }
        break;
        case Types.TIMESTAMP_WITH_TIMEZONE: {
            log.warn("TIMESTAMP_WITH_TIMEZONE not directly supported by MySQL/MariaDB engine. Using TIMESTAMP instead.");
            columnSpec.append("TIMESTAMP");
        }
        break;
        case Types.TINYINT: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("TINYINT");
            } else {
                columnSpec.append("TINYINT("
                        + String.valueOf(columnInfo.getPrecision()) + ")");
            }
        }
        break;
        case Types.VARBINARY: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("VARBINARY");
            } else {
                columnSpec.append("VARBINARY("
                        + String.valueOf(columnInfo.getPrecision()) + ")");
            }
        }
        break;
        case Types.NVARCHAR: // JDBC 4.0 type: NVARCHAR, since Java 1.6
        case Types.VARCHAR: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("VARCHAR");
            } else {
                columnSpec.append("VARCHAR("
                        + String.valueOf(columnInfo.getPrecision()) + ")");
            }
        }
        break;

        default:
            throw new DataTypeNotSupportedException(columnInfo.getType());
        }
        
        if (!columnInfo.isNullable())
            columnSpec.append(" NOT NULL");
        // NOT NULL is default for mysql's timestamp columns,
        // so enforce nullability for these.
        else if (columnInfo.getType() == Types.TIMESTAMP)
            columnSpec.append(" NULL");
        
        if (columnInfo.getDefaultValue() != null)
        {
            columnSpec.append(" default ");
            
            if (TypesHelper.isNumeric(columnInfo.getType()))
                columnSpec.append(columnInfo.getDefaultValue());
            else
            {
                columnSpec.append('\'');
                this.quoteString(columnSpec,columnInfo.getDefaultValue());
                columnSpec.append('\'');
            }
        }
        if (columnInfo.isAutoIncrement())
        {
            columnSpec.append(" AUTO_INCREMENT ");
        }
        
        return columnSpec.toString();
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#defaultDriverName()
     */
    public String defaultDriverName() {
        return defaultDriverName;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#quoteString(java.lang.StringBuffer, java.lang.String)
     */
    public void quoteString(StringBuffer sb, String s) {
        
        for (int i=0; i < s.length();++i)
        {
            char c = s.charAt(i);
            
            if (c == 0)
                sb.append("\\0");
            else {
                if (c=='\'' || c =='"' || c=='\\')
                    sb.append('\\');
               
                sb.append(c);
            }   
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#normalizeDefaultValue(int, java.lang.String)
     */
    public String normalizeDefaultValue(int type, String s) {
        
        // MariaDB connector sometimes returns default values for BIT columns
        // in the format b'0', suspsectedly also returned by newer MySQL variants,
        // see issue JDBC2XML-21
        if (type == Types.BIT && s != null && s.startsWith("b'") && s.endsWith("'")) {
            return s.substring(2,s.length()-1);
        }
        
        if (log.isDebugEnabled()) {
            log.debug("normalizeDefaultValue: {}",s);
        }
        
        if (this.isUnquoteDefaultValues()) {
            
            if (s==null || "NULL".equals(s)) {
                return null;
            }
            
            if (!TypesHelper.isNumeric(type) && s != null && s.startsWith("'") && s.endsWith("'")) {
                
                StringBuffer ret = new StringBuffer(s.length());
                
                for (int i=1;i<s.length()-1;++i) {
                    
                    char c = s.charAt(i);
                    
                    if (c=='\'' || c=='\\') {
                        
                        ++i;
                        if (i<s.length()-1) {
                            
                            char c2 = s.charAt(i);
                            
                            if (c == '\\') {
                                
                                if (c2 == 'n') {
                                    c = '\n';
                                }
                                else if (c2 == 'r') {
                                    c = '\r';
                                }
                                else if (c2 == 't') {
                                    c = '\t';
                                }
                                else {
                                    c = c2;
                                }
                            }
                            else {
                                c = c2;
                            }
                        }
                    }
                    ret.append(c);
                }
                
                return ret.toString();
            }
        }
        
        return s;
    }

    
    private static String buildDropForeignKey(TableInfo ti,
            ForeignKeyInfo fki)
    {
        return DDLHelper.buildDropForeignKey(ti.getName(),fki.getName(),"DROP FOREIGN KEY");
    }

    private static class AddForeignKeyCommand extends SimpleSqlCommand
    {
        private final String dropKey;
        
        public AddForeignKeyCommand(TableInfo ti,
                                    ForeignKeyInfo fki) throws SQLException {
            super(DDLHelper.buildAddForeignKey(ti.getName(), fki, false),buildDropForeignKey(ti, fki));
            this.dropKey = DDLHelper.buildDropIndex(ti.getName(),fki.getName(),true);
        }

        /* (non-Javadoc)
         * @see org.clazzes.jdbc2xml.sql.SimpleSqlCommand#rollback(java.sql.Connection)
         */
        @Override
        public void rollback(Connection connection) throws SQLException {
            super.rollback(connection);
            
            if (log.isDebugEnabled())
                log.debug("Running rollback statement ["+this.dropKey+"].");
                
            try {
                SQLHelper.executeUpdate(connection,this.dropKey);
            } catch(SQLException e) {
                
                // Ignore drop errors, because the backing index is not
                // there anymore. This may be the case, when the backing index
                // has been created by the user himself.
                //
                // Error: 1091 SQLSTATE: 42000 (ER_CANT_DROP_FIELD_OR_KEY)
                // Message: Can't DROP '%s'; check that column/key exists 
                //
                if (e.getErrorCode() == 1091) {
                    if (log.isDebugEnabled())
                        log.debug("SQL error 1091 dropping backing index through command ["+this.dropKey+
                                "] during rollback of add foreign key.");
                }
                else
                    throw e;
            }
       }
    }
    
    private static class DropForeignKeyCommand extends SimpleSqlCommand
    {
        private final String dropKey;
        
        public DropForeignKeyCommand(TableInfo ti,
                                    ForeignKeyInfo fki) throws SQLException {
            super(buildDropForeignKey(ti, fki),DDLHelper.buildAddForeignKey(ti.getName(), fki, false));
            this.dropKey = DDLHelper.buildDropIndex(ti.getName(),fki.getName(),true);
        }

        /* (non-Javadoc)
         * @see org.clazzes.jdbc2xml.sql.SimpleSqlCommand#rollback(java.sql.Connection)
         */
        @Override
        public void perform(Connection connection) throws SQLException {
            super.perform(connection);
            
            if (log.isDebugEnabled())
                log.debug("Running ["+this.dropKey+"].");
                
            try {
                SQLHelper.executeUpdate(connection,this.dropKey);
            } catch(SQLException e) {
                if (e.getErrorCode() == 1091) {
                    if (log.isDebugEnabled())
                        log.debug("SQL error 1091 dropping backing index through command ["+this.dropKey+"].");
                }
                else
                    throw e;
            }
       }
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushAddForeignKey(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ForeignKeyInfo)
     */
    public void pushAddForeignKey(SqlCommandQueue queue, TableInfo ti,
            ForeignKeyInfo fki) throws SQLException {
        queue.pushCommand(new AddForeignKeyCommand(ti,fki));
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropForeignKey(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ForeignKeyInfo)
     */
    public void pushDropForeignKey(SqlCommandQueue queue, TableInfo ti,
            ForeignKeyInfo fki) throws SQLException {
        queue.pushCommand(new DropForeignKeyCommand(ti,fki));
    }

    private static String buildAddIndex(TableInfo ti, IndexInfo indexInfo) throws SQLException
    {
        return DDLHelper.buildAddIndex(ti,indexInfo,true,false);
    }
    
    private static String buildDropIndex(TableInfo ti, IndexInfo indexInfo) throws SQLException
    {
        return DDLHelper.buildDropIndex(ti.getName(),indexInfo.getName(),true);
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushAddIndex(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.IndexInfo)
     */
    public void pushAddIndex(SqlCommandQueue queue, TableInfo ti,
            IndexInfo indexInfo) throws SQLException {
        queue.pushCommand(
                buildAddIndex(ti,indexInfo),
                buildDropIndex(ti,indexInfo));
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropIndex(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.IndexInfo)
     */
    public void pushDropIndex(SqlCommandQueue queue, TableInfo ti,
            IndexInfo indexInfo) throws SQLException {
        queue.pushCommand(
                buildDropIndex(ti,indexInfo),
                buildAddIndex(ti,indexInfo));
    }

    private String buildAddColumn(TableInfo ti, ColumnInfo ci)
    {
        return DDLHelper.buildAddColumn(ti.getName(),ci,this,addColumnCommand, false);
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushAddColumn(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void pushAddColumn(SqlCommandQueue queue, TableInfo ti, ColumnInfo ci)
            throws SQLException {
        
        queue.pushCommand(
                this.buildAddColumn(ti, ci),
                DDLHelper.buildDropColumn(ti.getName(),ci.getName()));
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropColumn(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo, boolean)
     */
    public void pushDropColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo ci, boolean force) throws SQLException {
        
        if (force)
            queue.pushCommand(
                    DDLHelper.buildDropColumn(ti.getName(),ci.getName()),
                    this.buildAddColumn(ti, ci));
        else
        {
            if (!ci.isNullable())
            {
                // assure, that the column is made nullable before backing up the data in
                // DropColumnCommand Otherwise, the rollback will fail, because
                // creating a not-nullable column on an non-empty table fails.
                ColumnInfo ciNull = ColumnHelper.adaptNullability(ci,true);
                pushModifyColumn(schemaEngine, queue,ti,ci,ciNull);
                queue.pushCommand(new DropColumnCommand(schemaEngine, ti,ciNull,this,null,addColumnCommand, false));
            }
            else
   
            queue.pushCommand(new DropColumnCommand(schemaEngine, ti,ci,this,createTableSuffix,addColumnCommand, false));
        }
    }

    private String buildChangeColumn(TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo)
    {
        final StringBuffer sql = new StringBuffer();

        sql.append("ALTER TABLE ");
        sql.append(ti.getName());
        sql.append(' ');
        sql.append("CHANGE COLUMN ");
        sql.append(oldColumnInfo.getName());
        sql.append(' ');
        
        sql.append(this.createColumnSpec(newColumnInfo));
        
        return sql.toString();
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushChangeColumn(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void pushChangeColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo)
            throws SQLException {
        
        queue.pushCommand(
                this.buildChangeColumn(ti, oldColumnInfo, newColumnInfo),
                this.buildChangeColumn(ti, newColumnInfo, oldColumnInfo) );
    }

    private String buildModifyColumn(TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo)
    {
        final StringBuffer sql = new StringBuffer();

        sql.append("ALTER TABLE ");
        sql.append(ti.getName());
        sql.append(' ');
        sql.append("MODIFY COLUMN ");
        sql.append(this.createColumnSpec(newColumnInfo));
        
        return sql.toString();
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushModifyColumn(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void pushModifyColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo)
            throws SQLException {
        
        queue.pushCommand(
                this.buildModifyColumn(ti, oldColumnInfo, newColumnInfo),
                this.buildModifyColumn(ti, newColumnInfo, oldColumnInfo) );
    }
    
    /**
     * This method acts as an extension point for supporting variaous MySQL-like dialects.
     * 
     * @return The URL scheme used for this dialect.
     */
    protected String getUrlScheme() {
        return "jdbc:mysql";
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#constructJDBCURL(java.lang.String, java.lang.Integer, java.lang.String, java.util.Properties)
     */
    public String constructJDBCURL(String hostname, Integer port,
            String databaseName, Properties properties) {
        // jdbc:mysql://localhost:3306/foodb
        StringBuffer url=new StringBuffer(this.getUrlScheme());
        url.append("://");
        if (hostname != null && hostname.length()>0)
            url.append(hostname);
        else
            url.append("localhost");
        url.append(':');
        if (port!=null && port>0)
            url.append(port);
        else
            url.append("3306");
        if (databaseName!=null && databaseName.length()>0) {
            url.append("/");
            url.append(databaseName);
        }
        if (properties!=null && properties.size()>0) {
            char paramSep='?';
            Enumeration<?> propertyNames=properties.propertyNames();
            while (propertyNames.hasMoreElements()) {
                String propertyName=propertyNames.toString();
                String propertyValue=properties.getProperty(propertyName);
                url.append(paramSep); paramSep='&'; // firs ?, then &
                url.append(propertyName);
                url.append('=');
                url.append(propertyValue);
            }
        }
        return url.toString();
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#getMappedSqlType(java.lang.String)
     */
    public int getMappedSqlType(String dialectDataType) {
        dialectDataType = dialectDataType.toUpperCase().replace(" ", "_").trim();
        switch (dialectDataType) {
            case "TIMESTAMP": return Types.TIMESTAMP;
            default: throw new DataTypeNotSupportedException(dialectDataType);
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#fetchAdditionalColumnInfo(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void fetchAdditionalColumnInfo(ISchemaEngine schemaEngine, TableInfo ti, 
            ColumnInfo ci) {
    }
}
