/***********************************************************
 * $Id$
 *
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 27.11.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.jdbc2xml.schema.impl;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.Enumeration;
import java.util.Properties;

import org.clazzes.jdbc2xml.helper.SQLHelper;
import org.clazzes.jdbc2xml.helper.TypesHelper;
import org.clazzes.jdbc2xml.schema.ColumnInfo;
import org.clazzes.jdbc2xml.schema.DataTypeNotSupportedException;
import org.clazzes.jdbc2xml.schema.ForeignKeyInfo;
import org.clazzes.jdbc2xml.schema.ISchemaEngine;
import org.clazzes.jdbc2xml.schema.IndexInfo;
import org.clazzes.jdbc2xml.schema.TableInfo;
import org.clazzes.jdbc2xml.sql.SqlCommand;
import org.clazzes.jdbc2xml.sql.SqlCommandQueue;
import org.clazzes.util.lang.Util;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class implements Dialect for MySQL 5.0.x
 *
 * @author lech
 */
public class MSSQLServerDialect extends AbstrDialectSupport {

    private static final Logger log = LoggerFactory.getLogger(MSSQLServerDialect.class);

    public static final String defaultDriverName="com.microsoft.sqlserver.jdbc.SQLServerDriver";

    private static final String addColumnCommand="ADD";

    private static final String renameTableCommand="EXEC sp_rename '%s', '%s'";

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#getID()
     */
    public String getID() {

        return "MSSQLServer_9";
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushCreateTable(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo)
     */
    public void pushCreateTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti) {

        queue.pushCommand(
                DDLHelper.buildCreateTable(ti,this,null),
                DDLHelper.buildDropTable(ti.getName())
        );
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushRenameTable(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, java.lang.String)
     */
    public void pushRenameTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            String newTableName) throws SQLException {

        queue.pushCommand(DDLHelper.buildRenameTable(renameTableCommand,ti.getName(),newTableName));
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropTable(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, boolean)
     */
    public void pushDropTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti, boolean force) throws SQLException {

        if (force)
            queue.pushCommand(
                    DDLHelper.buildDropTable(ti.getName()),
                    null
            );
        else
            queue.pushCommand(new DropTableCommand(schemaEngine, ti,this,null));
    }

    private static StringBuffer appendColumnType(StringBuffer columnSpec, ColumnInfo columnInfo)
    {
        switch (columnInfo.getType()) {
        case Types.BIGINT: {
            columnSpec.append("BIGINT");
        }
        break;
        case Types.BINARY: {
            if (columnInfo.getPrecision() == null)
                return null;
            columnSpec.append("BINARY("
                    + String.valueOf(columnInfo.getPrecision()) + ")");
        }
        break;
        case Types.BIT: {
            columnSpec.append("BIT");
        }
        break;
        case Types.BLOB: {
            columnSpec.append("IMAGE");
        }
        break;
        case Types.BOOLEAN: {
            // what's better here?
            columnSpec.append("BIT");
            //columnSpec.append("TINYINT(1)");
        }
        break;
        case Types.NCHAR:
        case Types.CHAR: {
            if (columnInfo.getPrecision() == null)
                return null;
            columnSpec.append("CHAR("
                    + String.valueOf(columnInfo.getPrecision()) + ")");
        }
        break;
        case Types.CLOB: {
            columnSpec.append("IMAGE");
        }
        break;
        case Types.DATE: {
            if (columnInfo.getPrecision() != null) {
                columnSpec.append("DATETIME2("+columnInfo.getPrecision()+")");
            } else {
                columnSpec.append("DATETIME");
            }
        }
        break;
        case Types.DECIMAL: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("DECIMAL");
            } else {
                if (columnInfo.getScale() == null)
                    columnSpec.append("DECIMAL("
                            + String.valueOf(columnInfo.getPrecision()) + ")");
                else
                    columnSpec.append("DECIMAL("
                            + String.valueOf(columnInfo.getPrecision()) + ","
                            + String.valueOf(columnInfo.getScale()) + ")");
            }
        }
        break;
        case Types.DOUBLE: {
            columnSpec.append("FLOAT(53)");
        }
        break;
        case Types.FLOAT: {
            columnSpec.append("FLOAT(24)");
        }
        break;
        case Types.INTEGER:
            columnSpec.append("INT");
        break;
        case Types.LONGVARBINARY: {
            columnSpec.append("IMAGE");
        }
        break;
        case Types.LONGVARCHAR:
        case Types.LONGNVARCHAR: {
            columnSpec.append("NTEXT");
        }
        break;
        case Types.NUMERIC: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("NUMERIC");
            } else {
                if (columnInfo.getScale() == null)
                    columnSpec.append("NUMERIC("
                            + String.valueOf(columnInfo.getPrecision()) + ")");
                else
                    columnSpec.append("NUMERIC("
                            + String.valueOf(columnInfo.getPrecision()) + ","
                            + String.valueOf(columnInfo.getScale()) + ")");
            }
        }
        break;
        case Types.REAL: {
            columnSpec.append("REAL");
        }
        break;
        case Types.SMALLINT: {
            columnSpec.append("SMALLINT");
        }
        break;
        case Types.TIME: {
            columnSpec.append("DATETIME");
        }
        break;
        case Types.TIMESTAMP: {
            columnSpec.append("DATETIME");
        }
        break;
        case Types.TIMESTAMP_WITH_TIMEZONE: {
            columnSpec.append("DATETIMEOFFSET");
        }
        break;
        case Types.TINYINT: {
            columnSpec.append("TINYINT");
        }
        break;
        case Types.VARBINARY: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("VARBINARY(255)");
            } else {
                columnSpec.append("VARBINARY("
                        + String.valueOf(columnInfo.getPrecision()) + ")");
            }
        }
        break;
        case Types.NVARCHAR:
        case Types.VARCHAR: {
            if (columnInfo.getPrecision() == null) {
                columnSpec.append("NVARCHAR(255)");
            } else {
                columnSpec.append("NVARCHAR("
                        + String.valueOf(columnInfo.getPrecision()) + ")");
            }
        }
        break;

        default:
            throw new DataTypeNotSupportedException(columnInfo.getType());
        }
        return columnSpec;
    }

    private void appendDefaultValue(StringBuffer columnSpec, ColumnInfo columnInfo, boolean query)
    {
        if (columnInfo.getDefaultValue() != null)
        {
            if (!query)
                columnSpec.append(" default ");

            if (TypesHelper.isNumeric(columnInfo.getType()))
                columnSpec.append(columnInfo.getDefaultValue());
            else
            {
                columnSpec.append('\'');
                this.quoteString(columnSpec,columnInfo.getDefaultValue());
                columnSpec.append('\'');
            }
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see org.clazzes.jdbc2xml.schema.Dialect#createColumnSpec(org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public String createColumnSpec(ColumnInfo columnInfo) {

        StringBuffer columnSpec = new StringBuffer();

        columnSpec.append(columnInfo.getName());
        columnSpec.append(" ");

        if (appendColumnType(columnSpec,columnInfo) == null) return null;

        this.appendDefaultValue(columnSpec, columnInfo, false);

        if (!columnInfo.isNullable())
            columnSpec.append(" NOT NULL");

        if (columnInfo.isAutoIncrement())
        {
            columnSpec.append(" IDENTITY NOT FOR REPLICATION");
        }
        return columnSpec.toString();
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#defaultDriverName()
     */
    public String defaultDriverName() {
        return defaultDriverName;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#quoteString(java.lang.StringBuffer, java.lang.String)
     */
    public void quoteString(StringBuffer sb, String s) {
        SQLHelper.quoteISOSqlString(sb,s);
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#normalizeDefaultValue(int, java.lang.String)
     */
    public String normalizeDefaultValue(int type, String s) {

        if (s==null)
            return null;

        if (TypesHelper.isNumeric(type))
        {
            if (s.startsWith("((") && s.endsWith("))")) {
                return s.substring(2,s.length()-2);
            }
        }
        else
        {
            if (s.startsWith("('") && s.endsWith("')")) {
                // unquote ISO SQL string '' -> '
                return s.substring(2,s.length()-2).replace("''","'");
            }
        }

        return s;
    }

    private static String buildDropForeignKey(TableInfo ti,
            ForeignKeyInfo fki)
    {
        return DDLHelper.buildDropForeignKey(ti.getName(),fki.getName(),"DROP CONSTRAINT");
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushAddForeignKey(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ForeignKeyInfo)
     */
    public void pushAddForeignKey(SqlCommandQueue queue, TableInfo ti,
            ForeignKeyInfo fki) throws SQLException {
        queue.pushCommand(
                DDLHelper.buildAddForeignKey(ti.getName(), fki, false),
                buildDropForeignKey(ti, fki));
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropForeignKey(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ForeignKeyInfo)
     */
    public void pushDropForeignKey(SqlCommandQueue queue, TableInfo ti,
            ForeignKeyInfo fki) throws SQLException {
        queue.pushCommand(
                buildDropForeignKey(ti, fki),
                DDLHelper.buildAddForeignKey(ti.getName(), fki, false));
    }

    private static String buildAddIndex(TableInfo ti, IndexInfo indexInfo) throws SQLException
    {
        // if SQL Server < 2008:
        //return DDLHelper.buildAddIndex(ti,indexInfo,false);
        // if SQL Server >= 2008: https://stackoverflow.com/questions/767657/how-do-i-create-a-unique-constraint-that-also-allows-nulls

        //https://docs.microsoft.com/en-us/sql/t-sql/statements/create-index-transact-sql?view=sql-server-ver15
        String baseAddIndex = DDLHelper.buildAddIndex(ti,indexInfo,true,true);
        StringBuilder sb = new StringBuilder();
        sb.append(baseAddIndex);
        boolean first = !baseAddIndex.contains(" WHERE "); // FIXME filterCondition is not set in baseAddIndex: indexInfo.getFilterCondition() == null || indexInfo.getFilterCondition().isEmpty();
        for(String columnName : indexInfo.getColumns()) {
            ColumnInfo column = ti.getColumnInfo(columnName);
            if (column.isNullable()) {
                if (first) {
                    first = false;
                    sb.append(" WHERE ");
                } else {
                    sb.append(" AND ");
                }
                sb.append(columnName);
                sb.append(" IS NOT NULL ");
            }
        }
        return sb.toString();
    }

    private static String buildDropIndex(TableInfo ti, IndexInfo indexInfo) throws SQLException
    {
        return DDLHelper.buildDropIndex(ti.getName(),indexInfo.getName(),true);
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushAddIndex(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.IndexInfo)
     */
    public void pushAddIndex(SqlCommandQueue queue, TableInfo ti,
            IndexInfo indexInfo) throws SQLException {
        queue.pushCommand(
                buildAddIndex(ti,indexInfo),
                buildDropIndex(ti,indexInfo));
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropIndex(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.IndexInfo)
     */
    public void pushDropIndex(SqlCommandQueue queue, TableInfo ti,
            IndexInfo indexInfo) throws SQLException {
        queue.pushCommand(
                buildDropIndex(ti,indexInfo),
                buildAddIndex(ti,indexInfo));
    }

    private String buildAddColumn(TableInfo ti, ColumnInfo ci)
    {
        return DDLHelper.buildAddColumn(ti.getName(),ci,this,addColumnCommand, false);
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushAddColumn(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void pushAddColumn(SqlCommandQueue queue, TableInfo ti, ColumnInfo ci)
            throws SQLException {

        if (ci.getDefaultValue() == null)
        {
            queue.pushCommand(
                    this.buildAddColumn(ti, ci),
                    DDLHelper.buildDropColumn(ti.getName(),ci.getName()));
        }
        else
        {
            // first, create the column with no default value.
            ColumnInfo ciNull = ColumnHelper.adaptNullability(ci,true);
            ColumnInfo ciNoDefNull = ColumnHelper.adaptDefault(ciNull,null);

            // first, create the column with no default value and with nullable true.
            queue.pushCommand(
                    this.buildAddColumn(ti, ciNoDefNull),
                    DDLHelper.buildDropColumn(ti.getName(),ci.getName()));

            // add default value.
            queue.pushCommand(new ModifyDefaultValueCommand(ti.getName(),ciNull,false,this));

            // Fill in the default value.
            StringBuffer sql = new StringBuffer();
            sql.append("UPDATE ");
            sql.append(ti.getName());
            sql.append(" SET ");
            sql.append(ci.getName());
            sql.append(" = ");
            this.appendDefaultValue(sql,ci,true);

            StringBuffer rsql = new StringBuffer();
            rsql.append("UPDATE ");
            rsql.append(ti.getName());
            rsql.append(" SET ");
            rsql.append(ci.getName());
            rsql.append(" = NULL");

            queue.pushCommand(sql.toString(),rsql.toString());

            // finally set nullability to false.
            if (!ci.isNullable())
                queue.pushCommand(
                        buildAlterColumn(ti.getName(),ci),
                        buildAlterColumn(ti.getName(),ciNull));
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropColumn(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo, boolean)
     */
    public void pushDropColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo ci, boolean force) throws SQLException {

        if (ci.getDefaultValue() != null)
            queue.pushCommand(new ModifyDefaultValueCommand(ti.getName(),ci,true,this));

        ColumnInfo ciNoDef = ColumnHelper.adaptDefault(ci,null);

        if (force)
            queue.pushCommand(
                    DDLHelper.buildDropColumn(ti.getName(),ci.getName()),
                    this.buildAddColumn(ti,ciNoDef));
        else
        {
            if (!ci.isNullable())
            {
                // assure, that the column is made nullable before backing up the data in
                // DropColumnCommand Otherwise, the rollback will fail, because
                // creating a not-nullable column on an non-empty table fails.
                ColumnInfo ciNull = ColumnHelper.adaptNullability(ciNoDef,true);
                pushModifyColumnType(queue,ti,ciNoDef,ciNull);
                queue.pushCommand(new DropColumnCommand(schemaEngine, ti,ciNull,this,null,addColumnCommand, false));
            }
            else
                queue.pushCommand(new DropColumnCommand(schemaEngine, ti,ciNoDef,this,null,addColumnCommand, false));
        }
    }

    private static class ModifyDefaultValueCommand implements SqlCommand
    {
        private String table;
        private String column;
        private String performSql;
        private String rollbackSql;

        private void dropDefaultValue(Connection connection) throws SQLException
        {
            if (this.table == null || this.column == null)
                throw new SQLException("This statement has already been committed.");

            Statement statement = connection.createStatement();

            StringBuffer sql = new StringBuffer();
            sql.append("SELECT dl.name FROM sys.tables AS tl,sys.columns AS cl,sys.default_constraints AS dl WHERE tl.name = '");
            sql.append(this.table);
            sql.append("' AND cl.object_id = tl.object_id AND cl.name = '");
            sql.append(this.column);
            sql.append("' AND dl.parent_object_id = tl.object_id AND dl.parent_column_id = cl.column_id");

            ResultSet rs = statement.executeQuery(sql.toString());

            if (!rs.next())
                throw new SQLException("Default value constraint for column ["+this.column+
                        "] on table ["+this.table+"] could not be found.");

            sql.delete(0,sql.length());

            sql.append("ALTER TABLE ");
            sql.append(this.table);
            sql.append(" DROP CONSTRAINT ");
            sql.append(rs.getString(1));

            rs.close();

            if (log.isDebugEnabled())
                log.debug("Executing drop default value statement ["+sql+"].");

            statement.executeUpdate(sql.toString());
            statement.close();
        }

        public ModifyDefaultValueCommand(String table, ColumnInfo column,
                boolean drop, MSSQLServerDialect dialect) {
            super();
            this.table = table;
            this.column = column.getName();

            StringBuffer sql = new StringBuffer();
            sql.append("ALTER TABLE ");
            sql.append(table);
            sql.append(" ADD CONSTRAINT DF__");
            sql.append(table);
            sql.append("__");
            sql.append(column.getName());
            sql.append("__");
            sql.append(DDLHelper.buildHexSuffix());
            dialect.appendDefaultValue(sql,column, false);
            sql.append(" FOR ");
            sql.append(column.getName());
            sql.append(" WITH VALUES");

            if (drop)
            {
                this.performSql = null;
                this.rollbackSql = sql.toString();
            }
            else
            {
                this.performSql = sql.toString();
                this.rollbackSql = null;
            }
        }

        public void perform(Connection connection) throws SQLException {

            if (this.performSql != null) {

                if (log.isDebugEnabled())
                    log.debug("Executing add default value statement ["+this.performSql+"].");
                SQLHelper.executeUpdate(connection,this.performSql);
            }
            else
                this.dropDefaultValue(connection);
        }
        public void rollback(Connection connection) throws SQLException {

            if (this.rollbackSql != null) {

                if (log.isDebugEnabled())
                    log.debug("Executing add default value statement ["+this.rollbackSql+"] during rollback.");

                SQLHelper.executeUpdate(connection,this.rollbackSql);
            }
            else
                this.dropDefaultValue(connection);
        }

        public void cleanupOnCommit(Connection connection) throws SQLException {
            this.column = null;
            this.table = null;
            this.performSql = null;
            this.rollbackSql = null;
        }

        public String getTempTableName() {
            return null;
        }

        public boolean isTempTableCreated() {
            return false;
        }
    }

    private static String buildAlterColumn(String tableName, ColumnInfo columnInfo) throws SQLException
    {
        StringBuffer sql = new StringBuffer();

        sql.append("ALTER TABLE ");
        sql.append(tableName);
        sql.append(' ');
        sql.append("ALTER COLUMN ");
        sql.append(columnInfo.getName());
        sql.append(' ');

        if (appendColumnType(sql,columnInfo) == null)
            throw new SQLException("Cannot modify column ["+columnInfo.getName()+"] to type ["+TypesHelper.typeToString(columnInfo.getType())+"]: Type is not supported.");

        if (columnInfo.isNullable())
            sql.append(" NULL");
        else
            sql.append(" NOT NULL");

        return sql.toString();
    }

    private static void pushModifyColumnType(SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo) throws SQLException
    {
        // assume, that the default value and the column name is equals.

        // check for NOOP.
        if (Util.equalsNullAware(oldColumnInfo.getPrecision(),newColumnInfo.getPrecision()) &&
                Util.equalsNullAware(oldColumnInfo.getScale(),newColumnInfo.getScale())&&
                oldColumnInfo.getType() == newColumnInfo.getType() &&
                oldColumnInfo.isNullable() == newColumnInfo.isNullable() )
            return;

        queue.pushCommand(
                buildAlterColumn(ti.getName(),newColumnInfo),
                buildAlterColumn(ti.getName(),oldColumnInfo));
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushChangeColumn(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void pushChangeColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo)
            throws SQLException {

        // first, append a new column with the new type and NULL values allowed.
        ColumnInfo tmpInfo = ColumnHelper.adaptNullability(newColumnInfo,true);

        this.pushAddColumn(queue,ti,tmpInfo);

        // Then copy the values over from the old column
        StringBuffer sql = new StringBuffer();
        sql.append("UPDATE ");
        sql.append(ti.getName());
        sql.append(" SET ");
        sql.append(newColumnInfo.getName());
        sql.append(" = ");
        sql.append(oldColumnInfo.getName());

        StringBuffer rsql = new StringBuffer();
        rsql.append("UPDATE ");
        rsql.append(ti.getName());
        rsql.append(" SET ");
        rsql.append(newColumnInfo.getName());
        rsql.append(" = NULL");

        queue.pushCommand(sql.toString(),rsql.toString());

        sql.delete(0,sql.length());
        rsql.delete(0,rsql.length());

        // change the type of the new column to the destination type,
        // because we cannot set not null flag before copying the actual data.
        if (!newColumnInfo.isNullable())
        {
            pushModifyColumnType(queue,ti,tmpInfo,newColumnInfo);
        }

        // drop the column.
        this.pushDropColumn(schemaEngine, queue,ti,oldColumnInfo,true);
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushModifyColumn(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void pushModifyColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo)
            throws SQLException {

        if (oldColumnInfo.getDefaultValue() != null)
            queue.pushCommand(new ModifyDefaultValueCommand(ti.getName(),oldColumnInfo,true,this));

        pushModifyColumnType(queue,ti,oldColumnInfo,newColumnInfo);

        if (newColumnInfo.getDefaultValue() != null)
            queue.pushCommand(new ModifyDefaultValueCommand(ti.getName(),newColumnInfo,false,this));

    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#constructJDBCURL(java.lang.String, java.lang.Integer, java.lang.String, java.util.Properties)
     */
    public String constructJDBCURL(String hostname, Integer port,
            String databaseName, Properties properties) {
        // jdbc:sqlserver://localhost:1433;databaseName=foodb;user=Administrator;password=secret;
        StringBuffer url=new StringBuffer("jdbc:sqlserver://");
        if (hostname != null && hostname.length()>0)
            url.append(hostname);
        else
            url.append("localhost");
        url.append(':');
        if (port!=null && port>0)
            url.append(port);
        else
            url.append("1433");
        url.append(';');
        if (databaseName!=null && databaseName.length()>0) {

            url.append("databaseName=");
            url.append(databaseName);
            url.append(';');
        }
        if (properties!=null && properties.size()>0) {
            Enumeration<?> propertyNames=properties.propertyNames();
            while (propertyNames.hasMoreElements()) {
                String propertyName=propertyNames.toString();
                String propertyValue=properties.getProperty(propertyName);
                url.append(propertyName);
                url.append('=');
                url.append(propertyValue);
                url.append(';');
            }
        }
        return url.toString();
    }
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#getMappedSqlType(java.lang.String)
     */
    public int getMappedSqlType(String dialectDataType) {
        //TODO: Add potential other types as well.
        dialectDataType = dialectDataType.toUpperCase().replace(" ", "_").trim();
        switch (dialectDataType) {
            case "DATETIMEOFFSET": return Types.TIMESTAMP_WITH_TIMEZONE;
            //The closest thing to what would constitute a timestamp type in MSSQL is DATETIME
            case "DATETIME": return Types.TIMESTAMP;
            default: throw new DataTypeNotSupportedException(dialectDataType);
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#fetchAdditionalColumnInfo(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void fetchAdditionalColumnInfo(ISchemaEngine schemaEngine, TableInfo ti,
            ColumnInfo ci) {
    }
}
