/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 15.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema.impl;

import java.sql.Connection;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;

import org.clazzes.jdbc2xml.helper.SQLHelper;
import org.clazzes.jdbc2xml.schema.ColumnInfo;
import org.clazzes.jdbc2xml.schema.Dialect;
import org.clazzes.jdbc2xml.schema.ISchemaEngine;
import org.clazzes.jdbc2xml.schema.TableInfo;
import org.clazzes.jdbc2xml.sql.SqlCommand;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class implements a rollback-enabled variant of a drop table
 * statement.
 */
public class DropTableCommand implements SqlCommand {

    private static final Logger log = LoggerFactory.getLogger(DropTableCommand.class);
    
    private final String tableName;
    private final String tempTableName;
    private final String createOriginalTable;
    private final String createTempTable;
    private boolean tempTableCreated;
    private SqlCommand renameTableCommand;
    private boolean mssql;
    private boolean autoIncrement;
    
    /**
     * Construct a drop table command for RDMS engines, which do not support
     * renaming of tables.
     * 
     * @param schemaEngine The SchemaEngine.
     * @param ti The description of the table to be dropped.
     * @param dialect The dialect used to build column specs.
     * @param suffix The RDMBS-specific create table suffix.
     * @throws SQLException Upon unsupported foreign key options.
     *                             
     * @see DDLHelper#buildCreateTable(TableInfo, Dialect, String)
     */
    public DropTableCommand(ISchemaEngine schemaEngine, TableInfo ti,
            Dialect dialect,
            String suffix) throws SQLException
    {
        this.mssql = (dialect instanceof MSSQLServerDialect);
        checkAutoIncrement(ti);
        this.tableName = ti.getName();
        this.tempTableName = NameHelper.buildTempTableName(schemaEngine, ti.getName(), null, DDLHelper.buildHexSuffix());
        
        TableInfo tempTableInfo = new TableInfo(this.tempTableName);  
        
        List<ColumnInfo> tempColumns = new ArrayList<ColumnInfo>(ti.getColumns().size());
        tempColumns.addAll(ti.getColumns());
        tempTableInfo.setColumns(tempColumns);
            
        this.createTempTable = DDLHelper.buildCreateTable(tempTableInfo, dialect, suffix);
        this.createOriginalTable = DDLHelper.buildCreateTable(ti, dialect, suffix);
    }
    
    private void checkAutoIncrement(TableInfo ti)
    {
        for (ColumnInfo ci : ti.getColumns())
        {
            if (ci.isAutoIncrement())
            {
                this.autoIncrement = true;
                return;
            }
        }
    }
    
    /**
     * Construct a drop table command for RDMS engines, which support
     * renaming of tables.
     * 
     * @param schemaEngine The SchemaEngine.
     * @param ti The description of the table to be dropped.
     * @param renameTableCommand A dialect specific rename table command, if the
     *                           rename table operation does conform to ISO SQL.
     *                           If set to <code>null</code>, the ISO SQL command
     *                           is used. For details see {@link DDLHelper#buildRenameTable(String, String, String)}. 
     */
    public DropTableCommand(ISchemaEngine schemaEngine, TableInfo ti, String renameTableCommand, Dialect dialect)
    {
        this.mssql = (dialect instanceof MSSQLServerDialect);
        checkAutoIncrement(ti);
        this.tableName = ti.getName();
        this.tempTableName = NameHelper.buildTempTableName(schemaEngine, ti.getName(), null, DDLHelper.buildHexSuffix());
        this.createTempTable = null;
        this.createOriginalTable = null;
        this.renameTableCommand = DDLHelper.buildRenameTable(renameTableCommand,this.tableName,this.tempTableName);
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.sql.SqlCommand#cleanupOnCommit(java.sql.Connection)
     */
    public void cleanupOnCommit(Connection connection) throws SQLException {
     
        if (this.tempTableCreated)
        {
            if (log.isDebugEnabled())
                log.debug("Dropping backup table ["+this.tempTableName+"] after commit.");
                
            SQLHelper.executeUpdate(connection,DDLHelper.buildDropTable(this.tempTableName));
            this.tempTableCreated = false;
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.sql.SqlCommand#perfom(java.sql.Connection)
     */
    public void perform(Connection connection) throws SQLException {
        
        if (this.createTempTable != null)
        {
            if (log.isDebugEnabled())
                log.debug("Creating backup table ["+this.tempTableName+"] for future rollback operations.");

            SQLHelper.executeUpdate(connection,this.createTempTable);
            this.tempTableCreated = true;
            
            try {
                if (log.isDebugEnabled())
                    log.debug("Populating backup table ["+this.tempTableName+"] with data from table ["+
                            this.tableName+"] for future rollback operations.");

                String sql = null;
                if (this.autoIncrement && this.mssql)
                {
                    StringBuilder colsSql = new StringBuilder();
                    Statement st = connection.createStatement();
                    if (st.execute("SELECT * from " + this.tableName))
                    {
                        ResultSetMetaData rs = st.getResultSet().getMetaData();
                        int n = rs.getColumnCount();
                        for (int i=1; i<=n; i++)
                        {
                            if (i > 1)
                                colsSql.append(",");
                            colsSql.append(rs.getColumnName(i));
                        }
                    }
                    sql = "set identity_insert " + this.tempTableName + " on" +
                        " INSERT INTO "+this.tempTableName+" (" + colsSql.toString() + ") SELECT " + colsSql.toString() + " FROM " + this.tableName +
                        " set identity_insert " + this.tempTableName + " off"       ;
                }
                else
                    sql = "INSERT INTO "+this.tempTableName+" SELECT * FROM " + this.tableName;
                
                SQLHelper.executeUpdate(connection, sql);
                
                if (log.isDebugEnabled())
                    log.debug("Dropping table ["+this.tableName+"].");

                SQLHelper.executeUpdate(connection,DDLHelper.buildDropTable(this.tableName));
                
            } catch (SQLException e) {
                
                log.error("Error during drop of table ["+this.tableName+"], dropping backup table ["+this.tempTableName+"].",e);
                SQLHelper.executeUpdate(connection,DDLHelper.buildDropTable(this.tempTableName));
                this.tempTableCreated = false;
                throw e;
            }
        }
        else
        {
            if (log.isDebugEnabled())
                log.debug("Renaming table ["+this.tableName+"] to backup table ["+this.tempTableName+"] for future rollback operations.");
            
            this.renameTableCommand.perform(connection);
            this.tempTableCreated = true;
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.sql.SqlCommand#rollback(java.sql.Connection)
     */
    public void rollback(Connection connection) throws SQLException {
        
        if (this.createOriginalTable != null)
        {
            if (log.isDebugEnabled())
                log.debug("Restoring structure of table ["+this.tableName+"] during rollback.");

            SQLHelper.executeUpdate(connection,this.createOriginalTable);
            
            // There's no need to restore the foreign keys here, because the foreign keys will
            // be dropped and restored by the upstream layer in
            // SchemaEngineImpl.dropTable().
            
            if (log.isDebugEnabled())
                log.debug("Populating table ["+this.tableName+"] with data from backup table ["+
                        this.tempTableName+"] during rollback.");
            
            String sql = null;
            if (this.autoIncrement && this.mssql)
            {
                StringBuilder colsSql = new StringBuilder();
                Statement st = connection.createStatement();
                if (st.execute("SELECT * from " + this.tempTableName))
                {
                    ResultSetMetaData rs = st.getResultSet().getMetaData();
                    int n = rs.getColumnCount();
                    for (int i=1; i<=n; i++)
                    {
                        if (i > 1)
                            colsSql.append(",");
                        colsSql.append(rs.getColumnName(i));
                    }
                }
                sql = "set identity_insert " + this.tableName + " on" +
                    " INSERT INTO "+this.tableName+" (" + colsSql.toString() + ") SELECT " + colsSql.toString() + " FROM " + this.tempTableName +
                    " set identity_insert " + this.tableName + " off"       ;
            }
            else
                sql = "INSERT INTO "+this.tableName+" SELECT * FROM " + this.tempTableName;
            SQLHelper.executeUpdate(connection, sql);
            
            if (log.isDebugEnabled())
                log.debug("Dropping backup table ["+this.tempTableName+"] during rollback.");

            SQLHelper.executeUpdate(connection,DDLHelper.buildDropTable(this.tempTableName));
        }
        else
        {
            if (log.isDebugEnabled())
                log.debug("Renaming backup table ["+this.tempTableName+"] to ["+this.tableName+"] during rollback.");
            
            this.renameTableCommand.rollback(connection);
        }
        
        this.tempTableCreated = false;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.sql.SqlCommand#isTempTableCreated()
     */
    public boolean isTempTableCreated() {
        return this.tempTableCreated;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.sql.SqlCommand#getTempTableName()
     */
    public String getTempTableName() {
        return this.tempTableName;
    }

}
