/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 15.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema.impl;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;

import org.clazzes.jdbc2xml.helper.JAVAHelper;
import org.clazzes.jdbc2xml.helper.SQLHelper;
import org.clazzes.jdbc2xml.schema.ColumnInfo;
import org.clazzes.jdbc2xml.schema.Dialect;
import org.clazzes.jdbc2xml.schema.ISchemaEngine;
import org.clazzes.jdbc2xml.schema.TableInfo;
import org.clazzes.jdbc2xml.sql.SqlCommand;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class implements a rollback-enabled variant of a drop column
 * statement.
 */
public class DropColumnCommand implements SqlCommand {

    private static final Logger log = LoggerFactory.getLogger(DropColumnCommand.class);
    
    private final String tableName;
    private final String columnName;
    private final String tempTableName;
    private final String createTempTable;
    private boolean tempTableCreated;
    private final String restoreOrignialColumn;
    private final String saveOriginalData;
    private final String selectBackupData;
    private final String restoreOriginalData;
    
    /**
     * Construct a drop column command with rollback support.
     * 
     * @param ti The description of the table.
     * @param ci The description of the column to be created.
     * @param dialect The dialect used to build column specs for the reverse operation.
     * @param suffix The RDMBS-specific create table suffix.
     * @throws SQLException Upon unsupported foreign key options.
     *                             
     * @see DDLHelper#buildAddColumn(String, ColumnInfo, Dialect, String, boolean)
     */
    public DropColumnCommand(ISchemaEngine schemaEngine, TableInfo ti,ColumnInfo ci,
            Dialect dialect,
            String suffix,
            String addColumnCommand, boolean addStatementInBrackets) throws SQLException
    {
        if (ti.getPrimaryKey() == null)
            throw new SQLException ("Dropping of columns from table without primary key is not supported.");
        
        this.tableName = ti.getName();
        this.columnName = ci.getName();
        this.tempTableName = NameHelper.buildTempTableName(schemaEngine, ti.getName(), ci.getName(), DDLHelper.buildHexSuffix());
        
        TableInfo tempTableInfo = new TableInfo(this.tempTableName);  
        List<ColumnInfo> tempColumns = new ArrayList<ColumnInfo>(ti.getColumns().size());
        
        for (String pkColumn : ti.getPrimaryKey().getColumns())
        {
            if (this.columnName.equals(pkColumn))
                throw new SQLException ("Dropping of primary key column ["+pkColumn+"] is not supported.");
            
            tempColumns.add(ti.getColumnInfo(pkColumn));
        }
        tempColumns.add(ci);
        
        tempTableInfo.setColumns(tempColumns);
            
        this.createTempTable = DDLHelper.buildCreateTable(tempTableInfo, dialect, suffix);
        this.restoreOrignialColumn = DDLHelper.buildAddColumn(this.tableName,ci,dialect,addColumnCommand, addStatementInBrackets);
        
        this.saveOriginalData =
            "INSERT INTO "+this.tempTableName+ "("+
            JAVAHelper.joinStrings(ti.getPrimaryKey().getColumns())+"," + this.columnName+
            ") SELECT "+
            JAVAHelper.joinStrings(ti.getPrimaryKey().getColumns())+"," + this.columnName+
            " FROM " + this.tableName;
        
        this.selectBackupData =
            "SELECT "+
            this.columnName+"," +JAVAHelper.joinStrings(ti.getPrimaryKey().getColumns())+
            " FROM " + this.tempTableName;
        
        StringBuffer sql = new StringBuffer();
        sql.append("UPDATE ");
        sql.append(this.tableName);
        sql.append(" SET ");
        sql.append(this.columnName);
        sql.append(" = ? WHERE ");
        
        boolean first = true;
        
        for (String pkColumn : ti.getPrimaryKey().getColumns())
        {
            if (first)
                first = false;
            else
                sql.append(" AND ");
           
            sql.append(pkColumn);
            sql.append("=?");
        }
   
        this.restoreOriginalData = sql.toString();
   }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.sql.SqlCommand#cleanupOnCommit(java.sql.Connection)
     */
    public void cleanupOnCommit(Connection connection) throws SQLException {
     
        if (this.tempTableCreated)
        {
            if (log.isDebugEnabled())
                log.debug("Dropping backup table ["+this.tempTableName+"] after commit.");
                
            SQLHelper.executeUpdate(connection,DDLHelper.buildDropTable(this.tempTableName));
            this.tempTableCreated = false;
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.sql.SqlCommand#perfom(java.sql.Connection)
     */
    public void perform(Connection connection) throws SQLException {
        
        if (log.isDebugEnabled())
            log.debug("Creating backup table ["+this.tempTableName+"] for future rollback operations.");

        SQLHelper.executeUpdate(connection,this.createTempTable);
        this.tempTableCreated = true;
            
        try {
            if (log.isDebugEnabled())
                log.debug("Populating backup table ["+this.tempTableName+"] with data from table ["+
                        this.tableName+"] for future rollback operations.");

            SQLHelper.executeUpdate(connection,this.saveOriginalData);
                
            if (log.isDebugEnabled())
                log.debug("Dropping column ["+this.columnName+"] of table ["+this.tableName+"].");
            
            SQLHelper.executeUpdate(connection,DDLHelper.buildDropColumn(this.tableName,this.columnName));
                
        } catch (SQLException e) {
            
            log.error("Error during drop of column ["+this.columnName+"] of table ["+this.tableName+"], dropping backup table ["+this.tempTableName+"].",e);
            SQLHelper.executeUpdate(connection,DDLHelper.buildDropTable(this.tempTableName));
            this.tempTableCreated = false;
            throw e;
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.sql.SqlCommand#rollback(java.sql.Connection)
     */
    public void rollback(Connection connection) throws SQLException {
        
        if (log.isDebugEnabled())
            log.debug("Restoring column ["+this.columnName+"] of table ["+this.tableName+"] during rollback.");

        SQLHelper.executeUpdate(connection,this.restoreOrignialColumn);
        
        if (log.isDebugEnabled())
        {
            log.debug("Populating table ["+this.tableName+"] with data from backup table ["+
                    this.tempTableName+"] during rollback.");
            log.debug("select sql ["+this.selectBackupData+"].");
            log.debug("populate sql ["+this.restoreOriginalData+"].");
        }
        
        Statement statement = connection.createStatement(ResultSet.TYPE_FORWARD_ONLY,ResultSet.CONCUR_READ_ONLY);
        
        ResultSet rs = statement.executeQuery(this.selectBackupData);
        
        PreparedStatement update = connection.prepareStatement(this.restoreOriginalData);
        
        while (rs.next())
        {
            for (int i=1; i<=rs.getMetaData().getColumnCount();++i)
            {
                SQLHelper.copyResultField(update,i,rs,i);   
            }
            
            update.execute();
        }
        update.close();
        statement.close();
        
        if (log.isDebugEnabled())
            log.debug("Dropping backup table ["+this.tempTableName+"] during rollback.");

        SQLHelper.executeUpdate(connection,DDLHelper.buildDropTable(this.tempTableName));

        this.tempTableCreated = false;
    }

    /**
     * @return the tempTableName
     */
    public String getTempTableName() {
        return this.tempTableName;
    }

    /**
     * @return the tempTableCreated
     */
    public boolean isTempTableCreated() {
        return this.tempTableCreated;
    }

}
