/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 27.11.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema.impl;

import java.sql.SQLException;
import java.sql.Types;
import java.util.Enumeration;
import java.util.Properties;

import org.clazzes.jdbc2xml.helper.SQLHelper;
import org.clazzes.jdbc2xml.helper.TypesHelper;
import org.clazzes.jdbc2xml.schema.ColumnInfo;
import org.clazzes.jdbc2xml.schema.DataTypeNotSupportedException;
import org.clazzes.jdbc2xml.schema.ForeignKeyInfo;
import org.clazzes.jdbc2xml.schema.ISchemaEngine;
import org.clazzes.jdbc2xml.schema.IndexInfo;
import org.clazzes.jdbc2xml.schema.TableInfo;
import org.clazzes.jdbc2xml.sql.SqlCommandQueue;
import org.clazzes.util.lang.Util;

/**
 * This class implements Dialect for Derby 10.x
 * 
 * @author lech
 */
public class DerbyDialect extends AbstrDialectSupport {
    
    public static final String defaultDriverName="org.apache.derby.jdbc.EmbeddedDriver";
    
    private static final String addColumnCommand="ADD COLUMN";
    private static final String BINARY_SUFFIX = " FOR BIT DATA"; 
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#getID()
     */
    public String getID() {
        
        return "Derby_10";
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushCreateTable(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo)
     */
    public void pushCreateTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti) {
     
        queue.pushCommand(
                DDLHelper.buildCreateTable(ti,this,null),
                DDLHelper.buildDropTable(ti.getName())
        );
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushRenameTable(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, java.lang.String)
     */
    public void pushRenameTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            String newTableName) throws SQLException {
        
        queue.pushCommand(DDLHelper.buildRenameTable(null,ti.getName(),newTableName));
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropTable(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, boolean)
     */
    public void pushDropTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti, boolean force) {
        
        if (force)
            queue.pushCommand(
                    DDLHelper.buildDropTable(ti.getName()),
                    null
            );
        else
            queue.pushCommand(new DropTableCommand(schemaEngine, ti,null, this));
    }

    private static StringBuffer appendColumnType(StringBuffer columnSpec, ColumnInfo columnInfo)
    {
        switch (columnInfo.getType()) {
        case Types.BIGINT:
            columnSpec.append("BIGINT");
         break;
        case Types.BINARY:
        case Types.BIT:
            SQLHelper.appendTypePrec(columnSpec,"CHAR",columnInfo.getPrecision());
            columnSpec.append(BINARY_SUFFIX);
        break;
        case Types.BLOB:
            SQLHelper.appendTypePrec(columnSpec,"BLOB",columnInfo.getPrecision());
        break;
        case Types.BOOLEAN:
            columnSpec.append("SMALLINT");
        break;
        case Types.CHAR:
        case Types.NCHAR:
            SQLHelper.appendTypePrec(columnSpec,"CHAR",columnInfo.getPrecision());
        break;
        case Types.CLOB:
            SQLHelper.appendTypePrec(columnSpec,"CLOB",columnInfo.getPrecision());
        break;
        case Types.DATE:
            columnSpec.append("DATE");
        break;
        case Types.DECIMAL:
            SQLHelper.appendTypePrecScale(columnSpec,"DECIMAL",
                    columnInfo.getPrecision(),columnInfo.getScale());
        break;
        case Types.DOUBLE: 
        case Types.FLOAT:
            columnSpec.append("DOUBLE");
        break;
        case Types.INTEGER:
            columnSpec.append("INTEGER");
        break;
        case Types.LONGVARBINARY:
            if (columnInfo.getPrecision() == null || columnInfo.getPrecision().intValue() <= 32700)
            {
                columnSpec.append("LONG VARCHAR");
                columnSpec.append(BINARY_SUFFIX);
            }
            else
            {
                SQLHelper.appendTypePrec(columnSpec,"BLOB",columnInfo.getPrecision());
            }
        break;
        case Types.LONGVARCHAR:
        case Types.LONGNVARCHAR:
            if (columnInfo.getPrecision() == null || columnInfo.getPrecision().intValue() <= 32700)
            {
                columnSpec.append("LONG VARCHAR");
            }
            else
            {
                SQLHelper.appendTypePrec(columnSpec,"CLOB",columnInfo.getPrecision());
            }
        break;
        case Types.NUMERIC:
            SQLHelper.appendTypePrecScale(columnSpec,"NUMERIC",
                    columnInfo.getPrecision(),columnInfo.getScale());
        break;
        case Types.REAL:
            columnSpec.append("REAL");
        break;
        case Types.SMALLINT:
            columnSpec.append("SMALLINT");
        break;
        case Types.TIME:
            columnSpec.append("TIME");
        break;
        case Types.TIMESTAMP:
        case Types.TIMESTAMP_WITH_TIMEZONE:
            columnSpec.append("TIMESTAMP");
        break;
        case Types.TINYINT:
            columnSpec.append("SMALLINT");
        break;
        case Types.VARBINARY:
            SQLHelper.appendTypePrec(columnSpec,"VARCHAR",columnInfo.getPrecision());
            columnSpec.append(BINARY_SUFFIX);
        break;
        case Types.NVARCHAR:
        case Types.VARCHAR:
            // the only character set for Derby's character strings is Unicode
            SQLHelper.appendTypePrec(columnSpec,"VARCHAR",columnInfo.getPrecision());
        break;
        
        default:
            throw new DataTypeNotSupportedException(columnInfo.getType());
        }
        return columnSpec;
    }
    
    private void appendDefaultValue(StringBuffer columnSpec, ColumnInfo columnInfo)
    {
        if (columnInfo.getDefaultValue() != null)
        {
            columnSpec.append(" DEFAULT ");
            
            if (TypesHelper.isNumeric(columnInfo.getType()))
                columnSpec.append(columnInfo.getDefaultValue());
            else
            {
                columnSpec.append('\'');
                this.quoteString(columnSpec,columnInfo.getDefaultValue());
                columnSpec.append('\'');
            }
        }
    }
    /*
     * (non-Javadoc)
     * 
     * @see org.clazzes.jdbc2xml.schema.Dialect#createColumnSpec(org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public String createColumnSpec(ColumnInfo columnInfo) {

        StringBuffer columnSpec = new StringBuffer();
        
        columnSpec.append(columnInfo.getName());
        columnSpec.append(" ");
        
        if (appendColumnType(columnSpec, columnInfo) == null)
            return null;
        
        if (!columnInfo.isNullable())
            columnSpec.append(" NOT NULL");
        
        this.appendDefaultValue(columnSpec, columnInfo);
        
        if (columnInfo.isAutoIncrement())
            columnSpec.append(" GENERATED BY DEFAULT AS IDENTITY");
        
        return columnSpec.toString();
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#defaultDriverName()
     */
    public String defaultDriverName() {
        return defaultDriverName;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#quoteString(java.lang.StringBuffer, java.lang.String)
     */
    public void quoteString(StringBuffer sb, String s) {
        SQLHelper.quoteISOSqlString(sb,s);
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#normalizeDefaultValue(int, java.lang.String)
     */
    public String normalizeDefaultValue(int type, String s) {
        if (s==null || "GENERATED_BY_DEFAULT".equals(s))
            return null;
        
        if (!TypesHelper.isNumeric(type))
        {
            if (s.startsWith("'") && s.endsWith("'")) {
                if (s.length() > 2) {
                    // unquote ISO SQL string '' -> '
                    return s.substring(1,s.length()-1).replace("''","'");
                }
            }
        }
        return s;
    }

    private static String buildDropForeignKey(TableInfo ti,
            ForeignKeyInfo fki)
    {
        return DDLHelper.buildDropForeignKey(ti.getName(),fki.getName(),"DROP FOREIGN KEY");
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushAddForeignKey(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ForeignKeyInfo)
     */
    public void pushAddForeignKey(SqlCommandQueue queue, TableInfo ti,
            ForeignKeyInfo fki) throws SQLException {
        queue.pushCommand(
                DDLHelper.buildAddForeignKey(ti.getName(), fki, false),
                buildDropForeignKey(ti, fki));
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropForeignKey(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ForeignKeyInfo)
     */
    public void pushDropForeignKey(SqlCommandQueue queue, TableInfo ti,
            ForeignKeyInfo fki) throws SQLException {
        queue.pushCommand(
                buildDropForeignKey(ti, fki),
                DDLHelper.buildAddForeignKey(ti.getName(), fki, false));
    }

    private static String buildAddIndex(TableInfo ti, IndexInfo indexInfo) throws SQLException
    {
        //https://db.apache.org/derby/docs/10.8/ref/rrefsqlj20937.html
        return DDLHelper.buildAddIndex(ti,indexInfo,false,false);
    }
    
    private static String buildDropIndex(TableInfo ti, IndexInfo indexInfo) throws SQLException
    {
        return DDLHelper.buildDropIndex(ti.getName(),indexInfo.getName(),false);
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushAddIndex(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.IndexInfo)
     */
    public void pushAddIndex(SqlCommandQueue queue, TableInfo ti,
            IndexInfo indexInfo) throws SQLException {
        queue.pushCommand(
                buildAddIndex(ti,indexInfo),
                buildDropIndex(ti,indexInfo));
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropIndex(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.IndexInfo)
     */
    public void pushDropIndex(SqlCommandQueue queue, TableInfo ti,
            IndexInfo indexInfo) throws SQLException {
        queue.pushCommand(
                buildDropIndex(ti,indexInfo),
                buildAddIndex(ti,indexInfo));
    }

    private String buildAddColumn(TableInfo ti, ColumnInfo ci)
    {
        return DDLHelper.buildAddColumn(ti.getName(),ci,this,addColumnCommand, false);
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushAddColumn(org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void pushAddColumn(SqlCommandQueue queue, TableInfo ti, ColumnInfo ci)
            throws SQLException {
        
        queue.pushCommand(
                this.buildAddColumn(ti, ci),
                DDLHelper.buildDropColumn(ti.getName(),ci.getName()));
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushDropColumn(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo, boolean)
     */
    public void pushDropColumn(ISchemaEngine schema, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo ci, boolean force) throws SQLException {
        
        if (force)
            queue.pushCommand(
                    DDLHelper.buildDropColumn(ti.getName(),ci.getName()),
                    this.buildAddColumn(ti, ci));
        else
        {
            if (!ci.isNullable())
            {
                // assure, that the column is made nullable before backing up the data in
                // DropColumnCommand Otherwise, the rollback will fail, because
                // creating a not-nullable column on an non-empty table fails.
                ColumnInfo ciNull = ColumnHelper.adaptNullability(ci,true);
                pushAdaptNullability(queue, ti, ci, ciNull);
                queue.pushCommand(new DropColumnCommand(schema, ti,ciNull,this,null,addColumnCommand, false));
            }
            else
                queue.pushCommand(new DropColumnCommand(schema, ti,ci,this,null,addColumnCommand, false));
        }
    }

    private static String buildAdaptNullability(String tableName, ColumnInfo columnInfo) 
    {
        StringBuffer sql = new StringBuffer();
        
        sql.append("ALTER TABLE ");
        sql.append(tableName);
        sql.append(' ');
        sql.append("ALTER COLUMN ");
        sql.append(columnInfo.getName());

        if (columnInfo.isNullable())
            sql.append(" NULL");
        else
            sql.append(" NOT NULL");

        return sql.toString();
    }
    
    private static void pushAdaptNullability(SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo) throws SQLException
    {
        // assume, that the default value, type and the column name is equals.
        
        // check for NOOP.
        if (oldColumnInfo.isNullable() == newColumnInfo.isNullable() )
            return;
        
        queue.pushCommand(
                buildAdaptNullability(ti.getName(),newColumnInfo),
                buildAdaptNullability(ti.getName(),oldColumnInfo) );
    }

    
    private static String buildAdaptDefaultValue(String tableName, ColumnInfo columnInfo) 
    {
        StringBuffer sql = new StringBuffer();
        
        sql.append("ALTER TABLE ");
        sql.append(tableName);
        sql.append(' ');
        sql.append("ALTER COLUMN ");
        sql.append(columnInfo.getName());

        if (columnInfo.isNullable())
            sql.append(" NULL");
        else
            sql.append(" NOT NULL");

        return sql.toString();
    }
    
    private static void pushAdaptDefaultValue(SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo)
    {
        // assume, that the nullability, type and the column name is equals.
        
        // check for NOOP.
        if (Util.equalsNullAware(oldColumnInfo.getDefaultValue(),newColumnInfo.getDefaultValue()))
            return;
        
        queue.pushCommand(
                buildAdaptDefaultValue(ti.getName(),newColumnInfo),
                buildAdaptDefaultValue(ti.getName(),oldColumnInfo) );
    }

    private void pushAdaptNullabilityAndDefaultValue(SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo) throws SQLException
    {
        ColumnInfo tmpInfo = ColumnHelper.adaptNullability(newColumnInfo,oldColumnInfo.isNullable());
        
        pushAdaptDefaultValue(queue,ti,oldColumnInfo,tmpInfo);
        pushAdaptNullability(queue,ti,tmpInfo,newColumnInfo); 
    }
    
    private void pushCreateCopyDrop(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo) throws SQLException
    {
        // first, append a new column with the new type and NULL values allowed.
        ColumnInfo tmpInfo = ColumnHelper.adaptNullability(newColumnInfo,true);
                
        this.pushAddColumn(queue,ti,tmpInfo);
                     
        // Then copy the values over from the old column
        StringBuffer sql = new StringBuffer();
        sql.append("UPDATE ");
        sql.append(ti.getName());
        sql.append(" SET ");
        sql.append(newColumnInfo.getName());
        sql.append(" = ");
        sql.append(oldColumnInfo.getName());
               
        StringBuffer rsql = new StringBuffer();
        rsql.append("UPDATE ");
        rsql.append(ti.getName());
        rsql.append(" SET ");
        rsql.append(newColumnInfo.getName());
        rsql.append(" = NULL");
        
        queue.pushCommand(sql.toString(),rsql.toString());
        
        sql.delete(0,sql.length());
        rsql.delete(0,rsql.length());
        
        // change the type of the new column to the destination type,
        // because we cannot set not null flag before copying the actual data.
        if (!newColumnInfo.isNullable())
        {
            pushAdaptNullability(queue,ti,tmpInfo,newColumnInfo);
        }
        
        // drop the column.
        this.pushDropColumn(schemaEngine, queue,ti,oldColumnInfo,true);
    }
    
    private static String buildRenameColumn(String tableName, ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo) 
    {
        StringBuffer sql = new StringBuffer();
        
        sql.append("RENAME COLUMN ");
        sql.append(tableName);
        sql.append('.');
        sql.append(oldColumnInfo.getName());
        sql.append(" TO ");
        sql.append(newColumnInfo.getName());

        return sql.toString();
    }
    
    private void pushRename(SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo) throws SQLException
    {
        
        queue.pushCommand(
                buildRenameColumn(ti.getName(),oldColumnInfo,newColumnInfo),
                buildRenameColumn(ti.getName(),newColumnInfo,oldColumnInfo) );
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushChangeColumn(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void pushChangeColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo)
            throws SQLException {

        // case 1: adpation of nullability and/or default value.
        if (oldColumnInfo.getType() == newColumnInfo.getType() &&
                        Util.equalsNullAware(oldColumnInfo.getPrecision(),newColumnInfo.getPrecision()) &&
                        Util.equalsNullAware(oldColumnInfo.getScale(),newColumnInfo.getScale()
                        ) )
        {
            ColumnInfo tmpInfo = ColumnHelper.rename(oldColumnInfo,newColumnInfo.getName());
            
            pushRename(queue, ti, oldColumnInfo, tmpInfo);
            
            pushAdaptNullabilityAndDefaultValue(queue,ti,tmpInfo,newColumnInfo);
            return;
        }
        
        // create->copy->drop workflow.
        pushCreateCopyDrop(schemaEngine, queue,ti,oldColumnInfo,newColumnInfo);
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#pushModifyColumn(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.sql.SqlCommandQueue, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void pushModifyColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo)
            throws SQLException {

        // adaption of nullability and/or default value.
        if (oldColumnInfo.getType() == newColumnInfo.getType() &&
                Util.equalsNullAware(oldColumnInfo.getPrecision(),newColumnInfo.getPrecision()) &&
                Util.equalsNullAware(oldColumnInfo.getScale(),newColumnInfo.getScale()))
        {
            pushAdaptNullabilityAndDefaultValue(queue,ti,oldColumnInfo,newColumnInfo);
            return;           
        }
        
        // rename column to *__DUP and proceed with create->copy->drop
        ColumnInfo tmpInfo = ColumnHelper.rename(oldColumnInfo,newColumnInfo.getName()+"__DUP");
        
        pushRename(queue, ti, oldColumnInfo, tmpInfo);
        pushCreateCopyDrop(schemaEngine, queue,ti,tmpInfo,newColumnInfo);
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#constructJDBCURL(java.lang.String, java.lang.Integer, java.lang.String, java.util.Properties)
     */
    public String constructJDBCURL(String hostname, Integer port,
            String databaseName, Properties properties) {
        // jdbc:derby://localhost:1527/mydb;create=true
        StringBuffer url=new StringBuffer("jdbc:derby:");
        if (   (hostname != null && hostname.length()>0)
            || (port!=null && port>0)                    )
        {
            url.append("//");
            if (hostname != null && hostname.length()>0)
                url.append(hostname);
            else
                url.append("localhost");
            url.append(':');
            if (port!=null && port>0)
                url.append(port);
            else
                url.append("1527");
            url.append('/');
        }
        if (databaseName!=null && databaseName.length()>0)
            url.append(databaseName);
        if (properties!=null && properties.size()>0) {
            Enumeration<?> propertyNames=properties.propertyNames();
            while (propertyNames.hasMoreElements()) {
                String propertyName=propertyNames.toString();
                String propertyValue=properties.getProperty(propertyName);
                url.append(';');
                url.append(propertyName);
                url.append('=');
                url.append(propertyValue);
            }
        }
        return url.toString();
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#getMappedSqlType(java.lang.String)
     */
    public int getMappedSqlType(String dialectDataType) {
        String effectiveType = dialectDataType.replace(' ', '_')
            .replaceAll("\\(.*?\\)", "")
            .trim();
        switch (effectiveType) {
            case "TIMESTAMP": return Types.TIMESTAMP;
            default: throw new DataTypeNotSupportedException(dialectDataType);
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.Dialect#fetchAdditionalColumnInfo(org.clazzes.jdbc2xml.schema.SchemaEngine, org.clazzes.jdbc2xml.schema.TableInfo, org.clazzes.jdbc2xml.schema.ColumnInfo)
     */
    public void fetchAdditionalColumnInfo(ISchemaEngine schemaEngine, TableInfo ti, 
            ColumnInfo ci) {
    }
}
