/***********************************************************
 * $Id$
 *
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 14.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.jdbc2xml.schema.impl;

import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.List;

import org.clazzes.jdbc2xml.helper.JAVAHelper;
import org.clazzes.jdbc2xml.schema.ColumnInfo;
import org.clazzes.jdbc2xml.schema.Dialect;
import org.clazzes.jdbc2xml.schema.ForeignKeyInfo;
import org.clazzes.jdbc2xml.schema.IndexInfo;
import org.clazzes.jdbc2xml.schema.PrimaryKeyInfo;
import org.clazzes.jdbc2xml.schema.TableInfo;
import org.clazzes.jdbc2xml.sql.SimpleSqlCommand;
import org.clazzes.jdbc2xml.sql.SqlCommand;

/**
 * Static helper methods to construct SQL queries for schema manipulations.
 *
 * @author wglas
 */
public abstract class DDLHelper {

    /**
     * Construct a create table statement.
     *
     * @param ti the table information.
     * @param dialect The dialect for creating column specs.
     * @param suffix An RDBMS-specific suffix, which will be appended
     *               after the create table clause.
     * @return A string, which carries the create table statement.
     */
    public static String buildCreateTable(TableInfo ti, Dialect dialect, String suffix)
    {
        StringBuffer sql = new StringBuffer();

        sql.append("CREATE TABLE ");
        sql.append(ti.getName());

        List<ColumnInfo> columns = ti.getColumns();

//        sql.append('\n');
        for (int i=0;i<columns.size();++i)
        {
            ColumnInfo ci = columns.get(i);

            if (i==0)
                sql.append(" (");
            else
                sql.append(" ,");

            String columnSpec = dialect.createColumnSpec(ci);
            sql.append(columnSpec);

//            sql.append('\n');
        }

        PrimaryKeyInfo pkInfo = ti.getPrimaryKey();

        if(pkInfo != null)
        {
            sql.append(" ,");
            if (pkInfo.getName()!=null && pkInfo.getName().length()>0)
                sql.append("CONSTRAINT "+pkInfo.getName());

            sql.append(" PRIMARY KEY ( " );
            JAVAHelper.joinStrings(sql,pkInfo.getColumns());
//            sql.append(" )\n" );
            sql.append(" )" );
        }


        sql.append(" )");

        if (suffix != null) {
//            sql.append('\n');
            sql.append(suffix);
        }

        return sql.toString();
    }

    /**
     * Build a drop table statement.
     *
     * @param tableName The name of the table to drop.
     * @return The SQL statement to drop the table.
     */
    public static String buildDropTable(String tableName)
    {
        StringBuffer sql = new StringBuffer();
        sql.append("DROP TABLE ");
        sql.append(tableName);
        return sql.toString();
    }

    /**
     * Legacy entry point retiaing compatibility wit jdbc2xml-1.3.x, equivalent to
     * <code>DDLHelper.buildAddIndex(ti,indexInfo,allowsMultipleNullsInUniqueIndizes,false)</code>.
     *
     * @param ti the table information.
     * @param indexInfo The description of the index to be created.
     * @param allowsMultipleNullsInUniqueIndizes Specifies if a UNIQUE INDEX may contain multiple NULL values.
     *                                   If false, INDEXes for NULLable columns must be created non-UNIQUE.
     * @return The SQL statement to add the index.
     * @throws SQLException If the index does not contain a name.
     */
    public static String buildAddIndex(TableInfo ti,
            IndexInfo indexInfo,
            boolean allowsMultipleNullsInUniqueIndizes) throws SQLException
    {
        return DDLHelper.buildAddIndex(ti,indexInfo,allowsMultipleNullsInUniqueIndizes,false);
    }

    /**
     * @param ti the table information.
     * @param indexInfo The description of the index to be created.
     * @param allowsMultipleNullsInUniqueIndizes Specifies if a UNIQUE INDEX may contain multiple NULL values.
     *                                   If false, INDEXes for NULLable columns must be created non-UNIQUE.
     * @return The SQL statement to add the index.
     * @throws SQLException If the index does not contain a name.
     */
    public static String buildAddIndex(TableInfo ti,
            IndexInfo indexInfo,
            boolean allowsMultipleNullsInUniqueIndizes,
            boolean allowsIncludeColumns) throws SQLException
    {
        StringBuffer sql = new StringBuffer();
        sql.append("CREATE ");
        if (indexInfo.isUnique()
                && (allowsMultipleNullsInUniqueIndizes
                        || !ti.indexContainsNullableColumns(indexInfo))) {
            sql.append("UNIQUE ");
        }
        sql.append("INDEX ");
        if (indexInfo.getName()==null || indexInfo.getName().length()==0)
            throw new SQLException("doAddIndexInfo: All indices must have non-null and non-empty names");

        sql.append(indexInfo.getName());
        sql.append(" ON ");
        sql.append(ti.getName());
        sql.append(" ( " );
        JAVAHelper.joinStrings(sql,indexInfo.getColumns());
        sql.append(" )" );

        boolean includeColumnsExists = indexInfo.getIncludeColumns() != null && !indexInfo.getIncludeColumns().isEmpty();
        if (allowsIncludeColumns && includeColumnsExists) {
            sql.append(" INCLUDE ( " );
            JAVAHelper.joinStrings(sql,indexInfo.getIncludeColumns());
            sql.append(" )" );
        }

        //FIXME indexInfo.filterCondition is not used!?

        return sql.toString();
    }

    /**
     * @param tableName The name of the table on which to drop the index.
     * @param indexName The name of the index to drop.
     * @param needsOnTableSuffix Specifies if DROP INDEX needs (or at least allows) the ON tableName suffix
     * @return The SQL statement to drop the index.
     * @throws SQLException  If the index does not contain a name.
     */
    public static String buildDropIndex(String tableName,
            String indexName, boolean needsOnTableSuffix) throws SQLException {

        StringBuffer sql = new StringBuffer();
        sql.append("DROP INDEX ");
        if (indexName==null || indexName.length()==0)
            throw new SQLException("dropIndex: Indices have non-null and non-empty names");
        sql.append(indexName);
        if (needsOnTableSuffix) {
            sql.append(" ON ");
            sql.append(tableName);
        }
        return sql.toString();
    }

    /**
     * @param tableName The name of the table on which to add a foreign key.
     * @param foreignKeyInfo The description of the foreign key.
     * @param addStmtInBrackets <code>true</code> will produce <code>ADD (statement)</code>, <code>false</code> for: <code>ADD statement</code>.
     * @return The SQL statement to create the foreign key.
     * @throws SQLException If the foreign key uses unimplemented features.
     */
    public static String buildAddForeignKey(String tableName, ForeignKeyInfo foreignKeyInfo, boolean addStmtInBrackets) throws SQLException
    {
        StringBuffer sql = new StringBuffer();
        sql.append("ALTER TABLE ");
        sql.append(tableName);
        sql.append(" ADD ");
        if (addStmtInBrackets)
            sql.append("(");
        if (foreignKeyInfo.getName()!=null && foreignKeyInfo.getName().length()>0)
            sql.append(" CONSTRAINT "+foreignKeyInfo.getName());
        sql.append(" FOREIGN KEY ( " );
        JAVAHelper.joinStrings(sql,foreignKeyInfo.getColumns());
        sql.append(" ) REFERENCES " );
        sql.append(foreignKeyInfo.getForeignTable());
        sql.append(" ( " );
        JAVAHelper.joinStrings(sql,foreignKeyInfo.getForeignColumns());
        sql.append(" )" );

        switch (foreignKeyInfo.getDeleteRule())
        {
        case DatabaseMetaData.importedKeyCascade:
            sql.append(" ON DELETE CASCADE");
            break;
        case DatabaseMetaData.importedKeyRestrict:
            // According to the JAVAdoc of DtabaseMetadata#getImportedKeys,
            // DatabaseMetaData#importedKeyRestrict is sometimes reported,
            // when DatabaseMetaData#importedKeyNoAction is meant.
            // (due to ODBC 2.0 compatibility...)
            // However, not all databases support the notion
            // of an 'ON DELETE RESTRICT' foreign key, so for both reasons
            // we never try to use DatabaseMetaData.importedKeyRestrict.
            break;
        case DatabaseMetaData.importedKeySetDefault:
            sql.append(" ON DELETE SET DEFAULT");
            break;
        case DatabaseMetaData.importedKeySetNull:
            sql.append(" ON DELETE SET NULL");
            break;
        }

        switch (foreignKeyInfo.getUpdateRule())
        {
        case DatabaseMetaData.importedKeyCascade:
            // FIXME: sql.append(" ON UPDATE CASCADE");
            break;
        case DatabaseMetaData.importedKeyRestrict:
            // see comments on DatabaseMetaData#importedKeyRestrict above.
            break;
        case DatabaseMetaData.importedKeySetDefault:
            sql.append(" ON UPDATE SET DEFAULT");
            break;
        case DatabaseMetaData.importedKeySetNull:
            sql.append(" ON UPDATE SET NULL");
            break;
        }

        if (foreignKeyInfo.getDeferrability() != DatabaseMetaData.importedKeyNotDeferrable)
            throw new SQLException("foreign key deferrability is not implemented yet.");
        if (addStmtInBrackets)
            sql.append(")");
        return sql.toString();
    }

    /**
     * @param tableName The name of the table on which to add a foreign key.
     * @param foreignKeyName The name of the foreign key to drop.
     * @param dropForeignKeyCommand The RDBMS-specific command after <code>ALTER TABLE</code>
     *             needed to drop a foreign key.
     * @return The SQL statement to create the foreign key.
     */
    public static String buildDropForeignKey(String tableName, String foreignKeyName,
            String dropForeignKeyCommand)
    {
        StringBuffer sql = new StringBuffer();
        sql.append("ALTER TABLE ");
        sql.append(tableName);
        sql.append(' ');
        sql.append(dropForeignKeyCommand);
        sql.append(' ');
        sql.append(foreignKeyName);

        return sql.toString();
    }

    /**
     * @param tableName the table name.
     * @param ci The description of the column to be created.
     * @param dialect The dialect for creating column specs.
     * @param addColumnCommand The RDBMS specific command after <code>ALTER TABLER</code>
     *               for adding a column.
     * @return The SQL statement to add the column.
     */
    public static String buildAddColumn(String tableName,
            ColumnInfo ci, Dialect dialect,
            String addColumnCommand, boolean addStmtInBrackets)
    {
        StringBuffer sql = new StringBuffer();
        sql.append("ALTER TABLE ");
        sql.append(tableName);
        sql.append(" ");
        sql.append(addColumnCommand);
        if (addStmtInBrackets)
            sql.append(" ( ");
        else
            sql.append(" ");

        sql.append(dialect.createColumnSpec(ci));
        if (addStmtInBrackets)
            sql.append(" )");
        return sql.toString();
    }

    /**
     * @param tableName The name of the table on which to drop the column.
     * @param columnName The name of the column to drop.
     * @return The SQL statement to drop the column.
     */
    public static String buildDropColumn(String tableName,
            String columnName) {
        final StringBuffer sql = new StringBuffer();
        sql.append("ALTER TABLE ");
        sql.append(tableName);
        sql.append(" ");
        sql.append("DROP COLUMN ");
        sql.append(columnName);
        return sql.toString();
     }

    /**
     * Generate a rename table command. Please note, that only some
     * RDBMS engines support renaming tables.
     *
     * @param renameTableCommand The dialect specific command to rename a table, which must contain
     *                          two <code>%s</code> placeholders, the first one for the original
     *                          table name, second one for the new table name.
     *                          If set to <code>null</code>, the ISO SQL command
     *                          <code>RENAME TABLE %s TO %s</code> is used for renaming the table.
     * @param tableName The name of the table which will be renamed.
     * @param newTableName The name of the table after renaming.
     * @return The SQL command to rename the table.
     */
    public static SqlCommand buildRenameTable(String renameTableCommand, String tableName,
            String newTableName) {

        if (renameTableCommand == null)
            renameTableCommand = "RENAME TABLE %s TO %s";

        return new SimpleSqlCommand(
                String.format(renameTableCommand, tableName, newTableName),
                String.format(renameTableCommand, newTableName, tableName)
                );
     }

    /**
     * @return An 8-digit hexadecimal suffix build from a hash of the urrent system time,
     *         which is used to arbitrate object names created by jdbc2xml.
     */
    public static String buildHexSuffix()
    {
        return Long.toHexString((System.currentTimeMillis()*479001599L) & 0xffffffffL);
    }
}
