/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Created: 17.12.2007
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema.impl;

import org.clazzes.jdbc2xml.schema.ColumnInfo;
import org.clazzes.util.lang.Util;

/**
 * Helper functions to play with ColumnInfo objects.
 * 
 * @author wglas
 */
public abstract class ColumnHelper {

    /**
     * Return a column information with the given nullability set.
     * 
     * @param ci The original column information.
     * @param nullable The nullable flag to set to the returned object.
     * @return A column description with the desired nullability flag.
     *         If the nullability of the original column description is equal to the
     *         desired nullability, the original description is returned.
     */
    public static ColumnInfo adaptNullability(ColumnInfo ci, boolean nullable)
    {
        if (ci.isNullable() == nullable) return ci;
        
        return new ColumnInfo(ci.getName(),ci.getType(),ci.getPrecision(),ci.getScale(),nullable,ci.getDefaultValue());
    }
    
    /**
     * Return a column information with the given default value.
     * 
     * @param ci The original column information.
     * @param defaultValue The default value to set to the returned object.
     * @return A column description with the desired default value.
     *         If the default value of the original column description is equal to the
     *         desired default value, the original description is returned.
     */
    public static ColumnInfo adaptDefault(ColumnInfo ci, String defaultValue)
    {
        if (Util.equalsNullAware(ci.getDefaultValue(),defaultValue)) return ci;
        
        return new ColumnInfo(ci.getName(),ci.getType(),ci.getPrecision(),ci.getScale(),ci.isNullable(),defaultValue);
    }
    
    /**
     * Return a column information with the given name.
     * 
     * @param ci The original column information.
     * @param name The name to set to the returned object.
     * @return A column description with the name.
     *         If the name of the original column description is equal to the
     *         desired name, the original description is returned.
     */
    public static ColumnInfo rename(ColumnInfo ci, String name)
    {
        if (Util.equalsNullAware(ci.getName(),name)) return ci;
        
        return new ColumnInfo(name,ci.getType(),ci.getPrecision(),ci.getScale(),ci.isNullable(),ci.getDefaultValue());
    }
    
}
