/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 12.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is used to analyse the foreign key dependencies of a list of tables.
 * 
 * @author wglas
 */
public abstract class TableSorter {

    private static Logger log = LoggerFactory.getLogger(TableSorter.class); 
    
    private static void getFKDepthRecursive(Map<String,SortableTableDescription> tablesByName,
            SortableTableDescription si)
    {
        si.setFkDepth(0);
        
        if (si.getTableInfo().getForeignKeys() != null)
        {
            for (ForeignKeyInfo fkInfo : si.getTableInfo().getForeignKeys())
            {
                // skip self-references.
                if (fkInfo.getForeignTable().equalsIgnoreCase(si.getTableInfo().getName())) continue;
                
                SortableTableDescription foreignSi = tablesByName.get(fkInfo.getForeignTable());
                
                if (foreignSi == null) continue;
                
                if (foreignSi.getFkDepth() < 0)
                    getFKDepthRecursive(tablesByName, foreignSi);
                    
                int fDepth = foreignSi.getFkDepth() + 1;
                
                if (fDepth > si.getFkDepth()) si.setFkDepth(fDepth);
            }
        }
        
        if (log.isDebugEnabled())
            log.debug("Table ["+si.getTableInfo().getName()+"] has a foreign key depth of ["+si.getFkDepth()+"].");
    }

    /**
     * This subroutine classifies tables by their foreign key dependencies.
     * The fkDepth field of the {@link SortableTableDescription} objects is
     * initialized and the list sort w.r.t. to this depth.
     * 
     * The sorting happens in-place and leaves the list in ascending order of depth.
     * 
     * @param tables A list of sortable table informations.
     */
    public static void sortTablesByFKDepth(List<? extends SortableTableDescription> tables)
    {
        Map<String,SortableTableDescription> tablesByName =
            new HashMap<String,SortableTableDescription>(tables.size());
        
        for (SortableTableDescription td : tables)
        {
            td.setFkDepth(-1);
            if (td.getTableInfo() == null) continue;
            
            tablesByName.put(td.getTableInfo().getName(),td);
        }
        
        for (SortableTableDescription td : tables)
        {
            TableInfo ti = td.getTableInfo();
            
            if (ti == null)
                td.setFkDepth(Integer.MAX_VALUE);
            else if (td.getFkDepth() < 0)
                getFKDepthRecursive(tablesByName,td);
         }
        
        Collections.sort(tables);
    }
}
