/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 28.11.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema;

import java.util.ArrayList;
import java.util.List;

/**
 * This class hold everything we need to known about a database table.
 * 
 * @author wglas
 */
public class TableInfo {

    private String name;
    private String comment;
    private List<ColumnInfo> columns;
    private PrimaryKeyInfo primaryKey;
    private List<ForeignKeyInfo> foreignKeys;
    private List<IndexInfo> indices;
    
    /**
     * Default constructor
     */
    public TableInfo()
    { }
    
    /**
     * Constructor with table name.
     */
    public TableInfo(String name)
    { 
        this.name = name;
    }

    /**
     * @return the name
     */
    public String getName() {
        return this.name;
    }

    /**
     * @param name the name to set
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * @return the comment
     */
    public String getComment() {
        return this.comment;
    }

    /**
     * @param comment the comment to set
     */
    public void setComment(String comment) {
        this.comment = comment;
    }

    /**
     * @return the columns
     */
    public List<ColumnInfo> getColumns() {
        return this.columns;
    }

    /**
     * @param columns the columns to set
     */
    public void setColumns(List<ColumnInfo> columns) {
        this.columns = columns;
    }

    /**
     * Add a column to our list of column descriptions.
     * 
     * @param ci The ColumnInfo to add.
     */
    public void addColumn(ColumnInfo ci) {
        
        if (this.columns == null)
            this.columns = new ArrayList<ColumnInfo>();
        
        this.columns.add(ci);
    }
    
    /**
     * Return the column info for the given column name.
     * 
     * @param columnName The name of the column to seek for.
     * @return The column information or null, if the column could not be found.
     */
    public ColumnInfo getColumnInfo(final String columnName)
    {
        if (this.columns == null || columnName == null)
            return null;
        for (final ColumnInfo ci : this.columns)
            if (ci.getName().equalsIgnoreCase(columnName))
                return ci;
     
        return null;
    }
    
    /**
     * Replace a defined <code>ColumnInfo</code> by another one.
     * If not found, nothing happens.
     * @param columnName Name of the column to replace.
     * @param newColumnInfo The new <code>ColumnInfo</code> for the specified column.
     */
    public void replaceColumnInfo(final String columnName, final ColumnInfo newColumnInfo)
    {
        if (this.columns == null || columnName == null || newColumnInfo == null)
            return;
        for (int i=0; i<this.columns.size(); i++)
        {
            final ColumnInfo ci = this.columns.get(i);
            if (ci.getName().equalsIgnoreCase(columnName))
            {
                this.columns.set(i, newColumnInfo);
                return;
            }
        }
    }
    
    /**
     * Remove a defined <code>ColumnInfo</code>.
     * If not found, nothing happens.
     * @param columnName Name of the column to remove.
     */
    public void removeColumnInfo(final String columnName)
    {
        if (this.columns == null || columnName == null)
            return;
        int index = -1;
        for (int i=0; i<this.columns.size(); i++)
        {
            final ColumnInfo ci = this.columns.get(i);
            if (ci.getName().equalsIgnoreCase(columnName))
            {
               index = i;
               break;
            }
        } 
        if (index >= 0)
            this.columns.remove(index);
    }
    
    /**
     * Remove a defined <code>indexInfo</code>.
     * If not found, nothing happens.
     * @param indexName Name of the index to remove.
     */
    public void removeIndex(final String indexName)
    {
        if (this.indices == null || indexName == null)
            return;
        int indexOffset = -1;
        for (int i=0; i<this.indices.size(); i++)
        {
            final IndexInfo ii = this.indices.get(i);
            if (indexName.equalsIgnoreCase(ii.getName()))
            {
               indexOffset = i;
               break;
            }
        } 
        if (indexOffset >= 0)
            this.indices.remove(indexOffset);
    }

    /**
     * @return the primaryKey
     */
    public PrimaryKeyInfo getPrimaryKey() {
        return this.primaryKey;
    }
    
    /**
     * @param primaryKey The primary key to set.
     */
    public void setPrimaryKey(PrimaryKeyInfo primaryKey) {
        this.primaryKey = primaryKey;
    }

    /**
     * @return the foreignKeys
     */
    public List<ForeignKeyInfo> getForeignKeys() {
        return this.foreignKeys;
    }

    /**
     * @param foreignKeys the foreignKeys to set
     */
    public void setForeignKeys(List<ForeignKeyInfo> foreignKeys) {
        this.foreignKeys = foreignKeys;
    }    
    
    /**
     * Add a foreign key to our list of foreign key descriptions.
     * 
     * @param fki The PrimaryKeyInfo to add.
     */
    public void addForeignKey(ForeignKeyInfo fki) {
        
        if (this.foreignKeys == null)
            this.foreignKeys = new ArrayList<ForeignKeyInfo>();
        
        this.foreignKeys.add(fki);
    }
    
    /**
     * Get a <code>ForeignKeyInfo</code> for a specified foreign-key-name.
     * @param fkName Name of the foreign key to get.
     * @return The <code>ForeignKeyInfo</code> found, or null if not found.
     */
    public ForeignKeyInfo getForeignKeyInfo(final String fkName)
    {
        if (this.foreignKeys == null || fkName == null)
            return null;
        for (ForeignKeyInfo fki : this.foreignKeys)
        {
            if (fki != null && fkName.equalsIgnoreCase(fki.getName()))
            {
                return fki;
            }
        }
        return null;
    }
    
    /**
     * Get a <code>IndexInfo</code> for a specified foreign-key-name.
     * @param idxName Name of the index to get.
     * @return The <code>IndexInfo</code> found, or null if not found.
     */
    public IndexInfo getIndexInfo(final String idxName)
    {
        if (this.indices == null || idxName == null)
            return null;
        for (IndexInfo idxInfo : this.indices)
        {
            if (idxInfo != null && idxName.equalsIgnoreCase(idxInfo.getName()))
            {
                return idxInfo;
            }
        }
        return null;
    }
    
    /**
     * Remove a defined <code>ForeignKeyInfo</code>.
     * If not found, nothing happens.
     * @param fkName Name of the foreign key to remove.
     */
    public void removeForeignKey(final String fkName)
    {
        if (this.foreignKeys == null || fkName == null)
            return;
        int index = -1;
        for (int i=0; i<this.foreignKeys.size(); i++)
        {
            if (fkName.equalsIgnoreCase(this.foreignKeys.get(i).getName()))
            {
                index = i;
                break;
            }
        }
        if (index >= 0)
            this.foreignKeys.remove(index);
    }

    /**
     * @return The list of table indices.
     */
    public List<IndexInfo> getIndices() {
        return this.indices;
    }

    /**
     * @param indices The list of indices to set.
     */
    public void setIndices(List<IndexInfo> indices) {
        this.indices = indices;
    }
    
    /**
     * Add an index to the list of indices.
     * 
     * @param indexInfo The index to add.
     */
    public void addIndex(IndexInfo indexInfo) {
     
        if (this.indices == null)
            this.indices = new ArrayList<IndexInfo>();
 
        this.indices.add(indexInfo);
    }

    /**
     * Add an index to the list of indices.
     * 
     * @param indexName name of the index
     * @param indexColumns column list
     * @param unique if the index should be unique
     * @param filterCondition
     */
    public void addIndex(String indexName, String indexColumns, boolean unique, String filterCondition) {
     
        IndexInfo indexInfo=new IndexInfo(indexName, indexColumns, unique, filterCondition);
 
        this.addIndex(indexInfo);
    }

    /**
     * Investigates if an IndexInfo contains a NULLable column
     * @param indexInfo the Index to investigate
     * @return true if the IndexInfo contains a NULLable column
     */
    public boolean indexContainsNullableColumns(IndexInfo indexInfo) {
        if (indexInfo!=null && indexInfo.getColumns()!=null && indexInfo.getColumns().size()>0) {
            for (String colName : indexInfo.getColumns()) {
                ColumnInfo colInfo = getColumnInfo(colName);
                if (colInfo.isNullable())
                    return true;
            }
        }
        return false;
    }

    @Override
    public String toString() {
        return "TableInfo{" +
                "name='" + name + '\'' +
                ", comment='" + comment + '\'' +
                '}';
    }
}
