/***********************************************************
 * $$Id$$
 *
 * J2EE-Sandbox classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.jdbc2xml.schema;

import java.lang.reflect.InvocationTargetException;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.sql.DataSource;

import org.clazzes.jdbc2xml.schema.impl.SchemaCheckerBean;
import org.clazzes.jdbc2xml.schema.impl.SchemaEngineFactoryImpl;
import org.clazzes.util.lang.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Bean for initial database setup.
 */
public class SchemaManager {

    private static final Logger log = LoggerFactory.getLogger(SchemaManager.class);

    /**
     * The table information for the default version history table.
     * This member is provided for binary backward compatibility.
     */
    @Deprecated
    public static final TableInfo VERSION_HISTORY = makeVersionHistoryInfo("SCHEMA_HISTORY",null);

    private static TableInfo makeVersionHistoryInfo(String tableName,String schemaName) {

        TableInfo ti = new TableInfo();
        ti.setName(schemaName == null ? tableName:schemaName+"."+tableName);
        ti.setColumns(
                      Arrays.asList(new ColumnInfo[] {
                              new ColumnInfo("VERSION", Types.VARCHAR, 10, null, false, null),
                              new ColumnInfo("DESCRIPTION", Types.VARCHAR, 512, null, true, null),
                              new ColumnInfo("CREATION_DATE", Types.TIMESTAMP, 12, null, true, null),
                              new ColumnInfo("SERIALNR", Types.INTEGER, 5, null, false, null)
                          })
                      );
        ti.setPrimaryKey(
                         new PrimaryKeyInfo(tableName+"_PK", "VERSION"));

        return ti;

    }

    private String baseVersion = "0.1.00";

    private String versionHistoryTable = "SCHEMA_HISTORY";

    private String versionHistorySchema = null;

    private String baseDescription = "initial database schema";

    private List<TableInfo> baseTables;

    private DataSource dataSource;

    private ISchemaEngine schemaEngine;

    private Map<String,String> dialectProperties;

    private Map<String, ISchemaUpdateSnippet> updateSnippets;

    private ISchemaUpdateSnippet baseMigration;

    private <T> T withSchema(String schemaName, Supplier<T> fn) {
        if(schemaName == null) {
            return fn.get();
        }
        String oldSchema = this.schemaEngine.getSchema();
        this.schemaEngine.setSchema(schemaName);
        try {
            return fn.get();
        } finally {
            this.schemaEngine.setSchema(oldSchema);
        }
    }

    public SchemaManager() {
        // Do nothing, use setters
    }

    public SchemaManager(String baseVersion, List<TableInfo> baseTables, Map<String, Class<? extends ISchemaUpdateSnippet>> updateSnippets) {
        ISchemaEngineFactory schemaEngineFactory = new SchemaEngineFactoryImpl();
        ISchemaEngine defaultSchemaEngine = schemaEngineFactory.newSchemaEngine();

        this.setSchemaEngine(defaultSchemaEngine);
        this.setBaseVersion(baseVersion);
        this.setBaseTables(baseTables);
        this.setUpateSnippets(updateSnippets);
    }

    /**
     * Setup-method. Checks if the schema-history table exists; if it does, it attempts to run a schema-update, otherwise
     * the database is set up according to the {@link TableInfo}s saved in {@link #baseTables}, and then runs an
     * update.
     * @throws SchemaManagementException
     */
    public void start() throws SchemaManagementException        {
        log.info("setting up database schema: performing initial checks...");

        if (this.dataSource == null) {

            throw new IllegalArgumentException("Improper Configuration. No dataSource present");
        }

        if ((this.baseTables == null || this.baseTables.isEmpty())
            ^ (this.baseMigration != null)) {
            throw new IllegalArgumentException("Improper Configuration. No table list set");
        }

        try {
            this.schemaEngine.setConnection(this.dataSource.getConnection());
            this.schemaEngine.getDialect().setProperties(this.dialectProperties);

        } catch (SQLException e) {

            // a SQL exception at this point is really a fatal attempt to connect to a misconfigured DB.
            throw new SchemaManagementException("Cannot create initial connection to DB, check your configuration",e);
        }

        try {
            if (log.isDebugEnabled())
                log.debug("Checking for {} table...",this.versionHistoryTable);

            TableInfo versionHistory = makeVersionHistoryInfo(this.versionHistoryTable,this.versionHistorySchema);

            TableInfo check = withSchema(this.versionHistorySchema,()->{
                    try {
                        return this.schemaEngine.fetchTableInfo(this.versionHistoryTable, null);
                    } catch (SQLException e) {
                        // a SQL exception at this point signifies, that the schema history does not exist.
                        if (log.isDebugEnabled())
                            log.debug("Caught SQLException [{}]; Expected clear database, attempting database setup from scratch...", e.getMessage());

                        setupDB();
                        initUpdate();

                        return null;
                    }
                });
            if (check == null) {
                return;
            }

            List<ColumnInfo> columns = check.getColumns();

            if (columns.size() != 4)
                throw new SchemaManagementException("Table "+versionHistory.getName()+" has not 4 columns");

            for (int i=0;i<4;++i) {

                if (!columns.get(i).getName().equalsIgnoreCase(versionHistory.getColumns().get(i).getName())) {
                    throw new SchemaManagementException("Column ["+i+
                                                        "] of table "+versionHistory.getName()+" has wrong name ["+columns.get(i).getName()+
                                                        "]  (expected ["+versionHistory.getColumns().get(i).getName()+"]).");
                }
            }

            if (log.isDebugEnabled())
                log.debug("Found {}; Initiating schema-update...",this.versionHistoryTable);

            initUpdate();
        } finally {
            Connection conn = this.schemaEngine.getConnection();
            try {
                this.schemaEngine.setConnection(null);
                conn.close();
            } catch (SQLException e) {
                throw new SchemaManagementException("Error closing connection to DB, please chack your database manually",e);
            }
        }
    }

    /**
     * Sets up the tables as described in {@link #baseTables}.
     * @throws SchemaManagementException upon errors
     */
    protected void setupDB() throws SchemaManagementException {

        log.info("Setting up initial database with version [{}]...",this.baseVersion);

        boolean success = false;

        try {
            this.schemaEngine.createTable(makeVersionHistoryInfo(this.versionHistoryTable,this.versionHistorySchema), false);
            this.schemaEngine.commit();

            log.info("Initial database with version [{}] has successfully been setup.",this.baseVersion);

            success = true;

        } catch (SQLException e) {

            throw new SchemaManagementException("Caught SQLException while setting up initial database",e);
        }
        finally {
            if (!success) {
                try {
                    this.schemaEngine.rollback();
                } catch (Throwable e) {
                    log.warn("Error rolling back the schema after an error setting up the intial database, your database may neeed manual cleanup",e);
                }
            }
        }
    }

    /**
     * Triggers a schema-update.
     */
    protected void initUpdate() throws SchemaManagementException {

        log.info("Performing database schema update...");
        // perform schemaUp

        try {

            SchemaCheckerBean schemaChecker = new SchemaCheckerBean(this.schemaEngine,this.updateSnippets,makeVersionHistoryInfo(this.versionHistoryTable,this.versionHistorySchema),this.getInitialMigration());
            schemaChecker.checkSchema();
        } catch (InstantiationException e) {

            throw new SchemaManagementException("InstantiationException while updating schema",e);

        } catch (IllegalAccessException e) {

            throw new SchemaManagementException("Caught IllegalAccessException while updating schema",e);

        } catch (SQLException e) {

            throw new SchemaManagementException("Caught SQLException while updating schema",e);
        }
    }

    public ISchemaUpdateSnippet getInitialMigration() {
        return this.baseMigration == null ? new ISchemaUpdateSnippet() {

                @Override
                public String getTargetVersion() {
                    return SchemaManager.this.baseVersion;
                }

                @Override
                public String getUpdateComment() {
                    return SchemaManager.this.baseDescription;
                }

                @Override
                public void performUpdate(ISchemaEngine schemaEngine)
                    throws SQLException {
                    for (TableInfo table : SchemaManager.this.baseTables) {
                        schemaEngine.createTable(table, false);

                        if(log.isDebugEnabled())
                            log.debug("Created Table [{}]", table.getName());
                    }

                    // Create foreign keys afterwards, as the destination table might not yet exist in the above loop.
                    for (TableInfo tableInfo : SchemaManager.this.baseTables) {
                        schemaEngine.createForeignKeys(tableInfo);
                        if (log.isDebugEnabled()) {
                            log.debug("Created Foreign Keys for Table [{}]", tableInfo.getName());
                        }
                    }
                }
            } : this.baseMigration;
    }

    /**
     * @return the dataSource
     */
    public DataSource getDataSource() {
        return this.dataSource;
    }

    /**
     * @param dataSource the dataSource to set
     */
    public void setDataSource(DataSource dataSource) {
        this.dataSource = dataSource;
    }

    /**
     * @return the schemaEngine
     */
    public ISchemaEngine getSchemaEngine() {
        return this.schemaEngine;
    }

    /**
     * @param schemaEngine the schemaEngine to set
     */
    public void setSchemaEngine(ISchemaEngine schemaEngine) {
        this.schemaEngine = schemaEngine;
    }

    /**
     * @return The initial schema version.
     */
    public String getBaseVersion() {
        return this.baseVersion;
    }

    /**
     * @param baseVersion The initial version to set.
     */
    public void setBaseVersion(String baseVersion) {
        this.baseVersion = baseVersion;
    }

    /**
     * @return The description of the initial database version.
     */
    public String getBaseDescription() {
        return this.baseDescription;
    }

    /**
     * @param baseDescription The description of the initial database version.
     */
    public void setBaseDescription(String baseDescription) {
        this.baseDescription = baseDescription;
    }

    /**
     * @return The list of tables in the initial database setup.
     */
    public List<TableInfo> getBaseTables() {
        return this.baseTables;
    }

    /**
     * @param tables The list of tables in the initial database setup to set.
     */
    public void setBaseTables(List<TableInfo> tables) {
        this.baseTables = tables;
    }

    /**
     * @return The map of update snippets keyed to their originating version.
     */
    public Map<String, Class<? extends ISchemaUpdateSnippet>> getUpateSnippets() {
        return this.updateSnippets
            .entrySet()
            .stream()
            .map(a -> new Pair<>(a.getKey(),a.getValue().getClass()))
            .collect(Collectors.toMap(Pair::getFirst,Pair::getSecond));
    }

    /**
     * @param upateSnippets The map of update snippets keyed to their originating version.
     */
    public void setUpateSnippets(
                                 Map<String, Class<? extends ISchemaUpdateSnippet>> upateSnippets) {
        this.updateSnippets = upateSnippets
            .entrySet()
            .stream()
            .map(a -> new Pair<>(a.getKey(),construct(a.getValue())))
            .collect(Collectors.toMap(Pair::getFirst,Pair::getSecond));
    }

    private static final <T> T construct(Class<? extends T> clazz) {
        try {
            return clazz.getConstructor().newInstance();
        } catch (InstantiationException | IllegalAccessException
                 | IllegalArgumentException | InvocationTargetException
                 | NoSuchMethodException | SecurityException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * @return The name of the table, which record version informations.
     */
    public String getVersionHistoryTable() {
        return this.versionHistoryTable;
    }

    /**
     * @param versionHistoryTable The name of the table, which record version informations.
     */
    public void setVersionHistoryTable(String versionHistoryTable) {
        this.versionHistoryTable = versionHistoryTable;
    }

    public void setVersionHistorySchema(String versionHistorySchema) {
        this.versionHistorySchema = versionHistorySchema;
    }

    /**
     * A key and a value to pass to {@link Dialect#setProperties(Map)}.
     *
     * @param key The dialect property.
     * @param value The value of the property to set.
     */
    public void setDialectProperty(String key, String value) {

        if (this.dialectProperties == null) {
            this.dialectProperties = new HashMap<String,String>();
        }
        this.dialectProperties.put(key,value);
    }

    /**
     * @return A set of properties to pass to {@link Dialect#setProperties(Map)}.
     */
    public Map<String, String> getDialectProperties() {
        return this.dialectProperties;
    }

    /**
     * @param dialectProperties A set of properties to pass to
     *            {@link Dialect#setProperties(Map)}.
     */
    public void setDialectProperties(Map<String, String> dialectProperties) {
        this.dialectProperties = dialectProperties;
    }

    public void setUpdateSnippetInstances(Map<String,ISchemaUpdateSnippet> updateSnippetInstances) {
        this.updateSnippets = updateSnippetInstances;
    }

    public Map<String,ISchemaUpdateSnippet> getUpdateSnippetInstances() {
        return this.updateSnippets;
    }

    public void setBaseMigration(Class<? extends ISchemaUpdateSnippet> baseMigration) {
        this.baseMigration = construct(baseMigration);
    }

    public void setBaseMigrationInstance(ISchemaUpdateSnippet baseMigration) {
        this.baseMigration = baseMigration;
    }
}
