/***********************************************************
 * $Id$
 *
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 28.11.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.jdbc2xml.schema;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.TimeZone;

import org.clazzes.jdbc2xml.helper.ServiceHelper;
import org.clazzes.jdbc2xml.schema.impl.SchemaEngineImpl;

/**
 * An implementation of this class is responsible for creating/modifying a database schema
 * based on table descriptions.
 */
public abstract class SchemaEngine implements ISchemaEngine {

    private final IDialectFactory dialectFactory;
    private Connection connection;
    private TimeZone timeZone;
    private String schema;
    private Dialect dialect;

    /**
     * <p>After generating a new SchemaEngine instance you usually have to
     * call {@link #setConnection(Connection)} in order to set up the database
     * on which to operate.</p>
     *
     * <p>Additionally, it is recommended, that you configure the default time zone
     * using {{@link #setTimeZone(TimeZone)}.</p>
     *
     * <p>This method does not work in OSGi containers. You should use the service
     * implementing {@link ISchemaEngineFactory}, which is registered, when jdbc2xml is
     * deployed inside a blueprint-enabled OSGi container.
     * </p>
     *
     * @return The first implementation registered under the resource path
     *         <code>META-INF/services/org.clazzes.jdbc2xml.schema.SchemaEngine</code>.
     */
    static public ISchemaEngine newInstance() {
        // Äquivalent zu new SchemaEngineImpl();
        return ServiceHelper.loadService(SchemaEngine.class,SchemaEngineImpl::new);
    }

    /**
     * Default constructor, which sets the default time zone.
     */
    protected SchemaEngine()
    {
        this.timeZone = TimeZone.getDefault();
        this.dialectFactory = DialectFactory.newInstance();
    }

    /**
     * Default constructor, which sets the default time zone.
     */
    protected SchemaEngine(IDialectFactory dialectFactory)
    {
        this.timeZone = TimeZone.getDefault();
        this.dialectFactory = dialectFactory;
    }

   /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.ISchemaEngine#getConnection()
     */
    public Connection getConnection() {
        return this.connection;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.ISchemaEngine#setConnection(java.sql.Connection)
     */
    public void setConnection(Connection connection) throws SQLException {
        this.connection = connection;
        if (this.dialect == null) {
            this.dialect = this.dialectFactory.newDialect(connection.getMetaData());
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.ISchemaEngine#getTimeZone()
     */
    public TimeZone getTimeZone() {
        return this.timeZone;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.ISchemaEngine#setTimeZone(java.util.TimeZone)
     */
    public void setTimeZone(TimeZone timeZone) {
        this.timeZone = timeZone;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.ISchemaEngine#getDialect()
     */
    public Dialect getDialect() {
        return this.dialect;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.ISchemaEngine#setDialect(org.clazzes.jdbc2xml.schema.Dialect)
     */
    public void setDialect(Dialect dialect) {
        this.dialect = dialect;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.ISchemaEngine#getSchema()
     */
    public String getSchema() {
        return this.schema;
    }

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.schema.ISchemaEngine#setSchema(java.lang.String)
     */
    public void setSchema(String schema) {
        this.schema = schema;
    }

}
