/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 28.11.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema;

import java.util.ArrayList;
import java.util.List;

import org.clazzes.jdbc2xml.helper.JAVAHelper;
import org.clazzes.jdbc2xml.helper.SQLHelper;
import org.clazzes.jdbc2xml.sql.SqlIdentifierMapper;
import org.xml.sax.Attributes;
import org.xml.sax.helpers.AttributesImpl;

/**
 * This class implements all infos we need to describe a primary key.
 * 
 * @author wglas
 */
public class PrimaryKeyInfo {
    
    //private static Logger log = LoggerFactory.getLogger(PrimaryKeyInfo.class);

    private String name;
    private List<String> columns;
    
    /**
     * Default constructor.
     */
    public PrimaryKeyInfo()
    {}
    
    /**
     * Construct a primary key object from a name and a comma-separated list
     * of column names.
     * 
     * @param name The name of the primary key.
     * @param columns The list of columns,separated by commas.
     */
    public PrimaryKeyInfo(String name, String columns)
    {
        this.name = name;
        this.columns = JAVAHelper.splitString(columns);
    }
    
    /**
     * Initialize this object from SAX attributes.
     * 
     * @param atts A SAX attributes bag to set to this object.
     */
    public PrimaryKeyInfo(Attributes atts)
    {
        this.name = atts.getValue("name");
        if ("PRIMARY".equalsIgnoreCase(this.name)) this.name=""; // workaround a bug parsing mysql meta data // (to be sure, ignore case for postgres)
        
        String col_s = atts.getValue("columns");
        
        if (col_s == null) {
            this.columns = null;
        } else {
            this.columns = JAVAHelper.splitString(col_s);
        }
    }
    
    /**
     * @return A SAX Attributes bag filled with the information about this private key.
     */
    public Attributes toAttributes(SqlIdentifierMapper mapper)
    {
        AttributesImpl atts = new AttributesImpl();
        
        if (this.name != null)
            atts.addAttribute("","","name","CDATA",mapper.toExternal(this.name));
        
        if (this.columns != null)
            atts.addAttribute("","","columns","CDATA",
                    SQLHelper.joinIdentifiers(this.columns,mapper));
        
        return atts;
    }
    
    /**
     * @return the name of the primary key.
     */
    public String getName() {
        return this.name;
    }
    /**
     * @param name the name to set
     */
    public void setName(String name) {
        this.name = name;
    }
    /**
     * @return the columns
     */
    public List<String> getColumns() {
        return this.columns;
    }
    /**
     * @param columns the columns to set
     */
    public void setColumns(List<String> columns) {
        this.columns = columns;
    }
    
    /**
     * @param column The column name to add to the list of columns.
     */
    public void addColumn(String column) {
        if (this.columns == null)
            this.columns = new ArrayList<String>();
        
        this.columns.add(column);
    }

    @Override
    public String toString() {
        return "PrimaryKeyInfo{" +
                "name='" + name + '\'' +
                ", columns=" + columns +
                '}';
    }
}
