/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 28.11.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema;

import java.util.ArrayList;
import java.util.List;

import org.clazzes.jdbc2xml.helper.JAVAHelper;
import org.clazzes.jdbc2xml.helper.SQLHelper;
import org.clazzes.jdbc2xml.sql.SqlIdentifierMapper;
import org.xml.sax.Attributes;
import org.xml.sax.helpers.AttributesImpl;

/**
 * This class implements all infos we need to describe a database index.
 * 
 * @author wglas
 */
public class IndexInfo {
    
    public enum Order { ASC, DESC };
    
    private String name;
    private List<String> columns;
    private List<String> includeColumns;
    private String filterCondition;
    private Order order;
    private boolean unique;
    
    /**
     * Default constructor.
     */
    public IndexInfo()
    {}
    
    /**
     * Constructor using name, table and columns.
     * 
     * @param name The name of the index.
     * @param columns A comma-separated list of columns in the own table.
     * @param unique Whether this is a unique index.
     * @param filterCondition The filter condition of the index.
     */
    public IndexInfo(String name, String columns, boolean unique, String filterCondition)
    {
        this.name = name;
        this.columns = JAVAHelper.splitString(columns);
        this.unique = unique;
        this.filterCondition = filterCondition;
    }

    /**
     * Constructor using name, table and columns.
     *
     * @param name The name of the index.
     * @param columns A comma-separated list of columns in the own table.
     * @param includeColumns A comma-separated list of columns in the own table.
     * @param unique Whether this is a unique index.
     * @param filterCondition The filter condition of the index.
     */
    public IndexInfo(String name, String columns, String includeColumns, boolean unique, String filterCondition)
    {
        this.name = name;
        this.columns = JAVAHelper.splitString(columns);
        this.includeColumns = JAVAHelper.splitString(includeColumns);
        this.unique = unique;
        this.filterCondition = filterCondition;
    }
    
    /**
     * Initialize this object from SAX attributes.
     * 
     * @param atts A SAX attributes bag to set to this object.
     */
    public IndexInfo(Attributes atts)
    {
        this.name = atts.getValue("name");
        this.filterCondition = atts.getValue("filterCondition");
        
        String unique_s = atts.getValue("unique");
        
        if (unique_s == null)
            this.unique = false;
        else
            this.unique = Boolean.parseBoolean(unique_s);
        
        String order_s = atts.getValue("order");
        
        if (order_s == null)
            this.order = null;
        else
            this.order = Order.valueOf(order_s);
        
        String col_s = atts.getValue("columns");
        if (col_s == null) {
            this.columns = null;
        } else {
            this.columns = JAVAHelper.splitString(col_s);
        }

        String includeColumns_s = atts.getValue("includeColumns");
        if (includeColumns_s == null) {
            this.includeColumns = null;
        } else {
            this.includeColumns = JAVAHelper.splitString(includeColumns_s);
        }
    }
    
    /**
     * @return A SAX Attributes bag filled with the information about this private key.
     */
    public Attributes toAttributes(SqlIdentifierMapper mapper)
    {
        AttributesImpl atts = new AttributesImpl();
        
        if (this.name != null)
            atts.addAttribute("","","name","CDATA",mapper.toExternal(this.name));
        
        if (this.filterCondition != null)
            atts.addAttribute("","","filterCondition","CDATA",mapper.toExternal(this.filterCondition));
        
        if (this.order != null)
            atts.addAttribute("","","order","CDATA",this.order.toString());
        
        atts.addAttribute("","","unique","CDATA",Boolean.toString(this.unique));
        
        if (this.columns != null)
            atts.addAttribute("","","columns","CDATA",SQLHelper.joinIdentifiers(this.columns,mapper));

        if (this.includeColumns != null)
            atts.addAttribute("","","includeColumns","CDATA",SQLHelper.joinIdentifiers(this.includeColumns,mapper));
     
        return atts;
    }
    
    /**
     * @return the name of the primary key.
     */
    public String getName() {
        return this.name;
    }
    /**
     * @param name the name to set
     */
    public void setName(String name) {
        this.name = name;
    }
    /**
     * @return the columns
     */
    public List<String> getColumns() {
        return this.columns;
    }
    /**
     * @param columns the columns to set
     */
    public void setColumns(List<String> columns) {
        this.columns = columns;
    }
    
    /**
     * @param column The column name to add to the list of columns.
     */
    public void addColumn(String column) {
        if (this.columns == null)
            this.columns = new ArrayList<String>();
        
        this.columns.add(column);
    }

    /**
     * @return the columns to include extra (if dialect allows)
     */
    public List<String> getIncludeColumns() {
        return this.includeColumns;
    }
    /**
     * @param includeColumns the columns to set
     */
    public void setIncludeColumns(List<String> includeColumns) {
        this.includeColumns = includeColumns;
    }

    /**
     * @param includeColumn The column name to add to the list of columns.
     */
    public void addIncludeColumn(String includeColumn) {
        if (this.includeColumns == null)
            this.includeColumns = new ArrayList<String>();

        this.includeColumns.add(includeColumn);
    }

    /**
     * @return the filter condition.
     */
    public String getFilterCondition() {
        return this.filterCondition;
    }

    /**
     * @param filterCondition the filter condition to set
     */
    public void setFilterCondition(String filterCondition) {
        this.filterCondition = filterCondition;
    }

    /**
     * @return Whether this index is a unique index.
     */
    public boolean isUnique() {
        return this.unique;
    }

    /**
     * @param unique The unique flag to set
     */
    public void setUnique(boolean unique) {
        this.unique = unique;
    }

    /**
     * @return the order
     */
    public Order getOrder() {
        return this.order;
    }

    /**
     * @param order the order to set
     */
    public void setOrder(Order order) {
        this.order = order;
    }

    @Override
    public String toString() {
        return "IndexInfo{" +
                (unique ? "UNIQUE " : "") +
                "name='" + name + '\'' +
                ", columns=" + columns +
                ", includeColumns=" + includeColumns +
                ", filterCondition='" + filterCondition + '\'' +
                ", order=" + order +
                '}';
    }
}
