/***********************************************************
 * $$Id$$
 * 
 * J2EE-Sandbox classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;
import java.util.TimeZone;

import org.clazzes.jdbc2xml.sql.SqlCommandQueue;

public interface ISchemaEngine {

    /**
     * @return The database connection.
     */
    public Connection getConnection();

    /**
     * This method initializes the database dialect too, if the
     * the dialect has not been set through {@link #setDialect(Dialect)}
     * before.
     * 
     * The dialect is retrieved through a call to
     * {@link DialectFactory#newDialect(java.sql.DatabaseMetaData)} on
     * the instance returned by {@link DialectFactory#newInstance()}.
     * 
     * @param connection The database connection to set.
     * @throws SQLException If the metadata retrieval on the database failed during
     *                      the dialect initialization.
     */
    public void setConnection(Connection connection)
            throws SQLException;

    /**
     * @return The default time zone used for transforming date values, if the
     *         database engine does not store a time zone information. 
     */
    public TimeZone getTimeZone();

    /**
     * @param timeZone The time zone to set.
     */
    public void setTimeZone(TimeZone timeZone);

    /**
     * @return The dialect used by this engine.
     */
    public Dialect getDialect();

    /**
     * Normally you need not call this method, because the
     * dialect is usually initialized by {@link #setConnection(Connection)}.
     * 
     * @param dialect The database dialect to set
     */
    public void setDialect(Dialect dialect);
    
    /**
     * @return The schema name used for retrieving schema metadata. The schema is
     *         most often null, because most databases use a single namespace for
     *         tables, if the connection is opened with a valid catalog.
     *         
     * @see DatabaseMetaData#getTables(String, String, String, String[])
     * @see DatabaseMetaData#getColumns(String, String, String, String)
     * @see DatabaseMetaData#getImportedKeys(String, String, String)
     * @see DatabaseMetaData#getExportedKeys(String, String, String)
     * @see DatabaseMetaData#getPrimaryKeys(String, String, String)
     */
    public String getSchema();

    /**
     * @param schema The schema to set
     */
    public void setSchema(String schema);

    /**
     * Retrieve the current layout of the given table in the database.
     * 
     * @param tableName The name of the table to search.
     * @param filter The filter retrieval configuration. If a null value is passed,
     *               a default filter will be used, which skips internal indices.
     * @return The complete table information.
     * @throws SQLException Upon database errors.
     * 
     * @see IndexFilter#isKeepInternalIndices()
     */
    public TableInfo fetchTableInfo(String tableName,
            IndexFilter filter) throws SQLException;

    /**
     * Return the layout of multiple table in the database.
     * 
     * @param filter If null, return all tables of the configured schema. Otherwise
     *               return only those tables, for which {@link TableFilter#processTable(String)}
     *               returns true. Additionally, a null value incurs default table filtering,
     *               which means to fetch all table infos.
     * @return The list of table informations.
     * @throws SQLException Upon database errors.
     */
    public List<TableInfo> fetchTableInfos(TableFilter filter)
            throws SQLException;

    /**
     * Create an insert statement for all column in the given table. The placeholders
     * in the insert statement are in the order of the ColumnInfo object inside the
     * TableInfo's list of column
     * 
     * @param ti The table metadata.
     * @param setAutoValues If <code>true</code>, columns that use auto increment will be set by this statement too. If <code>false</code> database should set those values.
     * @return A Prepared statement that may be used to insert data into the database.
     * @throws SQLException Upon database errors.
     */
    public PreparedStatement createInsertStatement(TableInfo ti,
            boolean setAutoValues) throws SQLException;

    /**
     * Create a table in the database, including primary key and indices (which must have been added already).
     * 
     * @param ti The table metadata.
     * @param addForeignKeys If this parameter is true, all foreign keys are generated
     *                       together with the table structure. If set to false, you may
     *                       create the foreign keys later on using {@link #createForeignKeys(TableInfo)}.
     * @throws SQLException upon database errors.
     */
    public void createTable(TableInfo ti, boolean addForeignKeys)
            throws SQLException;

    /**
     * Rename an existing table.
     * @param ti <code>TableInfo</code> describing the table to be renamed.
     * @param newTableName The new name of the table.
     * @return The updated table info, which contains the new name.
     * @throws SQLException Upon database errors.
     */
    public TableInfo renameTable(TableInfo ti, String newTableName)
            throws SQLException;

    /**
     * Create foreign keys of an existing table in the database.
     * 
     * @param ti The table metadata.
     * @throws SQLException Upon database errors.
     */
    public void createForeignKeys(TableInfo ti) throws SQLException;

    /**
     * Drop all existing foreign keys of an existing table.
     * @param ti <code>TableInfo</code> describing the table containing the foreign keys to drop.
     * @return The updated <code>TableInfo</code> without foreign keys.
     * @throws SQLException Upon database errors.
     */
    public TableInfo dropForeignKeys(final TableInfo ti)
            throws SQLException;

    /**
     * Drop an existing table.
     * @param ti <code>TableInfo</code> describing the table to be dropped.
     * @param force If set to true, the data is not backed up before dropping the
     *              table. Consequently, rollback is not supported, if
     *              <code>force</code> is set to true.
     * @throws SQLException Upon database errors.
     */
    public void dropTable(final TableInfo ti, boolean force)
            throws SQLException;

    /**
     * Drop a list of existing table.
     * @param tables A list of <code>TableInfo</code> objects describing
     *              the tables to be dropped.
     * @param force If set to true, the data is not backed up before dropping the
     *              table. Consequently, rollback is not supported, if
     *              <code>force</code> is set to true.
     * @throws SQLException Upon database errors.
     */
    public void dropTables(final List<TableInfo> tables, boolean force)
            throws SQLException;

    /**
     * Add a column to an existing table.
     * @param ti <code>TableInfo</code> describing the table to add column to.
     * @param columnInfo <code>ColumnInfo</code> holding data for the column to be added.
     * @return The updated <code>TableInfo</code>.
     * @throws SQLException Upon database errors.
     */
    public TableInfo addColumn(final TableInfo ti,
            final ColumnInfo columnInfo) throws SQLException;

    /**
     * Change an existing column of an existing table.
     * @param ti <code>TableInfo</code> describing the table containing the column to change.
     * @param columnInfo <code>ColumnInfo</code> holding new definition data for the column.
     * @return The updated <code>TableInfo</code>.
     * @throws SQLException Upon database errors.
     */
    public TableInfo modifyColumn(final TableInfo ti,
            final ColumnInfo columnInfo) throws SQLException;

    /**
     * Rename an existing column of an existing table and possibly change it's type.
     * @param ti <code>TableInfo</code> describing the table containing the column to change.
     * @param oldColumnName The original name of the column.
     * @param columnInfo <code>ColumnInfo</code> holding the new name and the
     *                   new definition data for the column.
     * @return The updated <code>TableInfo</code>.
     * @throws SQLException Upon database errors.
     */
    public TableInfo changeColumn(final TableInfo ti,
            String oldColumnName, final ColumnInfo columnInfo)
            throws SQLException;

    /**
     * Drop an existing column of an existing table.
     * @param ti <code>TableInfo</code> describing the table containing the column to drop.
     * @param columnName Name of the column to drop.
     * @param force If set to true, the data is not backed up before dropping the
     *              column. Consequently, rollback is not supported, if
     *              <code>force</code> is set to true.
     * @return The updated <code>TableInfo</code>.
     * @throws SQLException Upon database errors.
     */
    public TableInfo dropColumn(final TableInfo ti,
            final String columnName, boolean force) throws SQLException;

    /**
     * Add a foreign key to an existing table.
     * @param ti <code>TableInfo</code> describing the table to add the foreign key to.
     * @param fkInfo The <code>ForeignKeyInfo</code> to add.
     * @return The updated <code>TableInfo</code>.
     * @throws SQLException Upon database errors.
     */
    public TableInfo addForeignKey(final TableInfo ti,
            final ForeignKeyInfo fkInfo) throws SQLException;

    /** 
     * Drop an existing foreign key of an existing table.
     * @param ti <code>TableInfo</code> describing the table to drop the foreign key from.
     * @param fkName The <code>ForeignKeyInfo</code> to drop.
     * @return The updated <code>TableInfo</code>.
     * @throws SQLException Upon database errors.
     */
    public TableInfo dropForeignKey(TableInfo ti, String fkName)
            throws SQLException;

    /**
     * Add an index to an existing table.
     * @param tableinfo <code>TableInfo</code> describing the table to add the index.
     * @param indexInfo The <code>IndexInfo</code> to add.
     * @return The updated <code>TableInfo</code>.
     * @throws SQLException Upon database errors.
     */
    public TableInfo addIndex(final TableInfo tableinfo,
            final IndexInfo indexInfo) throws SQLException;

    /**
     * Drop an index from an existing table.
     * @param tableInfo <code>TableInfo</code> describing the table to drop the index from.
     * @param indexName The name of the <code>IndexInfo</code> to drop.
     * @return The updated <code>TableInfo</code>.
     * @throws SQLException Upon database errors.
     */
    public TableInfo dropIndex(TableInfo tableInfo,
            final String indexName) throws SQLException;

    /**
     * Some operations like drop table or drop column create backup tables for
     * performing rollback operations. If a rollback of such an operation fails or
     * a user does not {@link #commit()} after such operations, these backup tables
     * remain in the target database.
     * 
     * This call searches these tables, which are named like <code>JDBC2XML__*</code>
     * and drops them, if necessary.
     * 
     * @throws SQLException
     */
    public void dropStaleBackupTables() throws SQLException;

    /**
     * Perform all pending schema modifications and discard the rollback
     * history.
     * 
     * @throws SQLException Upon database errors.
     */
    public void commit() throws SQLException;

    /**
     * Roll back all schema operations undertaken since the last call
     * to {@link #commit()} or {@link #rollback()}.
     * 
     * @throws SQLException Upon database errors or when a rollback
     *                      of a schema operation is not supported.
     *                      
     */
    public void rollback() throws SQLException;

    /**
     * @return A list of temporary tables created by not yet committed
     *         schema operations.
     */
    public List<String> getTempTableNames();

    /**
     * @return The maximal length of table names as returned by the database
     *         engine using {@link Connection#getMetaData()}
     */
    public int getMaxTableNameLength();

    public SqlCommandQueue getQueue();

}
