/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 28.11.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema;

import java.sql.DatabaseMetaData;
import java.util.ArrayList;
import java.util.List;

import org.clazzes.jdbc2xml.helper.JAVAHelper;
import org.clazzes.jdbc2xml.helper.SQLHelper;
import org.clazzes.jdbc2xml.helper.TypesHelper;
import org.clazzes.jdbc2xml.sql.SqlIdentifierMapper;
import org.xml.sax.Attributes;
import org.xml.sax.helpers.AttributesImpl;

/**
 * This class implements all infos we need to describe a foreign key.
 * 
 * @author wglas
 */
public class ForeignKeyInfo {
    
    private String name;
    private List<String> columns;
    private String pkName;
    private String foreignTable;
    private List<String> foreignColumns;
    private short updateRule;
    private short deleteRule;
    private short deferrability;
    
    /**
     * Default constructor.
     */
    public ForeignKeyInfo()
    {
        this.deferrability = DatabaseMetaData.importedKeyNotDeferrable;
        this.deleteRule = DatabaseMetaData.importedKeyNoAction;
        this.updateRule = DatabaseMetaData.importedKeyNoAction;
    }
    
    /**
     * Constructor using name, table and columns.
     * 
     * @param name The name of the foreign key.
     * @param columns A comma-separated list of columns in the own table.
     * @param foreignTable The name of the referenced foreign table.
     * @param foreignColumns The referenced columns of the foreign table (comma-separated).
     */
    public ForeignKeyInfo(String name, String columns, String foreignTable, String foreignColumns)
    {
        this.name = name;
        this.columns = JAVAHelper.splitString(columns);
        this.foreignTable = foreignTable;
        this.foreignColumns = JAVAHelper.splitString(foreignColumns);
        
        this.deferrability = DatabaseMetaData.importedKeyNotDeferrable;
        this.deleteRule = DatabaseMetaData.importedKeyNoAction;
        this.updateRule = DatabaseMetaData.importedKeyNoAction;
    }
    
    /**
     * Initialize this object from SAX attributes.
     * 
     * @param atts A SAX attributes bag to set to this object.
     */
    public ForeignKeyInfo(Attributes atts)
    {
        this.name = atts.getValue("name");
        this.pkName = atts.getValue("pkName");
        this.foreignTable = atts.getValue("foreignTable");
        this.deleteRule = TypesHelper.fkRuleStringToShort(atts.getValue("deleteRule"));
        this.updateRule = TypesHelper.fkRuleStringToShort(atts.getValue("updateRule"));
        this.deferrability = TypesHelper.deferrabilityStringToShort(atts.getValue("deferrability"));
        
        String col_s = atts.getValue("columns");
        
        if (col_s == null) {
            this.columns = null;
        } else {
            this.columns = JAVAHelper.splitString(col_s);
        }
        
        col_s = atts.getValue("foreignColumns");
        
        if (col_s == null) {
            this.foreignColumns = null;
        } else {
            this.foreignColumns = JAVAHelper.splitString(col_s);
        }
    }
    
    /**
     * @return A SAX Attributes bag filled with the information about this private key.
     */
    public Attributes toAttributes(SqlIdentifierMapper mapper)
    {
        AttributesImpl atts = new AttributesImpl();
        
        if (this.name != null)
            atts.addAttribute("","","name","CDATA",mapper.toExternal(this.name));
        
        if (this.pkName != null)
            atts.addAttribute("","","pkName","CDATA",mapper.toExternal(this.pkName));
        
        if (this.foreignTable != null)
            atts.addAttribute("","","foreignTable","CDATA",mapper.toExternal(this.foreignTable));
        
        atts.addAttribute("","","deleteRule","CDATA",TypesHelper.fkRuleShortToString(this.deleteRule));
        atts.addAttribute("","","updateRule","CDATA",TypesHelper.fkRuleShortToString(this.updateRule));
        atts.addAttribute("","","deferrability","CDATA",TypesHelper.deferrabilityShortToString(this.deferrability));
        
        if (this.columns != null)
            atts.addAttribute("","","columns","CDATA",
                    SQLHelper.joinIdentifiers(this.columns,mapper));
     
        if (this.foreignColumns != null)
            atts.addAttribute("","","foreignColumns","CDATA",
                    SQLHelper.joinIdentifiers(this.foreignColumns,mapper));
        
        return atts;
    }
    
    /**
     * @return the name of the primary key.
     */
    public String getName() {
        return this.name;
    }
    /**
     * @param name the name to set
     */
    public void setName(String name) {
        this.name = name;
    }
    /**
     * @return the columns
     */
    public List<String> getColumns() {
        return this.columns;
    }
    /**
     * @param columns the columns to set
     */
    public void setColumns(List<String> columns) {
        this.columns = columns;
    }
    
    /**
     * @param column The column name to add to the list of columns.
     */
    public void addColumn(String column) {
        if (this.columns == null)
            this.columns = new ArrayList<String>();
        
        this.columns.add(column);
    }

    /**
     * @return the pkName
     */
    public String getPkName() {
        return this.pkName;
    }

    /**
     * @param pkName the pkName to set
     */
    public void setPkName(String pkName) {
        this.pkName = pkName;
    }

    /**
     * @return the foreignTable
     */
    public String getForeignTable() {
        return this.foreignTable;
    }

    /**
     * @param foreignTable the foreignTable to set
     */
    public void setForeignTable(String foreignTable) {
        this.foreignTable = foreignTable;
    }

    /**
     * @return the foreignColumns
     */
    public List<String> getForeignColumns() {
        return this.foreignColumns;
    }

    /**
     * @param foreignColumns the foreignColumns to set
     */
    public void setForeignColumns(List<String> foreignColumns) {
        this.foreignColumns = foreignColumns;
    }

    /**
     * @param column The column name to add to the list of foreign columns.
     */
    public void addForeignColumn(String column) {
        if (this.foreignColumns == null)
            this.foreignColumns = new ArrayList<String>();
        
        this.foreignColumns.add(column);
    }

    /**
     * @return the updateRule
     * 
     * @see DatabaseMetaData#importedKeyCascade
     * @see DatabaseMetaData#importedKeyNoAction
     * @see DatabaseMetaData#importedKeyRestrict
     * @see DatabaseMetaData#importedKeySetDefault
     * @see DatabaseMetaData#importedKeySetNull
     */
    public short getUpdateRule() {
        return this.updateRule;
    }

    /**
     * @param updateRule the updateRule to set
     */
    public void setUpdateRule(short updateRule) {
        this.updateRule = updateRule;
    }

    /**
     * @return the deleteRule
     * 
     * @see DatabaseMetaData#importedKeyCascade
     * @see DatabaseMetaData#importedKeyNoAction
     * @see DatabaseMetaData#importedKeyRestrict
     * @see DatabaseMetaData#importedKeySetDefault
     * @see DatabaseMetaData#importedKeySetNull
     */
    public short getDeleteRule() {
        return this.deleteRule;
    }

    /**
     * @param deleteRule the deleteRule to set
     */
    public void setDeleteRule(short deleteRule) {
        this.deleteRule = deleteRule;
    }

    /**
     * @return The deferrability.
     * 
     * @see DatabaseMetaData#importedKeyInitiallyDeferred
     * @see DatabaseMetaData#importedKeyNotDeferrable
     * @see DatabaseMetaData#importedKeyInitiallyImmediate
     */
    public short getDeferrability() {
        return this.deferrability;
    }

    /**
     * @param deferrability The deferrability to set.
     */
    public void setDeferrability(short deferrability) {
        this.deferrability = deferrability;
    }

    @Override
    public String toString() {
        return "ForeignKeyInfo{" +
                "pkName='" + pkName + '\'' +
                ", foreignTable='" + foreignTable + '\'' +
                ", name='" + name + '\'' +
                '}';
    }
}
