/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 27.11.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema;

import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.Map;
import java.util.Properties;

import org.clazzes.jdbc2xml.sql.SqlCommandQueue;

/**
 * SQL Dialects provide DBMS specific helpers to allow unified SQL commands.
 */
public interface Dialect {

    /**
     * Set a database specific property used for configuring the
     * schema engine.
     * 
     * Valid key names are contained as <code>*_PROPERTY</code> constants
     * in the various property dialogs.
     * 
     * @param key The key of the property.
     * @param value The value to set.
     */
    public void setProperty(String key, String value);
    
    /**
     * Set multiple database specific property used for configuring the
     * schema engine.
     * 
     * @param keyValues A map of key/values to be copied to the dialect's
     *            properties.
     */
    public void setProperties(Map<String,String> keyValues);

    /**
     * Get the value of a database specific property used for configuring the
     * schema engine.
     * 
     * @param key The key of the property.
     * @return The value of the property or <code>null</code>, if
     *         the property has not been defined.
     */
    public String getProperty(String key);
    
    /**
     * externalize id-strings to class/enum
     * @return A descriptive ID of the dialect, which contains no whitespace characters.
     */
    public String getID();
    
    /**
     * Quote the given string to the string buffer. This is for occasions, where
     * you want to build an insert statement without a placeholder in a
     * PreparedStatement.
     * 
     * However, using a PreparedStatemnt is highly recommended.
     * 
     * @param sb The buffer to write to.
     * @param s the String to quote.
     */
    public void quoteString(StringBuffer sb, String s);
    
    /**
     * Normalize a default value returned by
     * {@link DatabaseMetaData#getColumns(String, String, String, String)}
     * in the column <code>COLUMN_DEF</code>.
     * 
     * Some database engines return the default value in a quoted manner, so
     * we must perform unquoting here.
     * 
     * @param type The java.sql.Types constant of the column type.
     * @param s The string to normalize.
     * @return The normalized default value without amy quoting.
     */
    public String normalizeDefaultValue(int type, String s);
    
    /**
     * Computes a string to feed a CREATE TABLE or ALTER TABLE ADD COLUMN
     * statement, or returns null if the type is unsupported by the underlying
     * DBMS or by the Dialect implementation. This string includes the column name,
     * the database specific column type and possible NOT NULL clauses.
     * 
     * @param columnInfo
     * @return a string to feed a CREATE TABLE or ALTER TABLE ADD COLUMN
     *         statement, or null
     */
    public String createColumnSpec(ColumnInfo columnInfo);

    /**
     * Push the create table command to a command queue.
     * 
     * @param schemaEngine The SchemaEngine.
     * @param queue The queue to push to.
     * @param ti The table info for which to push the create table command.
     */
    public void pushCreateTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti) throws SQLException;
    
    /**
     * Push a rename table command to a command queue.
     * 
     * @param schemaEngine The SchemaEngine.
     * @param queue The queue to push to.
     * @param ti The table info for which to push the rename table command.
     * @param newTableName The new name of the table after the rename operation.b
     */
    public void pushRenameTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti, String newTableName) throws SQLException;
    
    /**
     * Push the drop table command to a command queue.
     * 
     * @param schemaEngine The SchemaEngine.
     * @param queue The queue to push to.
     * @param ti The table info for which to push the drop table command.
     * @param force If set to true, the data is not backed up before dropping the
     *              table. Consequently, rollback is not supported, if
     *              <code>force</code> is set to true.
     */
    public void pushDropTable(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti, boolean force) throws SQLException;
    
    /**
     * Push the add foreign key command to a command queue.
     * 
     * @param queue The queue to push to.
     * @param fki The foreign key description.
     * @param ti The table info for which to push the add foreign key command.
     */
    public void pushAddForeignKey(SqlCommandQueue queue, TableInfo ti, ForeignKeyInfo fki) throws SQLException;
    
    /**
     * Push the drop foreign key command to a command queue.
     * 
     * @param queue The queue to push to.
     * @param fki The foreign key description.
     * @param ti The table info for which to push the drop foreign key command.
     */
    public void pushDropForeignKey(SqlCommandQueue queue, TableInfo ti, ForeignKeyInfo fki) throws SQLException;
    
    /**
     * Push the add index command to a command queue.
     * 
     * @param queue The queue to push to.
     * @param indexInfo The index description.
     * @param ti The table info for which to push the add index command.
     */
    public void pushAddIndex(SqlCommandQueue queue, TableInfo ti, IndexInfo indexInfo) throws SQLException;
    
    /**
     * Push the drop index command to a command queue.
     * 
     * @param queue The queue to push to.
     * @param indexInfo The index description.
     * @param ti The table info for which to push the drop index command.
     */
    public void pushDropIndex(SqlCommandQueue queue, TableInfo ti, IndexInfo indexInfo) throws SQLException;
    
    /**
     * Push the add column command to a command queue.
     * 
     * @param queue The queue to push to.
     * @param ci The column description.
     * @param ti The table info for which to push the add column command.
     */
    public void pushAddColumn(SqlCommandQueue queue, TableInfo ti, ColumnInfo ci) throws SQLException;
    
    /**
     * Push the drop column command to a command queue.
     * 
     * @param schemaEngine The SchemaEngine.
     * @param queue The queue to push to.
     * @param ti The table info for which to push the add column command.
     * @param ci The column description.
     * @param force f set to true, the data is not backed up before dropping the
     *              column. Consequently, rollback is not supported, if
     *              <code>force</code> is set to true.
     */
    public void pushDropColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti, ColumnInfo ci, boolean force) throws SQLException;
    
    /**
     * Modify the datatype of a database column.
     * The calling engine asserts, that the old column name is identical to the new column name.
     * 
     * @param schemaEngine The SchemaEngine.
     * @param queue The queue to push to.
     * @param ti The description of the database table.
     * @param oldColumnInfo The column info as represented by the current schema.
     * @param newColumnInfo The column information carrying the column name and the new datatype.
     */
    public void pushModifyColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo) throws SQLException;
    
    /**
     * Rename and change the datatype of a database column.
     * The calling engine asserts, that the old column name is distinct from the new column name.
     * 
     * @param schemaEngine The SchemaEngine.
     * @param queue The queue to push to.
     * @param ti The description of the database table.
     * @param oldColumnInfo The original column info of the column to be changed.
     * @param newColumnInfo The destination column info carrying the new column name and the new datatype.
     */
    public void pushChangeColumn(ISchemaEngine schemaEngine, SqlCommandQueue queue, TableInfo ti,
            ColumnInfo oldColumnInfo, ColumnInfo newColumnInfo) throws SQLException;
    
    /**
     * @return the default driver name for this dialect
     */
    public String defaultDriverName();
    
    /**
     * Constructs a JDBC URL for the dialect; all parameters may be null, default values are used then.
     * @param hostname     Name of the host to connect to, use null get the default value "localhost" for Client/Server resp. non-net URLs for integrated engines.
     * @param port         IP port to connect to, use null to use the dialect specific default port for Client/Server resp. non-net URLs for integrated engines.
     * @param databaseName name of the database to connect to, use null to connect w/o a certain database
     * @param properties   Properties to add to the URL, usually for tweaking the JDBC driver, may be null.
     * @return the complete JDBC URL.
     */
    public String constructJDBCURL(String hostname, Integer port, String databaseName, Properties properties);
    
    
    /**
     * Get the JDBC SQL type for a specified dialect specific type. 
     * @param dialectDataType String representation of dialect specific type.
     * @return the mapped type as defined in {@link java.sql.Types}
     */
    public int getMappedSqlType(String dialectDataType);

    /**
     * Hook to fetch additional, Dialect specific column information (e.g. auto increment).
     * @param schemaEngine The calling {@link SchemaEngine}
     * @param ti The {@link TableInfo}
     * @param ci The {@link ColumnInfo}
     */
    public void fetchAdditionalColumnInfo(ISchemaEngine schemaEngine, TableInfo ti, ColumnInfo ci);
}
