/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 27.11.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.schema;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Types;

import org.clazzes.jdbc2xml.Constants;
import org.clazzes.jdbc2xml.helper.TypesHelper;
import org.clazzes.jdbc2xml.sql.SqlIdentifierMapper;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

/**
 * This class holds metadata of a database column.
 * 
 * @author wglas
 */
public class ColumnInfo {
    
    private String name;
    private int type;
    private Integer precision;
    private Integer scale;
    private boolean nullable;
    private String defaultValue;
    private boolean autoIncrement;
    private Integer nextValue;
    
    /**
     * @param name The column name.
     * @param type The Type of the column.
     * @param precision The field with or precision of the column.
     * @param scale The number of fractional digits.
     * @param nullable Whether this column allows null values.
     * @param defaultValue The default column value.
     * @param autoIncrement Whether this column is AUTO_INCREMENT
     */
    public ColumnInfo(String name, int type, Integer precision, Integer scale,
            boolean nullable, String defaultValue, boolean autoIncrement) {
        this(name, type, precision, scale, nullable, defaultValue);
        this.autoIncrement = autoIncrement;
    }
    
    /**
     * @param name The column name.
     * @param type The Type of the column.
     * @param precision The field with or precision of the column.
     * @param scale The number of fractional digits.
     * @param nullable Whether this column allows null values.
     * @param defaultValue The default column value.
     */
    public ColumnInfo(String name, int type, Integer precision, Integer scale,
            boolean nullable, String defaultValue) {
        super();
        this.name = name;
        this.type = type;
        this.precision = precision;
        this.scale = scale;
        this.nullable = nullable;
        this.defaultValue = defaultValue;
        this.autoIncrement = false;
    }

    /**
     * @param atts A bag of XML attributes.
     * @throws SAXException If the attributes do not contain a name attribute.
     */
    public ColumnInfo(Attributes atts) throws SAXException
    {
        super();
        this.name = atts.getValue(Constants.COLUMN_TAG_NAME_ATT);
        
        if (this.name == null)
            throw new SAXException("Column without a name attribute found.");
        
        String type_s = atts.getValue(Constants.COLUMN_TAG_TYPE_ATT);
        this.type = type_s == null ? java.sql.Types.VARCHAR : TypesHelper.stringToType(type_s);
        
        String prec_s = atts.getValue(Constants.COLUMN_TAG_PREC_ATT);
        this.precision = prec_s == null ? null : Integer.valueOf(prec_s);
        
        String scale_s = atts.getValue(Constants.COLUMN_TAG_SCALE_ATT);
        this.scale = scale_s == null ? null : Integer.valueOf(scale_s);
        
        String nullable_s = atts.getValue(Constants.COLUMN_TAG_NULLABLE_ATT);
        this.nullable = nullable_s == null ? true : Boolean.parseBoolean(nullable_s);
        
        String defValue = atts.getValue(Constants.COLUMN_TAG_DEFAULT_ATT);
        this.defaultValue = ((defValue == null) ? null : defValue.replace("\'\'", "\'"));
        
        String autoIncrement_s = atts.getValue(Constants.COLUMN_TAG_AUTOINCREMENT_ATT);
        this.autoIncrement = autoIncrement_s == null ? false : Boolean.parseBoolean(autoIncrement_s);
        
        String nextValue_s = atts.getValue(Constants.COLUMN_TAG_NEXTVALUE_ATT);
        this.nextValue = nextValue_s == null ? null : Integer.valueOf(nextValue_s);
    }
    
    /**
     * Set the column information from a column of a result set.
     * 
     * @param md Result set metadata information received trough {@link ResultSet#getMetaData()}.
     * @param column The column number of the result set to inspect.
     * @throws SQLException Upon Database errors.
     */
    public ColumnInfo(ResultSetMetaData md, int column) throws SQLException
    {
        super();
        this.name = md.getColumnName(column);
        this.type = md.getColumnType(column);
        
        int prec = md.getPrecision(column);
        
        if (prec > 0 && this.type != Types.DOUBLE && this.type != Types.FLOAT)
            this.precision = prec;
        else
            this.precision = null;
        
        if (    this.type == Types.NUMERIC ||
                this.type == Types.DECIMAL ||
                this.type == Types.REAL      )
            this.scale = md.getScale(column);
        else
            this.scale = null;
        
        this.nullable = md.isNullable(column) != ResultSetMetaData.columnNoNulls;
        this.defaultValue = null;
        this.autoIncrement = md.isAutoIncrement(column);
    }
    
    /**
     * @return A SAX Attributes bag filled with the information about this column.
     */
    public Attributes toAttributes(SqlIdentifierMapper mapper)
    {
        AttributesImpl atts = new AttributesImpl();
        atts.addAttribute("","",Constants.COLUMN_TAG_NAME_ATT,"CDATA",
                mapper.toExternal(this.getName()));
        atts.addAttribute("","",Constants.COLUMN_TAG_TYPE_ATT,"CDATA",TypesHelper.typeToString(this.getType()));
    
        if (this.getPrecision() != null)
            atts.addAttribute("","",Constants.COLUMN_TAG_PREC_ATT,"CDATA",this.getPrecision().toString());

        if (this.getScale() != null)
            atts.addAttribute("","",Constants.COLUMN_TAG_SCALE_ATT,"CDATA",this.getScale().toString());
    
        atts.addAttribute("","",Constants.COLUMN_TAG_NULLABLE_ATT,"CDATA",Boolean.toString(this.isNullable()));
        
        if (this.getDefaultValue() != null)
            atts.addAttribute("","",Constants.COLUMN_TAG_DEFAULT_ATT,"CDATA",this.getDefaultValue());
  
        if (this.isAutoIncrement())
            atts.addAttribute("", "", Constants.COLUMN_TAG_AUTOINCREMENT_ATT, "CDATA", Boolean.toString(this.isAutoIncrement()));
        
        if (this.isAutoIncrement() && this.nextValue != null)
            atts.addAttribute("", "", Constants.COLUMN_TAG_NEXTVALUE_ATT, "CDATA", this.getNextValue().toString());
            
        return atts;
    }
        
    /**
     * @return the defaultValue
     */
    public String getDefaultValue() {
        return this.defaultValue;
    }

    /**
     * @param defaultValue the defaultValue to set
     */
    public void setDefaultValue(String defaultValue) {
        this.defaultValue = defaultValue;
    }

    /**
     * @param name the name to set
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * @param type the type to set
     */
    public void setType(int type) {
        this.type = type;
    }

    /**
     * @param precision the precision to set
     */
    public void setPrecision(Integer precision) {
        this.precision = precision;
    }

    /**
     * @param scale the scale to set
     */
    public void setScale(Integer scale) {
        this.scale = scale;
    }

    /**
     * @param nullable the nullable to set
     */
    public void setNullable(boolean nullable) {
        this.nullable = nullable;
    }

    /**
     * @param autoIncrement the autoIncrement to set
     */
    public void setAutoIncrement(boolean autoIncrement) {
        this.autoIncrement = autoIncrement;
    }
    
    /**
     * 
     * @param nextValue The next to use value on an insert for the AUTO_INCREMENT column 
     */
    public void setNextValue(int nextValue){
        this.nextValue = nextValue;
    }

    /**
     * @return the name
     */
    public String getName() {
        return this.name;
    }

    /**
     * @return The SQL type.
     * @see Types
     */
    public int getType() {
        return this.type;
    }

    /**
     * @return The optional field precision.
     */
    public Integer getPrecision() {
        return this.precision;
    }

    /**
     * @return The optional field scale (the precision after the comma).
     */
    public Integer getScale() {
        return this.scale;
    }

    /**
     * @return Whether the column is nullable.
     */
    public boolean isNullable() {
        return this.nullable;
    }

    /**
     * @return the autoIncrement
     */
    public boolean isAutoIncrement() {
        return this.autoIncrement;
    }
    
    /**
     * 
     * @return The next to use value on an insert for the AUTO_INCREMENT column 
     */
    public Integer getNextValue() {
       return this.nextValue;
    }

    @Override
    public String toString() {
        return "ColumnInfo{" +
                "name='" + this.getName() + '\'' +
                ", type=" + this.getType() + "(" + this.getPrecision() + ")" + (this.isNullable() ? " NULLABLE" : "") +
                (this.isAutoIncrement() ? " AUTO_INC" : "") +
                '}';
    }
}
