/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Created: 13.10.2009
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.sax.impl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import org.clazzes.jdbc2xml.Constants;
import org.clazzes.jdbc2xml.sax.XmlPretty;
import org.clazzes.jdbc2xml.sax.XmlPrettyConfig;
import org.clazzes.util.xml.XMLSerializerHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

/**
 * {@link ContentHandler} to produce XML-Dump in a zipped format:
 * <ul>
 * <li>content.xml</li>
 * <li>tablename1_data.xml</li>
 * <li>tablename2_data.xml</li>
 * <li>...</li>
 * </ul>
 * @author rbreuss
 *
 */
public class ZipFileContentHandler implements ContentHandler {

    private static final Logger log = LoggerFactory.getLogger(ZipFileContentHandler.class);

    private final String mainEntryName;
    private final ByteArrayOutputStream bos;
    private final Deque<ContentHandler> serializer;
    private ZipOutputStream os;
    private String currentTableName;
    private XmlPrettyConfig xmlPrettyConfig;
    
    /**
     * Create a new instance.
     * @param os The {@link ZipOutputStream} to write to.<br> This stream has to be closed by the caller.
     * @param mainEntryName Name of the {@link ZipEntry} containing the DB-Table structure. 
     * @param prettyConfig Used to format output.
     */
    public ZipFileContentHandler(ZipOutputStream os, String mainEntryName, XmlPrettyConfig prettyConfig) {
        this.os = os;
        this.mainEntryName = mainEntryName;
        this.xmlPrettyConfig = prettyConfig;
        this.bos = new ByteArrayOutputStream();
        this.serializer = new ArrayDeque<ContentHandler>();
        ContentHandler contentHandler = XMLSerializerHelper.newSerializer(this.bos);
        XmlPretty xmlPretty =  new XmlPretty(contentHandler, this.xmlPrettyConfig);
        this.serializer.push(xmlPretty);
       
        if (log.isDebugEnabled())
            log.debug("ZipFileContentHandler instantiated: mainEntryName=[" + mainEntryName + "].");
    }
    
    private ContentHandler getContentHandler()
    {
        // current ContentHandler is the head.
        return this.serializer.peek();
    }
    
    /**
     * @param ch
     * @param start
     * @param length
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#characters(char[], int, int)
     */
    public void characters(char[] ch, int start, int length)
            throws SAXException {
        getContentHandler().characters(ch, start, length);
    }
    /**
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#endDocument()
     */
    public void endDocument() throws SAXException {
        if (log.isDebugEnabled())
            log.debug("endDocument: started ...");
        getContentHandler().endDocument();
        if (this.serializer.size() == 1)
        {
            log.info("end of main Document detected, writing ZIP-Entry: [" + this.mainEntryName + "] ...");
            try
            {
                this.os.putNextEntry(new ZipEntry(this.mainEntryName));
                this.os.write(this.bos.toByteArray());
                this.bos.close();
                this.os.closeEntry();
                this.os.finish();
            }
            catch (IOException e)
            {
                final String msg = "Error writing ZIP-File-Entry [" + this.mainEntryName + "].";
                log.error(msg, e);
                throw new SAXException(msg, e);
            }
        }
        if (log.isDebugEnabled())
            log.debug("endDocument: finished.");
    }
    /**
     * @param uri
     * @param localName
     * @param qName
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#endElement(java.lang.String, java.lang.String, java.lang.String)
     */
    public void endElement(String uri, String localName, String qName)
            throws SAXException {
        
        if (Constants.ROWSET_TAG_NAME.equalsIgnoreCase(qName))
        {
            getContentHandler().endElement(uri, localName, qName);
            getContentHandler().endPrefixMapping("");
            getContentHandler().endDocument();
            try 
            {
                this.os.closeEntry();
                log.info("ZIP-File-Entry closed.");
                this.serializer.poll();
                getContentHandler().endElement(Constants.W3_XINCLUDE_NS_URI, Constants.INCLUDE_TAG_NAME, Constants.INCLUDE_TAG_NAME);
                return;
            } 
            catch (IOException e) {
                final String msg = "Error closing ZIP-File-Entry.";
                log.error(msg, e);
                throw new SAXException(msg, e);
            }
        }
        getContentHandler().endElement(uri, localName, qName);
    }
    /**
     * @param prefix
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#endPrefixMapping(java.lang.String)
     */
    public void endPrefixMapping(String prefix) throws SAXException {
        getContentHandler().endPrefixMapping(prefix);
    }
    /**
     * @param ch
     * @param start
     * @param length
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#ignorableWhitespace(char[], int, int)
     */
    public void ignorableWhitespace(char[] ch, int start, int length)
            throws SAXException {
        getContentHandler().ignorableWhitespace(ch, start, length);
    }
    /**
     * @param target
     * @param data
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#processingInstruction(java.lang.String, java.lang.String)
     */
    public void processingInstruction(String target, String data)
            throws SAXException {
        getContentHandler().processingInstruction(target, data);
    }
    /**
     * @param locator
     * @see org.xml.sax.ContentHandler#setDocumentLocator(org.xml.sax.Locator)
     */
    public void setDocumentLocator(Locator locator) {
        getContentHandler().setDocumentLocator(locator);
    }
    /**
     * @param name
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#skippedEntity(java.lang.String)
     */
    public void skippedEntity(String name) throws SAXException {
        getContentHandler().skippedEntity(name);
    }
    /**
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#startDocument()
     */
    public void startDocument() throws SAXException {
        log.info("startDocument");
        if (this.os == null)
        {
            final String msg = "Error startDocument: ZipOutputStream is null.";
            log.error(msg);
            throw new SAXException(msg);
        }
        getContentHandler().startDocument();
    }
    /**
     * @param uri
     * @param localName
     * @param qName
     * @param atts
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    public void startElement(String uri, String localName, String qName,
            Attributes atts) throws SAXException {
        
        if (Constants.TABLE_TAG_NAME.equalsIgnoreCase(qName))
        {
            this.currentTableName = atts.getValue(Constants.TABLE_TAG_NAME_ATT);
            if (log.isDebugEnabled())
                log.debug("processing table: [" + this.currentTableName + "] ...");
        }
        else if (Constants.ROWSET_TAG_NAME.equalsIgnoreCase(qName))
        {
            ZipEntry entry = new ZipEntry(this.currentTableName + "_data.xml");
            log.info("Start new ZIP-File-Entry [" + entry.getName() + "] ...");
            AttributesImpl a = new AttributesImpl();
            
            a.addAttribute("", "", Constants.INCLUDE_REFERENCE_ATT, "CDATA", entry.getName());
            getContentHandler().startElement(Constants.W3_XINCLUDE_NS_URI,Constants.INCLUDE_TAG_NAME,Constants.INCLUDE_TAG_NAME,a);
            
            try 
            {
                this.os.putNextEntry(entry);

                ContentHandler contentHandler = XMLSerializerHelper.newSerializer(this.os);
                XmlPretty xmlPretty =  new XmlPretty(contentHandler, this.xmlPrettyConfig);
                this.serializer.push(xmlPretty);
                
                getContentHandler().startDocument();
                getContentHandler().startPrefixMapping("",Constants.JDBC2XML_NS_URI);
                getContentHandler().startElement(uri, localName, qName, atts);
                return;
            } 
            catch (IOException e) 
            {
                final String msg = "Error adding new Entry [" + entry.getName() + "] to ZIP-File.";
                log.error(msg, e);
                throw new SAXException(msg, e);
            }
        }

        getContentHandler().startElement(uri, localName, qName, atts);
    }
    /**
     * @param prefix
     * @param uri
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#startPrefixMapping(java.lang.String, java.lang.String)
     */
    public void startPrefixMapping(String prefix, String uri)
            throws SAXException {
        getContentHandler().startPrefixMapping(prefix, uri);
    }
    
}
