/***********************************************************
 * $Id$
 *
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 08.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.jdbc2xml.sax.impl;

import java.sql.SQLException;

import org.clazzes.jdbc2xml.Constants;
import org.clazzes.jdbc2xml.helper.JAVAHelper;
import org.clazzes.jdbc2xml.schema.ForeignKeyInfo;
import org.clazzes.jdbc2xml.schema.IndexInfo;
import org.clazzes.jdbc2xml.schema.TableInfo;
import org.clazzes.util.xml.SAXTagHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * A handler for the xmldump root element.
 *
 * @author wglas
 */
public class XmlDumpTagHandler implements SAXTagHandler {

    private static final Logger log = LoggerFactory.getLogger(XmlDumpTagHandler.class);

    private InsertContext context;

    /**
     * @param context The insert context holding database connection et al.
     */
    public XmlDumpTagHandler(InsertContext context) {
        super();
        this.context = context;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.xml.SAXTagHandler#closeTag(java.lang.String, java.lang.String, java.lang.String)
     */
    public void closeTag(String uri, String localName, String name)
    throws SAXException {

        if (this.context.getProcessRestrictionFilter().isProcessConstraints() &&
                this.context.getParsedTables() != null) {
            for (TableInfo tableInfo : this.context.getParsedTables()) {

                if (tableInfo.getForeignKeys() == null ||
                        this.context.getParsedTables().size()<=0)
                    continue;

                try {
                    log.info("Adding "+tableInfo.getForeignKeys().size()+" foreign keys to table ["+tableInfo.getName()+"].");

                    if (this.context.getProcessRestrictionFilter().isCreateFKIndices()) {
                        for (ForeignKeyInfo fki : tableInfo.getForeignKeys()) {

                            String iname;

                            if (fki.getName() == null) {
                                StringBuffer sb = new StringBuffer(tableInfo.getName());

                                for (String c : fki.getColumns()) {
                                    sb.append('_');
                                    sb.append(c);
                                }
                                sb.append("__FKIDX");
                                iname = sb.toString();
                            }
                            else {
                                iname = fki.getName() + "__IDX";
                            }

                            IndexInfo indexInfo =
                                new IndexInfo(iname,JAVAHelper.joinStrings(fki.getColumns()),null,
                                        false,null);

                            log.info("Adding extra foreign key index ["+iname+"].");
                            this.context.getSchemaEngine().addIndex(tableInfo,indexInfo);
                        }
                    }

                    this.context.getSchemaEngine().createForeignKeys(tableInfo);
                    log.info("Finished adding "+tableInfo.getForeignKeys().size()+" foreign keys to table ["+tableInfo.getName()+"].");
                } catch (SQLException e) {
                    throw new SAXException("SQL error adding foregin keys to table ["+tableInfo.getName()+"].",e);
                }
            }
        }
        this.context.setParsedTables(null);
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.xml.SAXTagHandler#processCharacters(char[], int, int)
     */
    public void processCharacters(char[] ch, int start, int length)
            throws SAXException {
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.xml.SAXTagHandler#startChildElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    public SAXTagHandler startChildElement(String uri, String localName,
            String name, Attributes atts) throws SAXException {

        if (!Constants.JDBC2XML_NS_URI.equals(uri))
            throw new SAXException("Namspace URI ["+uri+"] is not supported.");

        if (Constants.TABLE_TAG_NAME.equals(localName)) {

            String tableName = atts.getValue(Constants.TABLE_TAG_NAME_ATT);

            boolean processCurrentTable =
                this.context.getProcessRestrictionFilter().processTable(tableName);

            if (processCurrentTable)
                return new TableTagHandler(this.context,atts);
            else {
                log.info("Skipping table ["+tableName+"].");
                return null;
            }
        }
        else if (Constants.QUERY_TAG_NAME.equals(localName)) {

            log.warn("Ignoring data of dumped query ["+atts.getValue(Constants.QUERY_TAG_SQL_ATT)+"].");
            return null;
        }
        else
            throw new SAXException("xmldump child element ["+localName+"] is not supported.");
    }

}
