/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 08.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.sax.impl;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.TimeZone;

import org.clazzes.jdbc2xml.deserialization.DeserializationHandler;
import org.clazzes.jdbc2xml.deserialization.DeserializationHandlerFactory;
import org.clazzes.jdbc2xml.deserialization.IDeserializationHandlerFactory;
import org.clazzes.jdbc2xml.schema.ColumnInfo;
import org.clazzes.util.xml.SAXTagHandler;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * A handler for the columnset element.
 * 
 * @author wglas
 */
public class ValueTagHandler implements SAXTagHandler {

    private static final IDeserializationHandlerFactory deserializationHandlerFactory =
        DeserializationHandlerFactory.newInstance();
    
    private PreparedStatement insert;
    private DeserializationHandler deserializationHandler;
    private int column;
    private boolean tagStarted;
    
    /**
     * @param ci the column information on which to operate.
     * @param timeZone The default time zone for
     *         {@link DeserializationHandlerFactory#newDeserializationHandler(ColumnInfo, TimeZone)}.
     * @param insert The prepared statement, whose place holder is filled.
     * @param column The number of the placeholder in the prepared statement to fill.
     * @throws SAXException Upon errors during instantiation of a
     *                      {@link DeserializationHandler} instance.
     */
    public ValueTagHandler(ColumnInfo ci, TimeZone timeZone,
                           PreparedStatement insert, int column) throws SAXException {
        super();
        this.deserializationHandler = deserializationHandlerFactory.newDeserializationHandler(ci, timeZone);
        this.insert = insert;
        this.column = column;
        this.tagStarted = false;
    }

    /**
     * Start processing a new value tag.
     * 
     * @throws SAXException
     */
    public void startTag() throws SAXException {
        this.deserializationHandler.startCharacters();
        this.tagStarted = true;
    }
    
    /**
     * Close processing a value tag. This incurs setting the column value
     * of the prepared statement to null, if the tag has not yet been opened
     * by {@link #startTag()}.
     * 
     * @throws SAXException
     */
    public void finishTag() throws SAXException {
        
        if (this.tagStarted) {
            this.tagStarted = false;
        }
        else
            try {
                this.deserializationHandler.fillColumn(this.insert,this.column);
            } catch (SQLException e) {
                throw new SAXException("SQL error setting null value to column ["+this.column+"].",e);
            }
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.util.xml.SAXTagHandler#closeTag(java.lang.String, java.lang.String, java.lang.String)
     */
    public void closeTag(String uri, String localName, String name)
            throws SAXException {
        
        try {
            this.deserializationHandler.fillColumn(this.insert,this.column);
        } catch (SQLException e) {
            throw new SAXException("SQL error setting value to column ["+this.column+"].",e);
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.xml.SAXTagHandler#processCharacters(char[], int, int)
     */
    public void processCharacters(char[] ch, int start, int length)
            throws SAXException {
        this.deserializationHandler.addCharacters(ch,start,length);
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.xml.SAXTagHandler#startChildElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    public SAXTagHandler startChildElement(String uri, String localName,
            String name, Attributes atts) throws SAXException {
      
        throw new SAXException("value does not accept child elements.");
    }

}
