/***********************************************************
 * $Id$
 *
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 08.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.jdbc2xml.sax.impl;

import java.sql.SQLException;

import org.clazzes.jdbc2xml.Constants;
import org.clazzes.jdbc2xml.schema.PrimaryKeyInfo;
import org.clazzes.jdbc2xml.schema.TableInfo;
import org.clazzes.util.xml.SAXTagHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * A handler for the table element.
 *
 * @author wglas
 */
public class TableTagHandler implements SAXTagHandler {

    private static final Logger log = LoggerFactory.getLogger(TableTagHandler.class);

    private InsertContext context;
    private TableInfo tableInfo;
    private boolean tableCreated;

    private void createTable() throws SAXException
    {
        if (this.context.getProcessRestrictionFilter().isProcessSchema()) {

            try {
                this.context.getSchemaEngine().createTable(this.tableInfo,false);
            } catch (SQLException e) {
                throw new SAXException("SQL error creating table ["+this.tableInfo.getName()+"].",e);
            }

            this.context.addParsedTable(this.tableInfo);
        }
        else if (this.context.getProcessRestrictionFilter().isProcessConstraints()) {
            log.info("Adding table ["+this.tableInfo.getName()+"] for later constraint processing without creating it.");
            this.context.addParsedTable(this.tableInfo);
        }
        else {
            log.info("Not creating table ["+this.tableInfo.getName()+"] because schema processing is turned off.");
        }

        this.tableCreated = true;
    }

    /**
     * @param context The insert context holding database connection et al.
     */
    public TableTagHandler(InsertContext context, Attributes atts) {
        super();
        this.context = context;
        this.tableInfo = new TableInfo(atts.getValue(Constants.TABLE_TAG_NAME_ATT));
        this.tableInfo.setComment(atts.getValue(Constants.TABLE_TAG_COMMENT_ATT));
        this.tableCreated = false;

        log.info("Parsing table ["+this.tableInfo.getName()+"]...");
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.xml.SAXTagHandler#closeTag(java.lang.String, java.lang.String, java.lang.String)
     */
    public void closeTag(String uri, String localName, String name)
            throws SAXException {

        if (!this.tableCreated)
            createTable();

        log.info("Finished table ["+this.tableInfo.getName()+"].");
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.xml.SAXTagHandler#processCharacters(char[], int, int)
     */
    public void processCharacters(char[] ch, int start, int length)
            throws SAXException {
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.xml.SAXTagHandler#startChildElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    public SAXTagHandler startChildElement(String uri, String localName,
            String name, Attributes atts) throws SAXException {

        if (!Constants.JDBC2XML_NS_URI.equals(uri) && !Constants.W3_XINCLUDE_NS_URI.equals(uri))
            throw new SAXException("Namspace URI ["+uri+"] is not supported.");

        if (Constants.COLSET_TAG_NAME.equals(localName)) {

            return new ColumnSetTagHandler(this.tableInfo);
        }
        else if (Constants.PRIMARY_KEY_TAG_NAME.equals(localName)) {

            this.tableInfo.setPrimaryKey(new PrimaryKeyInfo(atts));
            return null;
        }
        else if (Constants.INDEXSET_TAG_NAME.equals(localName)) {

            return new IndexSetTagHandler(this.tableInfo);
        }
        else if (Constants.FOREIGN_KEYS_TAG_NAME.equals(localName)) {

            return new ForeignKeySetTagHandler(this.tableInfo);
        }
        else if (Constants.ROWSET_TAG_NAME.equals(localName)) {

            if (this.context.getProcessRestrictionFilter().isProcessData())
            {
                if (!this.tableCreated)
                    createTable();

                return new RowSetTagHandler(this.context,this.tableInfo);
            }
            else
            {
                log.info("Ignoring rowset of table ["+this.tableInfo.getName()+"], because --no-data has been specified.");
                return null;
            }
        }
        else
            throw new SAXException("xmldump child element ["+localName+"] is not supported.");
    }

}
