/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Created: 13.10.2009
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.sax;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;

/**
 * A {@link ContentHandler} that will perform some formatting issues
 * (indentation, line breaks) before delegating calls to its owned handler.<br/>
 * An instance of this class could simply be inserted into a chain of content handlers to produce well formatted (human readable) XML-Files.<br/>
 * <br/>
 * <code>
 *     ...<br/>
 *     XmlPrettyConfig cfg = new XmlPrettyConfig();<br/>
 *     ContentHandler formatHandler = new XmlPretty(handler, cfg);<br/>
 *     formatHandler.startDocument();<br/>
 *     ...
 * </code>
 * @author rbreuss
 * 
 */
public class XmlPretty implements ContentHandler {

    private final ContentHandler contentHandler;
    private char[] prettyChars;
    private int prettyDepth;
    private int singleLineCount;
    private String lastStartElement;
    private XmlPrettyConfig prettyConfig;
    private StringBuilder characterBuffer;
    
    /**
     * Create a new instance using the passed handler.
     * @param contentHandler The handler to delegate calls to.
     */
    public XmlPretty(final ContentHandler contentHandler) {
        this.contentHandler = contentHandler;
        this.characterBuffer = new StringBuilder();
        this.prettyConfig = new XmlPrettyConfig(2, 8);
        initPrettyParams();
    }
    
    /**
     * Create a new instance using the passed handler.
     * @param contentHandler The handler to delegate calls to.
     * @param prettyConfig Formatting configuration.
     */
    public XmlPretty(final ContentHandler contentHandler, XmlPrettyConfig prettyConfig) {
        this.contentHandler = contentHandler;
        this.characterBuffer = new StringBuilder();
        this.prettyConfig = prettyConfig;
        initPrettyParams();
    }
    
    protected void initPrettyParams()
    {
        this.prettyDepth = 0;
        int maxPrettyLength = 1 + this.prettyConfig.getWidth() * this.prettyConfig.getMaxDepth();
        StringBuffer sb = new StringBuffer("\n");
        for (int i=0; i<maxPrettyLength; i++) 
        { 
            sb.append(' '); 
        }
        this.prettyChars = sb.toString().toCharArray();
        this.singleLineCount = 0;
    }

    private void nextLine() throws SAXException
    {
        if (this.prettyConfig.isActive())
            this.contentHandler.ignorableWhitespace(this.prettyChars, 
                    0, 
                    1 + this.prettyConfig.getWidth() * Math.min(this.prettyDepth, this.prettyConfig.getMaxDepth()));
    }

    /**
     * @param ch
     * @param start
     * @param length
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#characters(char[], int, int)
     */
    public void characters(char[] ch, int start, int length)
            throws SAXException {
        // we collect the characters and decide in 'endElement' whether we will
        // deliver them to the contentHandler, or not.
        // Characters may consist of whitespace only, and we are not able to
        // distinguish between XML-Formatting whitespace and Value-Specific
        // Whitespace at the moment.
        this.characterBuffer.append(ch, start, length);
    }

    /**
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#endDocument()
     */
    public void endDocument() throws SAXException {
        this.contentHandler.endDocument();
    }

    /**
     * @param uri
     * @param localName
     * @param qName
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#endElement(java.lang.String, java.lang.String, java.lang.String)
     */
    public void endElement(String uri, String localName, String qName)
            throws SAXException {
        // If we did collect characters between start and end of the current
        // element, it's not a whitespace, it's data!
        if (this.lastStartElement.equalsIgnoreCase(qName)
                && this.characterBuffer.length() > 0)
            this.contentHandler.characters(this.characterBuffer.toString().toCharArray(), 0, this.characterBuffer.length());
        this.characterBuffer.setLength(0);
        
        this.prettyDepth--;
        if (!qName.equals(this.lastStartElement) && this.singleLineCount == 0)
            this.nextLine();
        if (this.prettyConfig.getSingleLineTags().contains(qName))
            this.singleLineCount--;
        this.contentHandler.endElement(uri, localName, qName);
    }

    /**
     * @param prefix
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#endPrefixMapping(java.lang.String)
     */
    public void endPrefixMapping(String prefix) throws SAXException {
        this.contentHandler.endPrefixMapping(prefix);
    }

    /**
     * @param ch
     * @param start
     * @param length
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#ignorableWhitespace(char[], int, int)
     */
    public void ignorableWhitespace(char[] ch, int start, int length)
            throws SAXException {
        // don't pass any whitespace
        this.contentHandler.ignorableWhitespace(ch, 0, 0);
    }

    /**
     * @param target
     * @param data
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#processingInstruction(java.lang.String, java.lang.String)
     */
    public void processingInstruction(String target, String data)
            throws SAXException {
        this.contentHandler.processingInstruction(target, data);
    }

    /**
     * @param locator
     * @see org.xml.sax.ContentHandler#setDocumentLocator(org.xml.sax.Locator)
     */
    public void setDocumentLocator(Locator locator) {
        this.contentHandler.setDocumentLocator(locator);
    }

    /**
     * @param name
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#skippedEntity(java.lang.String)
     */
    public void skippedEntity(String name) throws SAXException {
        this.contentHandler.skippedEntity(name);
    }

    /**
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#startDocument()
     */
    public void startDocument() throws SAXException {
        this.contentHandler.startDocument();
    }

    /**
     * @param uri
     * @param localName
     * @param qName
     * @param atts
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    public void startElement(String uri, String localName, String qName,
            Attributes atts) throws SAXException {

        if (this.singleLineCount == 0)
            this.nextLine();
        if (this.prettyConfig.getSingleLineTags().contains(qName))
            this.singleLineCount++;
        this.lastStartElement = qName;
        this.contentHandler.startElement(uri, localName, qName, atts);
        this.prettyDepth++;
        this.characterBuffer.setLength(0);
    }

    /**
     * @param prefix
     * @param uri
     * @throws SAXException
     * @see org.xml.sax.ContentHandler#startPrefixMapping(java.lang.String, java.lang.String)
     */
    public void startPrefixMapping(String prefix, String uri)
            throws SAXException {
        this.contentHandler.startPrefixMapping(prefix, uri);
    }

    /**
     * @return the prettyConfig
     */
    public XmlPrettyConfig getPrettyConfig() {
        return this.prettyConfig;
    }
    
}
