/***********************************************************
 * $Id$
 *
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 26.11.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.jdbc2xml.sax;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.TimeZone;

import org.clazzes.jdbc2xml.Constants;
import org.clazzes.jdbc2xml.helper.TypesHelper;
import org.clazzes.jdbc2xml.schema.ColumnInfo;
import org.clazzes.jdbc2xml.schema.ForeignKeyInfo;
import org.clazzes.jdbc2xml.schema.ISchemaEngine;
import org.clazzes.jdbc2xml.schema.IndexInfo;
import org.clazzes.jdbc2xml.schema.SchemaEngine;
import org.clazzes.jdbc2xml.schema.SortableTableDescription;
import org.clazzes.jdbc2xml.schema.TableInfo;
import org.clazzes.jdbc2xml.schema.TableSorter;
import org.clazzes.jdbc2xml.serialization.ISerializationHandlerFactory;
import org.clazzes.jdbc2xml.serialization.SerializationHandler;
import org.clazzes.jdbc2xml.serialization.SerializationHandlerFactory;
import org.clazzes.jdbc2xml.sql.ISqlIdentifierMapperFactory;
import org.clazzes.jdbc2xml.sql.SqlIdentifierMapper;
import org.clazzes.jdbc2xml.sql.SqlIdentifierMapperFactory;
import org.clazzes.jdbc2xml.tools.ProcessRestrictionFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

/**
 * This class writes a series of JDBC select queries to an XML document using
 * a SAX ContentHandler instance.
 *
 * @author wglas
 */
public class JDBCToSAXWriter {

    static private final Logger log = LoggerFactory.getLogger(JDBCToSAXWriter.class);

    static private final ISerializationHandlerFactory serializationHandlerFactory=
            SerializationHandlerFactory.newInstance();

    static private final ISqlIdentifierMapperFactory sqlIdentifierMapperFactory=
            SqlIdentifierMapperFactory.newInstance();

    private static class StatementInfo extends SortableTableDescription {

        public final String sql;

        public StatementInfo(String sql, TableInfo tableInfo)
        {
            super(tableInfo);
            this.sql = sql;
        }
    }

    private List<StatementInfo> statements;
    private ISchemaEngine schemaEngine;
    private SqlIdentifierMapper sqlIdentifierMapper;
    private ContentHandler contentHandler;
    private ProcessRestrictionFilter processRestrictionFilter;

    /**
     * Create an uninitialized instance.
     * Use {@link #setConnection(Connection)}, {@link #setContentHandler(ContentHandler)} and
     * {@link #setTimeZone(TimeZone)}
     * in order to initialize this instance.
     */
    public JDBCToSAXWriter()
    {
        super();
        this.processRestrictionFilter = new ProcessRestrictionFilter();
    }

    /**
     * @param connection The database connection to use.
     * @param timeZone The time zone used to process date values.
     * @param contentHandler The SAX content handler which drives the XML output.
     * @throws SQLException Upon errors when inspecting the database.
     */
    public JDBCToSAXWriter(Connection connection, TimeZone timeZone,
            ContentHandler contentHandler) throws SQLException {
        this(SchemaEngine.newInstance(), contentHandler);
        this.schemaEngine.setConnection(connection);
        this.schemaEngine.setTimeZone(timeZone);
    }

    /**
     * @param schemaEngine The initialized SchemaEngine carrying the database connection
     *                     and default time zone.
     * @param contentHandler The SAX content handler which drives the XML output.
     * @throws SQLException Upon errors when inspecting the database.
     */
    public JDBCToSAXWriter(ISchemaEngine schemaEngine,
            ContentHandler contentHandler) throws SQLException {
        this();
        this.schemaEngine = schemaEngine;
        this.contentHandler = contentHandler;
    }

    /**
     * @return the connection
     */
    public Connection getConnection() {
        return this.schemaEngine.getConnection();
    }


    /**
     * @param connection the connection to set
     * @throws SQLException
     */
    public void setConnection(Connection connection) throws SQLException {
        this.schemaEngine.setConnection(connection);
    }


    /**
     * @return the contentHandler
     */
    public ContentHandler getContentHandler() {
        return this.contentHandler;
    }


    /**
     * @param contentHandler the contentHandler to set
     */
    public void setContentHandler(ContentHandler contentHandler) {
        this.contentHandler = contentHandler;
    }

    /**
     * @return The default time zone for parsing date values.
     */
    public TimeZone getTimeZone() {
        return this.schemaEngine.getTimeZone();
    }

    /**
     * @param timeZone the timeZone to set
     */
    public void setTimeZone(TimeZone timeZone) {
        this.schemaEngine.setTimeZone(timeZone);
    }

    /**
     * @return The schema to which the written tables belong.
     */
    public String getSchema() {
        return this.schemaEngine.getSchema();
    }

    /**
     * @param schema The schema to which the written tables belong to set.
     */
    public void setSchema(String schema) {
        this.schemaEngine.setSchema(schema);
    }

    private TableInfo makeTableInfo(String table) throws SQLException
    {
        return this.schemaEngine.fetchTableInfo(table,this.processRestrictionFilter);
    }

    /**
     * Add the given list of database tables to the data being dumped.
     *
     * @param tables A list of database table names.
     * @throws SQLException
     */
    public void addTables(Collection<String> tables) throws SQLException {

        for (String table : tables)
        {
            this.addTable(table);
        }
    }

    /**
     * Add the given database table to the data being dumped.
     *
     * @param table A database table name.
     * @throws SQLException
     */
    public void addTable(String table) throws SQLException {

        this.addTable(this.makeTableInfo(table));
    }

    /**
     * Add the given database table to the data being dumped.
     *
     * The caller is responsible for assuring, that the passed table
     * description is actually in sync with the database.
     *
     * @param tableInfo A database table description.
     * @throws SQLException
     */
    public void addTable(TableInfo tableInfo) throws SQLException {

        if (this.statements == null)
            this.statements = new LinkedList<StatementInfo>();

        String sql = "select * from " + tableInfo.getName();
        this.statements.add(new StatementInfo(sql,tableInfo));
    }

    /**
     * Add the given database table and a query, which selects a subset
     * of the table data to the data being dumped.
     *
     * @param table A database table name.
     * @param sql A query that selects data from this table, but restrict the
     *            returned row to a subset of the whole table. Typically, such
     *            a query looks like
     * <pre>
     *   select a.* from account a, person p where a.person_id=p.id and person.name='Me';
     * </pre>
     * assuming that table account has a foreign key to table person.
     * @throws SQLException
     */
    public void addRestrictedTable(String table, String sql) throws SQLException {

        if (this.statements == null)
            this.statements = new LinkedList<StatementInfo>();

        this.statements.add(new StatementInfo(sql,this.makeTableInfo(table)));
    }

    /**
     * Add the given list of queries to the data being dumped.
     *
     * @param queries A list of SQL queries.
     */
    public void addQueries(Collection<String> queries) {

        for (String sql : queries)
        {
            this.addQuery(sql);
        }
    }

    /**
     * Add the given SQL select query to the data being dumped.
     *
     * @param sql An SQL query statement.
     */
    public void addQuery(String sql) {

        if (this.statements == null)
            this.statements = new LinkedList<StatementInfo>();

        this.statements.add(new StatementInfo(sql,null));
    }

    private static void addAttr(AttributesImpl atts, String key, String value)
    {
        atts.addAttribute("","",key,"CDATA",value);
    }

    private void startElement(String tag, Attributes atts) throws SAXException
    {
        this.contentHandler.startElement(Constants.JDBC2XML_NS_URI,tag,tag,atts);
    }

    private void endElement(String tag) throws SAXException
    {
        this.contentHandler.endElement(Constants.JDBC2XML_NS_URI,tag,tag);
    }

    private void emptyElement(String tag, Attributes atts) throws SAXException
    {
        this.startElement(tag,atts);
        this.endElement(tag);
    }

    private void writeForeignKeys(List<ForeignKeyInfo> fkInfos) throws SAXException
    {
        this.startElement(Constants.FOREIGN_KEYS_TAG_NAME,null);

        for (ForeignKeyInfo fkInfo : fkInfos) {
            this.emptyElement(Constants.FOREIGN_KEY_TAG_NAME,
                    fkInfo.toAttributes(this.sqlIdentifierMapper));
        }
        this.endElement(Constants.FOREIGN_KEYS_TAG_NAME);
    }

    private void writeIndices(List<IndexInfo> indexInfos) throws SAXException
    {
        this.startElement(Constants.INDEXSET_TAG_NAME,null);

        for (IndexInfo indexInfo : indexInfos) {

            this.emptyElement(Constants.INDEX_TAG_NAME,
                    indexInfo.toAttributes(this.sqlIdentifierMapper));
        }
        this.endElement(Constants.INDEXSET_TAG_NAME);
    }

    private void writeTableContraints(TableInfo tableInfo) throws SAXException
    {
        // <primaryKey>
        if (tableInfo.getPrimaryKey() != null) {
            this.emptyElement(Constants.PRIMARY_KEY_TAG_NAME,
                    tableInfo.getPrimaryKey().toAttributes(this.sqlIdentifierMapper));
        }

        // <indexset>
        if (tableInfo.getIndices() != null) {
            this.writeIndices(tableInfo.getIndices());
        }

        // <foreignkeyset>
        if (this.processRestrictionFilter.isProcessConstraints() &&
                tableInfo.getForeignKeys() != null) {
            this.writeForeignKeys(tableInfo.getForeignKeys());
        }
    }

    /**
     * Proceed by executing all SQL statement and writing the resulting data
     * to the ContentHandler.
     *
     * @throws SQLException
     * @throws SAXException
     */
    public void processData() throws SQLException, SAXException
    {
        if (this.sqlIdentifierMapper == null)
            this.sqlIdentifierMapper =
            sqlIdentifierMapperFactory.newMapper(this.processRestrictionFilter.getIdMapper());

        List<String> tmpTableNameList = this.schemaEngine.getTempTableNames();
        int n = tmpTableNameList.size();
        if (n > 0)
        {
            String[] tmpTableNames = new String[n];
            for (int i=0; i<n; i++)
            {
                tmpTableNames[i] = tmpTableNameList.get(i);
            }
            this.processRestrictionFilter.setExcludedTableNames(tmpTableNames);
        }

        Connection connection = this.schemaEngine.getConnection();

        boolean oldReadOnly = connection.isReadOnly();
        int oldIsolationLevel = connection.getTransactionIsolation();

        try {
            if (!oldReadOnly)
                connection.setReadOnly(true);

            if (this.processRestrictionFilter.isTransactional()) {
                log.info("Starting a read-only transaction.");

                try {
                    if (log.isDebugEnabled())
                        log.debug("Setting the transaction isolation level to ["+Connection.TRANSACTION_SERIALIZABLE+"].");

                    connection.setTransactionIsolation(Connection.TRANSACTION_SERIALIZABLE);
                } catch(SQLException e) {
                    log.warn("TRANSACTION_SERIALIZABLE isolation level not supported, expect inconsistent result despite.",e);
                    oldIsolationLevel = Connection.TRANSACTION_NONE;
                }

                connection.setAutoCommit(false);
            }

            log.info("Writing catalog ["+this.schemaEngine.getConnection().getCatalog()+"] to XML...");

            List<StatementInfo> statementsWritten;

            if (this.statements == null) {
                // fetch all table meta data.
                List<TableInfo> tables = this.schemaEngine.fetchTableInfos(this.processRestrictionFilter);
                statementsWritten = new ArrayList<StatementInfo>(tables.size());

                for (TableInfo ti : tables) {

                    String sql = "select * from " + ti.getName();

                    String whereClause =
                        this.processRestrictionFilter.getTableClause(ti.getName());

                    if (whereClause != null) {
                        sql += " WHERE " + whereClause;
                    }

                    statementsWritten.add(new StatementInfo(sql,ti));
                }
            } else {
                statementsWritten = this.statements;
            }

            TableSorter.sortTablesByFKDepth(statementsWritten);

            this.contentHandler.startDocument();
            this.contentHandler.startPrefixMapping("",Constants.JDBC2XML_NS_URI);
            this.contentHandler.startPrefixMapping("xi",Constants.W3_XINCLUDE_NS_URI);

            this.startElement(Constants.TOP_TAG_NAME,null);

            for (StatementInfo info : statementsWritten)
            {
                // <table> or <query>
                AttributesImpl atts = new AttributesImpl();
                if (info.getTableInfo() == null)
                {
                    log.info("Writing query ["+info.sql+"]...");

                    addAttr(atts,Constants.QUERY_TAG_SQL_ATT,this.sqlIdentifierMapper.toExternal(info.sql));

                    this.startElement(Constants.QUERY_TAG_NAME,atts);
                } else
                {
                    log.info("Writing table ["+info.getTableInfo().getName()+"]...");

                    addAttr(atts,Constants.TABLE_TAG_NAME_ATT,
                            this.sqlIdentifierMapper.toExternal(info.getTableInfo().getName()));
                    if (info.getTableInfo().getComment() != null)
                        addAttr(atts,Constants.TABLE_TAG_COMMENT_ATT,info.getTableInfo().getComment());
                    this.startElement(Constants.TABLE_TAG_NAME,atts);
                }

                long count = 0;

                if (info.getTableInfo() == null || this.processRestrictionFilter.isProcessData())
                {
                    try(Statement statement = connection.createStatement(ResultSet.TYPE_FORWARD_ONLY,
                            ResultSet.CONCUR_READ_ONLY)) {

                        statement.setFetchSize(this.processRestrictionFilter.getBatchSize());

                        try(ResultSet resultSet = statement.executeQuery(info.sql)) {

                            ResultSetMetaData metaData = resultSet.getMetaData();

                            SerializationHandler[] handlers = new SerializationHandler[metaData.getColumnCount()];

                            // <colset>
                            this.startElement(Constants.COLSET_TAG_NAME,null);

                            for (int column=0;column<metaData.getColumnCount();++column)
                            {
                                ColumnInfo ci;

                                if (info.getTableInfo() != null)
                                {
                                    ci = info.getTableInfo().getColumnInfo(metaData.getColumnName(column+1));
                                    // FIXME  info.getTableInfo().getColumnInfo() does not provide correct autoIncrement flags for all dialects!

                                    if (ci == null)
                                        throw new SQLException("Column ["+metaData.getColumnName(column+1)+"] of query ["+info.sql+
                                                "] could not be found in the metadata for table ["+info.getTableInfo().getName()+"].");
                                }
                                else
                                {
                                    ci = new ColumnInfo(metaData,column+1);
                                }

                                handlers[column] =
                                        serializationHandlerFactory.newSerializationHandler(ci,this.getTimeZone());

                                if (handlers[column] == null)
                                    throw new SQLException("Unsupported type ["+TypesHelper.typeToString(ci.getType())+
                                            "] in column ["+(column+1)+"] of query ["+info.sql+"].");

                                this.emptyElement(Constants.COLUMN_TAG_NAME,
                                        ci.toAttributes(this.sqlIdentifierMapper));
                            }
                            // </colset>
                            this.endElement(Constants.COLSET_TAG_NAME);

                            if (info.getTableInfo() != null)
                                this.writeTableContraints(info.getTableInfo());

                            // <rowset>
                            this.startElement(Constants.ROWSET_TAG_NAME,null);

                            while (resultSet.next())
                            {
                                ++count;
                                // <row>
                                this.startElement(Constants.ROW_TAG_NAME,null);

                                for (int column=0;column<metaData.getColumnCount();++column)
                                {
                                    handlers[column].fetchData(resultSet, column+1);
                                    if (handlers[column].isNull()) continue;

                                    atts = new AttributesImpl();
                                    addAttr(atts,Constants.VALUE_TAG_COL_ATT,Integer.toString(column));

                                    this.startElement(Constants.VALUE_TAG_NAME,atts);
                                    handlers[column].pushData(this.contentHandler);
                                    this.endElement(Constants.VALUE_TAG_NAME);
                                }
                                // </row>
                                this.endElement(Constants.ROW_TAG_NAME);
                            }
                            // </rowset>
                            this.endElement(Constants.ROWSET_TAG_NAME);
                        }
                    }
                }
                else
                {
                    // schema-only dump of a table
                    // <colset>
                    this.startElement(Constants.COLSET_TAG_NAME,null);

                    for (ColumnInfo ci:info.getTableInfo().getColumns())
                    {
                        this.emptyElement(Constants.COLUMN_TAG_NAME,
                                ci.toAttributes(this.sqlIdentifierMapper));
                    }

                    // </colset>
                    this.endElement(Constants.COLSET_TAG_NAME);

                    this.writeTableContraints(info.getTableInfo());

                    count = -1;
                }

                // </table> or </query>
                if (info.getTableInfo() == null) {
                    this.endElement(Constants.QUERY_TAG_NAME);
                    log.info("["+count+"] rows of query ["+info.sql+"] have been successfully written.");
                }
                else {
                    this.endElement(Constants.TABLE_TAG_NAME);
                    if (count >= 0)
                        log.info("["+count+"] rows of table ["+info.getTableInfo().getName()+"] have been successfully written.");
                    else
                        log.info("Table ["+info.getTableInfo().getName()+"] has been successfully written.");
                }

            }

            this.endElement(Constants.TOP_TAG_NAME);
            this.contentHandler.endPrefixMapping("");
            this.contentHandler.endPrefixMapping("xi");
            this.contentHandler.endDocument();
            log.info("Finished writing catalog ["+this.schemaEngine.getConnection().getCatalog()+"] to XML.");

        } finally {

            if (this.processRestrictionFilter.isTransactional()) {
                log.info("Closing the read-only transaction.");
                connection.setAutoCommit(true);

                if (oldIsolationLevel != Connection.TRANSACTION_NONE &&
                        oldIsolationLevel != connection.getTransactionIsolation()) {
                    try {

                        if (log.isDebugEnabled())
                            log.debug("Resetting the transaction isolation level to ["+oldIsolationLevel+"].");

                        connection.setTransactionIsolation(oldIsolationLevel);
                    } catch(SQLException e) {
                        log.warn("Error resetting the transaction isolation level to ["+oldIsolationLevel+"].",e);
                    }
                }
            }

            if (!oldReadOnly)
                connection.setReadOnly(false);
        }
    }

    /**
     * @return The processRestrictionFilter, which carries all configurable settings
     *         for the data export.
     */
    public ProcessRestrictionFilter getProcessRestrictionFilter() {
        return this.processRestrictionFilter;
    }

    /**
     * @param processFilter The processRestrictionFilter to set.
     */
    public void setProcessRestrictionFilter(ProcessRestrictionFilter processFilter) {
        this.processRestrictionFilter = processFilter;
        this.sqlIdentifierMapper = null;

        this.schemaEngine.getDialect().setProperties(processFilter.getDialectProps());
    }
}
