/***********************************************************
 * $Id$
 *
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 08.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.jdbc2xml.sax;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.TimeZone;

import org.clazzes.jdbc2xml.Constants;
import org.clazzes.jdbc2xml.sax.impl.InsertContext;
import org.clazzes.jdbc2xml.sax.impl.XmlDumpTagHandler;
import org.clazzes.jdbc2xml.schema.ISchemaEngine;
import org.clazzes.jdbc2xml.schema.SchemaEngine;
import org.clazzes.jdbc2xml.tools.ProcessRestrictionFilter;
import org.clazzes.util.xml.SAXTagHandler;
import org.clazzes.util.xml.SimpleContentHandler;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * @author wglas
 *
 */
public class InsertContentHandler extends SimpleContentHandler {

    private InsertContext context;

    /**
     * Create an insert handler for the given connection and use a schema engine
     * created by {@link SchemaEngine#newInstance()}.
     *
     * @param connection The database connection on which to insert data.
     * @param timeZone The default time zone to use for date values.
     * @throws SQLException
     */
    public InsertContentHandler(Connection connection, TimeZone timeZone) throws SQLException
    {
        ISchemaEngine schemaEngine = SchemaEngine.newInstance();
        schemaEngine.setConnection(connection);
        schemaEngine.setTimeZone(timeZone);
        this.context = new InsertContext(schemaEngine);
    }

    /**
     * Create an insert handler for the given connection and use the supplied
     * schema engine.
     *
     * @param schemaEngine The schema engine carrying the DB connection to insert to and
     *                     the default time zone.
     * @throws SQLException
     */
    public InsertContentHandler(ISchemaEngine schemaEngine) throws SQLException
    {
        this.context = new InsertContext(schemaEngine);
    }

    /**
     * @return the processRestrictionFilter
     */
    public ProcessRestrictionFilter getProcessRestrictionFilter() {
        return this.context.getProcessRestrictionFilter();
    }

    /**
     * @param processFilter The processRestrictionFilter to set.
     */
    public void setProcessRestrictionFilter(ProcessRestrictionFilter processFilter) {
        this.context.setProcessRestrictionFilter(processFilter);

        this.context.getSchemaEngine().getDialect().setProperties(processFilter.getDialectProps());
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.xml.SimpleContentHandler#getRootHandler(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    @Override
    protected SAXTagHandler getRootHandler(String uri, String localName,
            String name, Attributes atts) throws SAXException {

        if (!Constants.JDBC2XML_NS_URI.equals(uri))
            throw new SAXException("Namspace URI ["+uri+"] is not supported.");

        if (!Constants.TOP_TAG_NAME.equals(localName))
            throw new SAXException("Top level element ["+localName+"] is not supported.");

        if (this.context.getProcessRestrictionFilter() == null)
            this.context.setProcessRestrictionFilter(new ProcessRestrictionFilter());

        return new XmlDumpTagHandler(this.context);
    }

}
