/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 26.11.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.helper;

import java.sql.DatabaseMetaData;
import java.sql.Types;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Static helper methods for SQL types.
 * 
 * @author wglas
 */
public abstract class TypesHelper {

    private static final Logger log = LoggerFactory.getLogger(TypesHelper.class);
    
    private static final String[] sortedTypeNames =
    {
        "ARRAY",
        "BIGINT",
        "BINARY",
        "BIT",
        "BLOB",
        "BOOLEAN",
        "CHAR",
        "CLOB",
        "DATALINK",
        "DATE",
        "DECIMAL",
        "DISTINCT",
        "DOUBLE",
        "FLOAT",
        "INTEGER",
        "JAVA_OBJECT",
        "LONGVARBINARY",
        "LONGVARCHAR",
        "NULL",
        "NUMERIC",
        "OTHER",
        "REAL",
        "REF",
        "SMALLINT",
        "STRUCT",
        "TIME",
        "TIMESTAMP",
        "TIMESTAMP_WITH_TIMEZONE",
        "TINYINT",
        "VARBINARY",
        "VARCHAR"

    };
    
    private static final int sortedTypes[] =
    {
        java.sql.Types.ARRAY,
        java.sql.Types.BIGINT,
        java.sql.Types.BINARY,
        java.sql.Types.BIT,
        java.sql.Types.BLOB,
        java.sql.Types.BOOLEAN,
        java.sql.Types.CHAR,
        java.sql.Types.CLOB,
        java.sql.Types.DATALINK,
        java.sql.Types.DATE,
        java.sql.Types.DECIMAL,
        java.sql.Types.DISTINCT,
        java.sql.Types.DOUBLE,
        java.sql.Types.FLOAT,
        java.sql.Types.INTEGER,
        java.sql.Types.JAVA_OBJECT,
        java.sql.Types.LONGVARBINARY,
        java.sql.Types.LONGVARCHAR,
        java.sql.Types.NULL,
        java.sql.Types.NUMERIC,
        java.sql.Types.OTHER,
        java.sql.Types.REAL,
        java.sql.Types.REF,
        java.sql.Types.SMALLINT,
        java.sql.Types.STRUCT,
        java.sql.Types.TIME,
        java.sql.Types.TIMESTAMP,
        java.sql.Types.TIMESTAMP_WITH_TIMEZONE,
        java.sql.Types.TINYINT,
        java.sql.Types.VARBINARY,
        java.sql.Types.VARCHAR
    };
    
    /**
     * Convert an SQL type to a string.
     * 
     * @param type A constant of {@link Types}
     * @return A string corresponding to the JAVA SQL type constant. 
     */
    static public String typeToString(int type) {
        switch (type) {
        
        case java.sql.Types.ARRAY:
            return "ARRAY";
            
        case java.sql.Types.BIGINT:
            return "BIGINT";

        case java.sql.Types.BINARY:
            return "BINARY";
            
        case java.sql.Types.BIT:
            return "BIT";
            
        case java.sql.Types.BLOB:
            return "BLOB";
            
        case java.sql.Types.BOOLEAN:
            return "BOOLEAN";
            
        case java.sql.Types.CHAR:
        case java.sql.Types.NCHAR:
            return "CHAR";
            
        case java.sql.Types.CLOB:
            return "CLOB";
            
        case java.sql.Types.DATALINK:
            return "DATALINK";
            
        case java.sql.Types.DATE:
            return "DATE";
            
        case java.sql.Types.DECIMAL:
            return "DECIMAL";
            
        case java.sql.Types.DISTINCT:
            return "DISTINCT";
            
        case java.sql.Types.DOUBLE:
            return "DOUBLE";
            
        case java.sql.Types.FLOAT:
            return "FLOAT";
            
        case java.sql.Types.INTEGER:
            return "INTEGER";
            
        case java.sql.Types.JAVA_OBJECT:
            return "JAVA_OBJECT";
            
        case java.sql.Types.LONGVARBINARY:
            return "LONGVARBINARY";
            
        case java.sql.Types.LONGVARCHAR:
        case java.sql.Types.LONGNVARCHAR:
            return "LONGVARCHAR";
            
        case java.sql.Types.NULL:
            return "NULL";
            
        case java.sql.Types.NUMERIC:
            return "NUMERIC";
            
        case java.sql.Types.OTHER:
            return "OTHER";
            
        case java.sql.Types.REAL:
            return "REAL";
            
        case java.sql.Types.REF:
            return "REF";
            
        case java.sql.Types.SMALLINT:
            return "SMALLINT";
            
        case java.sql.Types.STRUCT:
            return "STRUCT";
            
        case java.sql.Types.TIME:
            return "TIME";
            
        case java.sql.Types.TIMESTAMP:
            return "TIMESTAMP";

        case java.sql.Types.TIMESTAMP_WITH_TIMEZONE:
            return "TIMESTAMP_WITH_TIMEZONE";

        case java.sql.Types.TINYINT:
            return "TINYINT";
            
        case java.sql.Types.VARBINARY:
            return "VARBINARY";
            
        case java.sql.Types.VARCHAR:
        case java.sql.Types.NVARCHAR:
            return "VARCHAR";
            
        default:
            return null;        
        }
    }
    
    /**
     * @param s A String representing the JAVA SQL type.
     * @return The JAVA SQL type constant.If the string is unknown, return
     *         {@link java.sql.Types#OTHER}.
     */
    static public int stringToType(String s) {
 
        if (s == null) return java.sql.Types.OTHER;
        
        int i0 = 0;
        int i1 = sortedTypeNames.length;
        
        while (i1 > i0) {
            int i = i0 + (i1-i0)/2;
            
            int r = sortedTypeNames[i].compareToIgnoreCase(s);
         
            if (r == 0)
                return sortedTypes[i];
            
            if (r < 0)
                i0 = i + 1;
            else
                i1 = i;
        }
        
        if (i0 < sortedTypeNames.length &&
                sortedTypeNames[i0].compareToIgnoreCase(s) == 0)
            return sortedTypes[i0];
        
        return java.sql.Types.OTHER;
    }
    
    /**
     * @param type An JDBC type contant.
     * @return Whether this type is numeric,
     *         i.e. values of this type do not need to be quoted.
     */
    public static boolean isNumeric(int type)
    {
        return
        type == Types.BIGINT ||
        type == Types.DECIMAL ||
        type == Types.DOUBLE ||
        type == Types.FLOAT ||
        type == Types.INTEGER ||
        type == Types.NUMERIC ||
        type == Types.REAL ||
        type == Types.SMALLINT ||
        type == Types.TINYINT ||
        type == Types.BOOLEAN ||
        type == Types.BIT;
    }
    
    /**
     * @param type An JDBC type contant.
     * @return Whether this type is a string value,
     *         i.e. values of this type are not binary not numeric and
     *         not of date/time style and
     *         are not a large character object (CLOB/LONGVARCHAR).
     */
    public static boolean isString(int type)
    {
        return
        type == Types.CHAR ||
        type == Types.NCHAR ||
        type == Types.VARCHAR ||
        type == Types.NVARCHAR;
    }
    
    /**
     * Parse deferrability string of the deferrability attribute of a foreignkey tag.
     * 
     * @param s <code>"NotDeferrable"</code>, <code>"InitiallyDeferred"</code> or
     *          <code>"InitiallyImmediate"</code>.
     * @return {@link DatabaseMetaData#importedKeyNotDeferrable},
     *         {@link DatabaseMetaData#importedKeyInitiallyDeferred} or
     *         {@link DatabaseMetaData#importedKeyInitiallyImmediate}.
     *         If <code>s</code> is <code>null</code> or an unrecognized string,
     *         the default value {@link DatabaseMetaData#importedKeyNotDeferrable}
     *         is assumed.
     */
    public static short deferrabilityStringToShort(String s)
    {
        if (s==null || "NotDeferrable".equals(s))
            return DatabaseMetaData.importedKeyNotDeferrable;
        if ("InitiallyDeferred".equals(s))
            return DatabaseMetaData.importedKeyInitiallyDeferred;
        if ("InitiallyImmediate".equals(s))
            return DatabaseMetaData.importedKeyInitiallyImmediate;
        
        log.warn("Invalid deferrability string ["+s+"] specified; assuming default value [NotDeferrable].");
        
        return DatabaseMetaData.importedKeyNotDeferrable;
    }
    
    /**
     * Parse deferrability string of the deferrability attribute of a foreignkey tag.
     * 
     * @param s {@link DatabaseMetaData#importedKeyNotDeferrable},
     *         {@link DatabaseMetaData#importedKeyInitiallyDeferred} or
     *         {@link DatabaseMetaData#importedKeyInitiallyImmediate}.
     * @return <code>"NotDeferrable"</code>, <code>"InitiallyDeferred"</code> or
     *         <code>"InitiallyImmediate"</code>.
     *         If <code>s</code> is an unrecognized integer,
     *         the default value <code>"NotDeferrable"</code> is assumed.
     */
    public static String deferrabilityShortToString(short s)
    {
        switch (s)
        {
        case DatabaseMetaData.importedKeyNotDeferrable:
            return "NotDeferrable";
        case DatabaseMetaData.importedKeyInitiallyDeferred:
            return "InitiallyDeferred";
        case DatabaseMetaData.importedKeyInitiallyImmediate:
            return "InitiallyImmediate";
        }
                
        log.warn("Invalid deferrability value ["+s+"] specified; assuming default value [NotDeferrable].");
        
        return "NotDeferrable";
    }
    
    /**
     * Parse a delete/update rule string of a deleteRule or updateRule attribute of a foreignkey tag.
     * 
     * @param s <code>"NoAction"</code>, <code>"Cascade"</code>, 
     *          <code>"Restrict"</code>, <code>"SetDefault"</code> or
     *          <code>"SetNull"</code>.
     * @return {@link DatabaseMetaData#importedKeyNoAction},
     *         {@link DatabaseMetaData#importedKeyCascade},
     *         {@link DatabaseMetaData#importedKeyRestrict},
     *         {@link DatabaseMetaData#importedKeySetDefault} or
     *         {@link DatabaseMetaData#importedKeySetNull}.
     *         If <code>s</code> is <code>null</code> or an unrecognized string,
     *         the default value {@link DatabaseMetaData#importedKeyNoAction}
     *         is assumed.
     */
    public static short fkRuleStringToShort(String s)
    {
        if (s==null || "NoAction".equals(s))
            return DatabaseMetaData.importedKeyNoAction;
        if ("Cascade".equals(s))
            return DatabaseMetaData.importedKeyCascade;
        if ("Restrict".equals(s))
            return DatabaseMetaData.importedKeyRestrict;
        if ("SetDefault".equals(s))
            return DatabaseMetaData.importedKeySetDefault;
        if ("SetNull".equals(s))
            return DatabaseMetaData.importedKeySetNull;
        
        log.warn("Invalid foreign key rule string ["+s+"] specified; assuming default value [NoAction].");
        
        return DatabaseMetaData.importedKeyNoAction;
    }
    
    /**
     * Parse rule string of a deleteRule or updateRule attribute of a foreignkey tag.
     * 
     * @param s {@link DatabaseMetaData#importedKeyNoAction},
     *          {@link DatabaseMetaData#importedKeyCascade},
     *          {@link DatabaseMetaData#importedKeyRestrict},
     *          {@link DatabaseMetaData#importedKeySetDefault} or
     *          {@link DatabaseMetaData#importedKeySetNull}.
     * @return <code>"NoAction"</code>, <code>"Cascade"</code>, 
     *          <code>"Restrict"</code>, <code>"SetDefault"</code> or
     *          <code>"SetNull"</code>.
     *         If <code>s</code> is an unrecognized integer,
     *         the default value <code>"NoAction"</code> is assumed.
     */
    public static String fkRuleShortToString(short s)
    {
        switch (s)
        {
        case DatabaseMetaData.importedKeyNoAction:
            return "NoAction";
        case DatabaseMetaData.importedKeyCascade:
            return "Cascade";
        case DatabaseMetaData.importedKeyRestrict:
            return "Restrict";
        case DatabaseMetaData.importedKeySetDefault:
            return "SetDefault";
        case DatabaseMetaData.importedKeySetNull:
            return "SetNull";
        }
                
        log.warn("Invalid foreign key rule value ["+s+"] specified; assuming default value [NoAction].");
        
        return "NoAction";
    }
}
