/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Created: 05.08.2008
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.helper;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipOutputStream;

import org.apache.commons.compress.compressors.bzip2.BZip2CompressorInputStream;
import org.apache.commons.compress.compressors.bzip2.BZip2CompressorOutputStream;

/**
 * Static functions for setting up compressed/uncompressed streams.
 * 
 * @author wglas
 */
public abstract class StreamHelper {

    /**
     * Creates an input stream, which possibly does decompression based on the
     * extension of the given filename.
     * 
     * @param filename The name of the file to open.
     * @return A {@link GZIPInputStream}, if the filename ends with <code>.gz</code>,
     *   a {@link BZip2CompressorInputStream}, if the filename ends with <code>.bz2</code>,
     *   {@link System#in} if filename is null or
     *   a bare {@link FileInputStream} in all other situations.
     * @throws IOException If the file does not exist or the headers of compressed
     *                     file are invalid.
     */
    public static InputStream makeInputStream(String filename) throws IOException {
        
        if (filename == null) return System.in;
        
        InputStream is = new FileInputStream(filename);
        
        if (filename.endsWith(".gz")) {
            
            is = new GZIPInputStream(is);
        }
        else if (filename.endsWith(".bz2")) {
            
            is = new BZip2CompressorInputStream(is);
        }
        
        return is;
    }
    
    public static OutputStream makeOutpuStream(String filename, Integer compression) throws IOException {
        
        if (filename==null) return System.out;
        
        OutputStream os = new FileOutputStream(filename);
        
        if (filename.endsWith(".gz")) {
            
            os = new GZIPOutputStream(os);
        }
        else if (filename.endsWith(".bz2")) {
            
            if (compression == null)
                os = new BZip2CompressorOutputStream(os);
            else
                os = new BZip2CompressorOutputStream(os,compression.intValue());
        }
        else if (filename.endsWith(".zip"))
        {
            os = new ZipOutputStream(os);
            ((ZipOutputStream) os).setLevel(java.util.zip.Deflater.BEST_COMPRESSION);
        }
         
        return os;
    }
}
