/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 03.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.helper;

import java.math.BigDecimal;
import java.sql.Array;
import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.Collection;

import org.clazzes.jdbc2xml.sql.SqlIdentifierMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Helper functions common for many SQL engines.
 * 
 * @author wglas
 */
public abstract class SQLHelper {

    private static final Logger log = LoggerFactory.getLogger(SQLHelper.class);
            
    /**
     * Quote the given string as defined by ISO SQL by doubling
     * single quotes. 
     * 
     * @param sb The buffer to put the result to.
     * @param s The String to quote.
     * @see #unquoteISOSqlString(String)
     */
    public static void quoteISOSqlString(StringBuffer sb, String s)
    {
        for (int i=0; i < s.length();++i)
        {
            char c = s.charAt(i);
            
            // double quotes.
            if (c == '\'')
                sb.append('\'');

            sb.append(c);   
        }
    }
    
    /**
     * Unquote the given string as defined by ISO SQL by doubling
     * single quotes. 
     * 
     * @param s The String to unquote.
     * @return the unquoted String.
     * @see #quoteISOSqlString(StringBuffer, String)
     */
    public static String unquoteISOSqlString(String s)
    {
        return s.replace("\'\'", "\'");
    }
    
    /**
     * Append an SQL type name together with a precision to a string buffer.
     * 
     * @param sb The buffer to put the result to. 
     * @param type The SQL name of the type.
     * @param prec The optional precision (field width).
     */
    public static void appendTypePrec(StringBuffer sb, String type, Integer prec)
    {
        sb.append(type);
        if (prec != null)
        {
            sb.append('(');
            sb.append(prec.intValue());
            sb.append(')');
        }
    }
    
    /**
     * Append an SQL type name together with a precision and a scale to a string buffer.
     * 
     * @param sb The buffer to put the result to.
     * @param type The SQL name of the type.
     * @param prec The optional precision (field width).
     * @param scale The optional scale (fractional digits).
     */
    public static void appendTypePrecScale(StringBuffer sb, String type, Integer prec, Integer scale)
    {
        sb.append(type);
        if (prec != null)
        {
            sb.append('(');
            sb.append(prec.intValue());
            
            if (scale != null) {
                sb.append(',');
                sb.append(scale.intValue());
            }
            
            sb.append(')');
        }
    }
    
    /**
     * Perform an SQL update statement.
     * 
     * @param connection The JDBC connection to use.
     * @param sql The SQL update statement.
     * @throws SQLException Upon database errors
     * 
     * @see Statement#executeUpdate(String)
     */
    public static void executeUpdate(Connection connection, String sql) throws SQLException
    {
        try(Statement statement = connection.createStatement()) {
        
            if (log.isTraceEnabled())
                log.trace("executeUpdate sql=[" + sql + "].");  
            try {
                statement.executeUpdate(sql);
            } catch (SQLException  e) {
                throw new SQLException("Error in query [" + sql + "]",e.getSQLState(),e.getErrorCode(),e);
             } 
        }
    }
    
    /**
     * Perform an SQL querie statement.
     * 
     * @param connection The JDBC connection to use.
     * @param sql The SQL querie statement
     * @return ResultSet the result returned by the querie
     * @throws SQLException Upon database errors
     * 
     * @see Statement#executeUpdate(String)
     */
    public static ResultSet executeQuerie(Connection connection, String sql) throws SQLException
    {
        try(Statement statement = connection.createStatement())
        {    
            if (log.isTraceEnabled())
                log.trace("executeUpdate sql=[" + sql + "].");
            ResultSet result;
            try {
                result = statement.executeQuery(sql);
                return result;
            } catch (SQLException  e) {
                throw new SQLException("Error in query [" + sql + "]",e.getSQLState(),e.getErrorCode(),e);
            } 
        }
    }
    
    /**
     * Copies a column from a result set to a placeholder of a PreparedStatment.
     * 
     * @param ps The prepared statement.
     * @param pc The column number of the column to be filled.
     * @param rs The result set from which to retrieve the data.
     * @param rc The column of the result set to be evaluated.
     * @throws SQLException 
     */
    public static void copyResultField(PreparedStatement ps, int pc, ResultSet rs, int rc) throws SQLException
    {
        ResultSetMetaData md = rs.getMetaData();
        
        int type = md.getColumnType(rc);
        
        switch (type)
        {
        case Types.ARRAY:
            Array a = rs.getArray(rc);
            if (rs.wasNull())
                ps.setNull(pc,type);
            else
                ps.setArray(pc,a);
            break;
            
        case Types.BIGINT:
        case Types.DECIMAL:
        case Types.NUMERIC:
        case Types.REAL:
            BigDecimal bd = rs.getBigDecimal(rc);
            if (rs.wasNull())
                ps.setNull(pc,type);
            else
                ps.setBigDecimal(pc,bd);
            break;
            
        case Types.BINARY:
        case Types.LONGVARBINARY:
        case Types.VARBINARY:
        case Types.BLOB:
            byte[] ba = rs.getBytes(rc);
            if (rs.wasNull())
                ps.setNull(pc,type);
            else
                ps.setBytes(pc,ba);
            break;
            
        case Types.BIT:
            byte by = rs.getByte(rc);
            if (rs.wasNull())
                ps.setNull(pc,type);
            else
                ps.setByte(pc,by);
            break;
            
        case Types.BOOLEAN:
            boolean bo = rs.getBoolean(rc);
            if (rs.wasNull())
                ps.setNull(pc,type);
            else
                ps.setBoolean(pc,bo);
            break;
            
            /* This does not seem to work with some JDBC drivers, so copy blobs as byte arrays.  
        case Types.BLOB:
            Blob blob = rs.getBlob(rc);
            if (rs.wasNull())
                ps.setNull(pc,type);
            else
                ps.setBlob(pc,blob);
            break;
           */ 
        case Types.CHAR:
        case Types.CLOB:
        case Types.LONGVARCHAR:
        case Types.VARCHAR:
            String str = rs.getString(rc);
            if (rs.wasNull())
                ps.setNull(pc,type);
            else
                ps.setString(pc,str);
            break;
            
            /* This does not seem to work with some JDBC drivers, so copy clobs as strings.  
        case Types.CLOB:
            Clob clob = rs.getClob(rc);
            if (rs.wasNull())
                ps.setNull(pc,type);
            else
                ps.setClob(pc,clob);
            break;
            */
            
        case Types.DATE:
            Date dat = rs.getDate(rc);
            if (rs.wasNull())
                ps.setNull(pc,type);
            else
                ps.setDate(pc,dat);
            break;
            
        case Types.DOUBLE:
        case Types.FLOAT:
            double dbl = rs.getDouble(rc);
            if (rs.wasNull())
                ps.setNull(pc,type);
            else
                ps.setDouble(pc,dbl);
            break;

        case Types.INTEGER:
        case Types.SMALLINT:
        case Types.TINYINT:
            long intg = rs.getLong(rc);
            if (rs.wasNull())
                ps.setNull(pc,type);
            else
                ps.setLong(pc,intg);
            break;
            
        case Types.REF:
            Ref ref = rs.getRef(rc);
            if (rs.wasNull())
                ps.setNull(pc,type);
            else
                ps.setRef(pc,ref);
            break;
            
        case Types.TIME:
            Time time = rs.getTime(rc);
            if (rs.wasNull())
                ps.setNull(pc,type);
            else
                ps.setTime(pc,time);
            break;
            
        case Types.TIMESTAMP:
        case Types.TIMESTAMP_WITH_TIMEZONE:
            Timestamp timestamp = rs.getTimestamp(rc);
            if (rs.wasNull())
                ps.setNull(pc,type);
            else
                ps.setTimestamp(pc,timestamp);
            break;

        default:
            Object obj = rs.getObject(rc);
            if (rs.wasNull())
                ps.setNull(pc,type);
            else
                ps.setObject(pc,obj);
            break;
        }
    }
    /**
     * Simply concatenate a collection of SQL identifiers by using a comma
     * as separator. No quoting is undertaken, as this function is usually
     * used for SQL column names.
     * 
     * @param sb The string buffer to write the concatenated list of string to.
     * @param strings A collection of strings.
     * @param mapper A mapper for transforming SQL identifiers to their external form.
     */
    public static void joinIdentifiers(StringBuffer sb, Collection<String> strings,
            SqlIdentifierMapper mapper)
    {
        boolean firstItem = true;
        
        for (String s : strings)
        {
            if (firstItem)
                firstItem = false;
            else
                sb.append(',');

            sb.append(mapper.toExternal(s));
        }
    }
    
    /**
     * Simply concatenate a collection of SQL identifiers by using a comma
     * as separator. No quoting is undertaken, as this function is usually
     * used for SQL column names.
     * 
     * @param strings A collection of strings.
     * @param mapper A mapper for transforming SQL identifiers to their external form.
     * @return A string containing all strings delimited by a comma.
     */
    public static String joinIdentifiers(Collection<String> strings,
            SqlIdentifierMapper mapper)
    {
        StringBuffer sb = new StringBuffer();
        joinIdentifiers(sb,strings,mapper);
        return sb.toString();
    }

}
