/***********************************************************
 * $Id$
 * 
 * JDB to XML bridge of the clazzes project.
 * http://www.clazzes.org
 *
 * Created: 26.11.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.jdbc2xml.deserialization.impl;

import java.sql.Types;
import java.util.TimeZone;

import org.clazzes.jdbc2xml.deserialization.DeserializationHandler;
import org.clazzes.jdbc2xml.deserialization.DeserializationHandlerFactory;
import org.clazzes.jdbc2xml.schema.ColumnInfo;

/**
 * @author wglas
 *
 */
public class DeserializationHandlerFactoryImpl extends
        DeserializationHandlerFactory {

    /* (non-Javadoc)
     * @see org.clazzes.jdbc2xml.deserialization.DeserializationHandlerFactory#newDeserializationHandler(org.clazzes.jdbc2xml.schema.ColumnInfo, java.util.TimeZone)
     */
    public DeserializationHandler newDeserializationHandler(ColumnInfo ci, TimeZone timeZone) {
        
        switch (ci.getType())
        {
        case Types.SMALLINT:
        case Types.INTEGER:
        case Types.TINYINT:
            return new IntegerDeserializationHandler(ci.getType());
        
        case Types.NUMERIC:
            // for number with no decimal places and less than or equal 18 ddecimal places,
            // we use a deserialization handler based on 64bit integers.
            if ((ci.getScale() == null || ci.getScale() == 0) && ci.getPrecision() != null && ci.getPrecision() <= 18)
                return new IntegerDeserializationHandler(ci.getType());
            else
                return new BigDecimalDeserializationHandler(ci.getType());
        
        case Types.FLOAT:
            return new FloatDeserializationHandler(ci.getType());
        
        case Types.DOUBLE:
            return new DoubleDeserializationHandler(ci.getType());
        
        case Types.LONGVARCHAR:
        case Types.VARCHAR:
        case Types.CHAR:
        case Types.CLOB:
            return new StringDeserializationHandler(ci.getType());
        
        case Types.BIGINT:
        case Types.DECIMAL:
        case Types.REAL:
            return new BigDecimalDeserializationHandler(ci.getType());
            
        case Types.DATE:
            return new DateDeserializationHandler(timeZone,ci.getType() );
            
        case Types.TIMESTAMP:
        case Types.TIMESTAMP_WITH_TIMEZONE:
        // In the future, we could possibly use a TimestampWithTimezoneDeserializationHandler for this type.
        // However, as of now, it might cause problems with the API due to requiring the engine metadata to correctly
        // be able to read the configured time zone of the column.
            return new TimestampDeserializationHandler(timeZone, ci.getType());
        
        case Types.LONGVARBINARY:
        case Types.VARBINARY:
        case Types.BINARY:
        case Types.BLOB:
            return new BinaryDeserializationHandler(ci.getType());

        case Types.BOOLEAN:
        case Types.BIT:
            return new BooleanDeserializationHandler(ci.getType());
        }
        
        return null;
    }

}
