/***********************************************************
 *
 * FancyMail standalone OSGi server of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.fancymail.server.service.impl;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.Arrays;

import javax.sql.DataSource;

import org.clazzes.jdbc2xml.schema.ColumnInfo;
import org.clazzes.jdbc2xml.schema.ForeignKeyInfo;
import org.clazzes.jdbc2xml.schema.ISchemaEngine;
import org.clazzes.jdbc2xml.schema.IndexInfo;
import org.clazzes.jdbc2xml.schema.PrimaryKeyInfo;
import org.clazzes.jdbc2xml.schema.TableInfo;
import org.clazzes.util.aop.DAOException;
import org.clazzes.util.aop.jdbc.JdbcDAOSupport;
import org.clazzes.util.aop.jdbc.JdbcPreparedStatementAction;
import org.clazzes.util.aop.jdbc.JdbcStatementAction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Maintain the database schema.
 */
public class SchemaChecker extends JdbcDAOSupport {

    private static final String CURRENT_VERSION = "1.2";

    private static final Logger log = LoggerFactory.getLogger(SchemaChecker.class);

    private static final String SCHEMA_VERSION_TABLE = "FANCYMAIL_SCHEMA_VERSION";

    private final ISchemaEngine schemaEngine;


    private String commentColumnName;

    public SchemaChecker(ISchemaEngine schemaEngine) {
        this.commentColumnName = "DESCRIPTION";
        this.schemaEngine = schemaEngine;
    }

    @Override
    protected Connection getConnection() {

        return this.schemaEngine.getConnection();
    }

    private TableInfo createSchemaVersionTable() throws SQLException {

        log.info("Creating table ["+SCHEMA_VERSION_TABLE+"].");
        TableInfo schemaVersion = new TableInfo(SCHEMA_VERSION_TABLE);

        schemaVersion.addColumn(new ColumnInfo("ID",Types.INTEGER,6,null,false,null));
        schemaVersion.addColumn(new ColumnInfo("VERSION",Types.VARCHAR,20,null,false,null));
        schemaVersion.addColumn(new ColumnInfo("UPDATED",Types.TIMESTAMP,null,null,false,null));
        schemaVersion.addColumn(new ColumnInfo("DESCRIPTION",Types.VARCHAR,256,null,false,null));

        schemaVersion.setPrimaryKey(new PrimaryKeyInfo(SCHEMA_VERSION_TABLE+"_PK","ID"));

        this.schemaEngine.createTable(schemaVersion,true);
        this.schemaEngine.commit();

        if (log.isDebugEnabled())
            log.debug("Successfully created table ["+SCHEMA_VERSION_TABLE+"].");

       return schemaVersion;
    }

    private Integer getMaxIdFromTable(final String table) {

        return this.performWithStatement(new JdbcStatementAction<Integer>() {

            @Override
            public Integer perform(Statement statement) throws Exception {
                ResultSet rs = statement.executeQuery("select MAX(ID) from "+table);

                if (rs.next()) {

                    int v = rs.getInt(1);

                    if (rs.wasNull())
                        return null;
                    else
                        return Integer.valueOf(v);
                }
                else
                    return null;
            }
        });
    }

    private Integer getLastId() {

        if (log.isDebugEnabled())
            log.debug("Fetching last schema version Id from ["+SCHEMA_VERSION_TABLE+"]...");

        Integer ret = this.getMaxIdFromTable(SCHEMA_VERSION_TABLE);

        if (log.isDebugEnabled())
            log.debug("Last schema version Id from ["+SCHEMA_VERSION_TABLE+"] is ["+ret+"].");

        return ret;
    }

    private String getLastVersion(final int lastId) {

        if (log.isDebugEnabled())
            log.debug("Fetching last schema version from ["+SCHEMA_VERSION_TABLE+"]...");

        String ret = this.performWithPreparedStatement(
                "select VERSION from "+SCHEMA_VERSION_TABLE+" where ID=?",
                new JdbcPreparedStatementAction<String>() {

            @Override
            public String perform(PreparedStatement statement) throws Exception {

                statement.setInt(1,lastId);
                ResultSet rs = statement.executeQuery();

                if (!rs.next()) return null;
                return rs.getString(1);
            }
        });

        log.info("Last schema version from ["+SCHEMA_VERSION_TABLE+"] is ["+ret+"].");

        return ret;
    }

    private void saveVersion(final int id, final String version, final String comment) {

        final Timestamp timestamp = new Timestamp(System.currentTimeMillis());

        this.performWithPreparedStatement(
                "insert into "+SCHEMA_VERSION_TABLE+"(ID,VERSION,UPDATED,"+this.commentColumnName+") values (?,?,?,?)",
                new JdbcPreparedStatementAction<Void>() {

            @Override
            public Void perform(PreparedStatement statement) throws Exception {

                statement.setInt(1,id);
                statement.setString(2,version);
                statement.setTimestamp(3,timestamp);
                statement.setString(4,comment);
                statement.executeUpdate();
                return null;
            }
        });
    }

    private void createIdGenerator() throws SQLException {

        TableInfo idGenerator=new TableInfo("ID_GENERATOR");
        idGenerator.setColumns(Arrays.asList(
                       new ColumnInfo("NEXT_ID",Types.BIGINT,20,null,false,null)
        ));

        idGenerator.setPrimaryKey(new PrimaryKeyInfo("ID_GENERATOR_PK","NEXT_ID"));

        this.schemaEngine.createTable(idGenerator,true);
    }

    private String initialize_current_1_2(int id) throws SQLException {

        log.info("Creating schema version [{}]...",CURRENT_VERSION);

        this.createIdGenerator();

        // ===================================== EMAIL =========================================

        TableInfo emailSender = new TableInfo("EMAIL_SENDER");

        emailSender.addColumn(new ColumnInfo("ID",Types.INTEGER,18,null,false,null));
        emailSender.addColumn(new ColumnInfo("ADDRESS",Types.VARCHAR,80,null,false,null));
        emailSender.addColumn(new ColumnInfo("PERSONAL_NAME",Types.VARCHAR,80,null,true,null));
        emailSender.addColumn(new ColumnInfo("REPLY_TO",Types.VARCHAR,80,null,true,null));

        emailSender.setPrimaryKey(new PrimaryKeyInfo("EMAIL_SENDER_PK","ID"));
        emailSender.addIndex(new IndexInfo("EMAIL_SENDER_ADDRESS_IDX","ADDRESS",true,null));

        this.schemaEngine.createTable(emailSender,true);

        TableInfo email = new TableInfo("EMAIL");

        email.addColumn(new ColumnInfo("ID",Types.INTEGER,20,null,false,null));
        email.addColumn(new ColumnInfo("SENDER_ID",Types.INTEGER,18,null,false,null));
        email.addColumn(new ColumnInfo("CREATED",Types.TIMESTAMP,null,null,false,null));
        email.addColumn(new ColumnInfo("SENT",Types.TIMESTAMP,null,null,true,null));
        email.addColumn(new ColumnInfo("SUBJECT",Types.VARCHAR,200,null,false,null));
        email.addColumn(new ColumnInfo("BODY",Types.LONGVARCHAR,65535,null,false,null));
        email.addColumn(new ColumnInfo("STATUS",Types.INTEGER,6,null,false,null));
        email.addColumn(new ColumnInfo("ERROR_COUNT",Types.INTEGER,6,null,false,null));
        email.addColumn(new ColumnInfo("LAST_ERROR_TEXT",Types.VARCHAR,200,null,true,null));

        email.setPrimaryKey(new PrimaryKeyInfo("EMAIL_PK","ID"));
        email.addIndex(new IndexInfo("EMAIL_CREATED_IDX","CREATED",false,null));
        email.addIndex(new IndexInfo("EMAIL_SENT_IDX","SENT",false,null));
        email.addIndex(new IndexInfo("EMAIL_STATUS_IDX","STATUS",false,null));
        email.addIndex(new IndexInfo("EMAIL_SENDER_IDX","SENDER_ID,CREATED",false,null));

        email.addForeignKey(new ForeignKeyInfo("EMAIL_EMAIL_SENDER_FK",
                "SENDER_ID","EMAIL_SENDER","ID"));

        this.schemaEngine.createTable(email,true);

        TableInfo emailRecipient = new TableInfo("EMAIL_RECIPIENT");

        emailRecipient.addColumn(new ColumnInfo("ID",Types.INTEGER,18,null,false,null));
        emailRecipient.addColumn(new ColumnInfo("EMAIL_ID",Types.INTEGER,18,null,false,null));
        emailRecipient.addColumn(new ColumnInfo("ENTITLEMENT",Types.INTEGER,2,null,false,null));
        emailRecipient.addColumn(new ColumnInfo("ADDRESS",Types.VARCHAR,80,null,false,null));
        emailRecipient.addColumn(new ColumnInfo("PERSONAL_NAME",Types.VARCHAR,80,null,true,null));

        emailRecipient.setPrimaryKey(new PrimaryKeyInfo("EMAIL_RECIPIENT_PK","ID"));
        emailRecipient.addIndex(new IndexInfo("EMAIL_RECIPIENT_ADDRESS_IDX","ADDRESS",false,null));
        emailRecipient.addForeignKey(new ForeignKeyInfo("EMAIL_RECIPIENT_EMAIL_FK",
                "EMAIL_ID","EMAIL","ID"));

        this.schemaEngine.createTable(emailRecipient,true);

        // ===================================== SMS =========================================

        this.createSmsTables_1_0();

        // ===================================== attachments =========================================

        this.createAttachmentTable_1_1();

        // advanced send diagnostics.

        this.addSendDiagnostics_1_2();

        this.saveVersion(id,CURRENT_VERSION,"Initially create tables EMAIL_SENDER, EMAIL, EMAIL_RECIPIENT, EMAIL_ATTACHMENT, SMS_SENDER, SMS, SMS_RECIPIENT and ID_GENERATOR.");

        this.schemaEngine.commit();
        log.info("Successfully created schema version [{}}].",CURRENT_VERSION);

        return CURRENT_VERSION;
    }

    private void createSmsTables_1_0() throws SQLException {
        TableInfo smsSender = new TableInfo("SMS_SENDER");

        smsSender.addColumn(new ColumnInfo("ID",Types.INTEGER,18,null,false,null));
        smsSender.addColumn(new ColumnInfo("SOURCE_NUMBER",Types.VARCHAR,20,null,false,null));
        smsSender.addColumn(new ColumnInfo("PERSONAL_NAME",Types.VARCHAR,80,null,true,null));

        smsSender.setPrimaryKey(new PrimaryKeyInfo("SMS_SENDER_PK","ID"));
        smsSender.addIndex(new IndexInfo("SMS_SENDER_SOURCE_NUMBER_IDX","SOURCE_NUMBER",true,null));

        this.schemaEngine.createTable(smsSender,true);

        TableInfo sms = new TableInfo("SMS");

        sms.addColumn(new ColumnInfo("ID",Types.INTEGER,20,null,false,null));
        sms.addColumn(new ColumnInfo("SENDER_ID",Types.INTEGER,18,null,false,null));
        sms.addColumn(new ColumnInfo("CREATED",Types.TIMESTAMP,null,null,false,null));
        sms.addColumn(new ColumnInfo("SENT",Types.TIMESTAMP,null,null,true,null));
        sms.addColumn(new ColumnInfo("TEXT",Types.VARCHAR,255,null,false,null));
        sms.addColumn(new ColumnInfo("STATUS",Types.INTEGER,6,null,false,null));
        sms.addColumn(new ColumnInfo("ERROR_COUNT",Types.INTEGER,6,null,false,null));
        sms.addColumn(new ColumnInfo("LAST_ERROR_TEXT",Types.VARCHAR,200,null,true,null));

        sms.setPrimaryKey(new PrimaryKeyInfo("SMS_PK","ID"));
        sms.addIndex(new IndexInfo("SMS_CREATED_IDX","CREATED",false,null));
        sms.addIndex(new IndexInfo("SMS_SENT_IDX","SENT",false,null));
        sms.addIndex(new IndexInfo("SMS_STATUS_IDX","STATUS",false,null));
        sms.addIndex(new IndexInfo("SMS_SENDER_IDX","SENDER_ID,CREATED",false,null));

        sms.addForeignKey(new ForeignKeyInfo("SMS_SMS_SENDER_FK",
                "SENDER_ID","SMS_SENDER","ID"));

        this.schemaEngine.createTable(sms,true);

        TableInfo smsRecipient = new TableInfo("SMS_RECIPIENT");

        smsRecipient.addColumn(new ColumnInfo("ID",Types.INTEGER,18,null,false,null));
        smsRecipient.addColumn(new ColumnInfo("SMS_ID",Types.INTEGER,18,null,false,null));
        smsRecipient.addColumn(new ColumnInfo("DEST_NUMBER",Types.VARCHAR,20,null,false,null));
        smsRecipient.addColumn(new ColumnInfo("PERSONAL_NAME",Types.VARCHAR,80,null,true,null));

        smsRecipient.setPrimaryKey(new PrimaryKeyInfo("SMS_RECIPIENT_PK","ID"));
        smsRecipient.addIndex(new IndexInfo("SMS_RECIPIENT_DEST_NUMBER_IDX","DEST_NUMBER",false,null));
        smsRecipient.addForeignKey(new ForeignKeyInfo("SMS_RECIPIENT_EMAIL_FK",
                "SMS_ID","SMS","ID"));

        this.schemaEngine.createTable(smsRecipient,true);
    }

    private void createAttachmentTable_1_1() throws SQLException {

        TableInfo email = this.schemaEngine.fetchTableInfo("EMAIL",null);

        ColumnInfo idColumn = email.getColumnInfo("ID");

        TableInfo attachment = new TableInfo("EMAIL_ATTACHMENT");

        attachment.addColumn(new ColumnInfo("ID",Types.INTEGER,20,null,false,null));
        // set email_id column to the same taype than email.id in order to avoid
        // mysql errno 150 "Foreign key constraint is incorrectly formed"
        attachment.addColumn(new ColumnInfo("EMAIL_ID",idColumn.getType(),idColumn.getPrecision(),null,false,null));
        attachment.addColumn(new ColumnInfo("MIME_TYPE",Types.VARCHAR,40,null,false,null));
        attachment.addColumn(new ColumnInfo("PRETTY_NAME",Types.VARCHAR,255,null,false,null));
        attachment.addColumn(new ColumnInfo("CONTENT",Types.BLOB,null,null,false,null));

        attachment.setPrimaryKey(new PrimaryKeyInfo("EMAIL_ATT_PK","ID"));

        attachment.addForeignKey(new ForeignKeyInfo("EMAIL_ATT_EMAIL_FK",
                "EMAIL_ID","EMAIL","ID"));

        this.schemaEngine.createTable(attachment,true);
    }

    private void addSendDiagnostics_1_2() throws SQLException {

        TableInfo email = this.schemaEngine.fetchTableInfo("EMAIL",null);

        this.schemaEngine.addColumn(email,
                new ColumnInfo("SENDING",Types.TIMESTAMP,null,null,true,null));
        this.schemaEngine.addColumn(email,
                new ColumnInfo("LAST_ERROR_EXCEPTION",Types.VARCHAR,256,null,true,null));

        this.schemaEngine.addIndex(email,
                new IndexInfo("EMAIL_SENDING_IDX","SENDING",false,null));

        TableInfo sms = this.schemaEngine.fetchTableInfo("SMS",null);

        this.schemaEngine.addColumn(sms,
                new ColumnInfo("SENDING",Types.TIMESTAMP,null,null,true,null));
        this.schemaEngine.addColumn(sms,
                new ColumnInfo("LAST_ERROR_EXCEPTION",Types.VARCHAR,256,null,true,null));

        this.schemaEngine.addIndex(sms,
                new IndexInfo("SMS_SENDING_IDX","SENDING",false,null));
    }

    // update from 0.1 -> 0.2
    private String update_0_2(int id) throws SQLException {

        log.info("Updating schema to version [0.2]...");

        Integer maxId = this.getMaxIdFromTable("EMAIL");
        Integer maxId2 = this.getMaxIdFromTable("EMAIL_RECIPIENT");
        Integer maxId3 = this.getMaxIdFromTable("EMAIL_SENDER");

        if (maxId == null || (maxId2 != null && maxId2.intValue() > maxId.intValue())) {
            maxId = maxId2;
        }

        if (maxId == null || (maxId3 != null && maxId3.intValue() > maxId.intValue())) {
            maxId = maxId3;
        }

        TableInfo emailSender = this.schemaEngine.fetchTableInfo("EMAIL_SENDER",null);
        this.schemaEngine.changeColumn(emailSender,"ID",new ColumnInfo("ID",Types.INTEGER,20,null,false,null));

        TableInfo email = this.schemaEngine.fetchTableInfo("EMAIL",null);
        this.schemaEngine.changeColumn(email,"ID",new ColumnInfo("ID",Types.INTEGER,20,null,false,null));

        TableInfo emailRecipient = this.schemaEngine.fetchTableInfo("EMAIL_RECIPIENT",null);
        this.schemaEngine.changeColumn(emailRecipient,"ID",new ColumnInfo("ID",Types.INTEGER,20,null,false,null));

        this.createIdGenerator();

        if (maxId != null) {

            final int maxIdFinal = ((maxId.intValue() + 100) / 100) * 100;

            log.info("Setting next ID of ID_GENARATOR to ["+maxIdFinal+"].");

            this.performWithPreparedStatement("insert into ID_GENERATOR (NEXT_ID) values(?)",
                    new JdbcPreparedStatementAction<Void>() {

                @Override
                public Void perform(PreparedStatement statement) throws Exception {

                    statement.setInt(1,maxIdFinal);
                    statement.executeUpdate();
                    return null;
                }
            });
        }

        this.saveVersion(id,"0.2","Remove auto_increment from EMAIL_SENDER, EMAIL and EMAIL_RECIPIENT, add table ID_GENERATOR.");

        this.schemaEngine.commit();
        log.info("Successfully updated schema to version [0.2].");

        return "0.2";
    }

    private String update_1_0(int id) throws SQLException {
        log.info("Updating schema to version [1.0]...");

        this.createSmsTables_1_0();

        this.saveVersion(id,"1.0","Adding tables SMS, SMS_SENDER, SMS_RECIPIENT");

        log.info("Successfully updated schema to version [1.0].");

        return "1.0";
    }

    private String update_1_1(int id) throws SQLException {
        log.info("Updating schema to version [1.1]...");

        this.createAttachmentTable_1_1();

        this.saveVersion(id,"1.1","Adding tables EMAIL_ATTACHMENT");

        log.info("Successfully updated schema to version [1.1].");

        return "1.1";
    }

    private String update_1_2(int id) throws SQLException {
        log.info("Updating schema to version [1.2]...");

        this.addSendDiagnostics_1_2();

        this.saveVersion(id,"1.2","Adding columns SENDING and LAST_ERROR_EXCEPTION to EMAIL and SMS");

        log.info("Successfully updated schema to version [1.2].");

        return "1.2";
    }

    private void checkSchema() throws SQLException {

        TableInfo schemaVersion = null;

        try {
            if (log.isDebugEnabled())
                log.debug("Checking for table ["+SCHEMA_VERSION_TABLE+"]...");

            schemaVersion = this.schemaEngine.fetchTableInfo(SCHEMA_VERSION_TABLE,null);

            if (log.isDebugEnabled())
                log.debug("Table ["+SCHEMA_VERSION_TABLE+"] exists.");
        }
        catch (SQLException e) {
            if (log.isDebugEnabled())
                log.debug("Table ["+SCHEMA_VERSION_TABLE+"] has not been found",e);
        }

        Integer lastId;

        if (schemaVersion == null) {

            // oracle does not like "COMMENT" as column name, so we switched to "DESCRIPTION" as of version 1.0.0
            schemaVersion = this.createSchemaVersionTable();
            lastId = null;
        }
        else {

            // detect legacy databases, where the schem update remarks are save in column
            if (schemaVersion.getColumnInfo("COMMENT") != null) {
                log.info("Detected pre 1.0.0 database history table, using COMMENT as DESCRIPTION column.");
                this.commentColumnName = "COMMENT";
            }

            lastId = this.getLastId();
        }

        String lastVersion = null;
        int id;

        if (lastId != null) {
            lastVersion = this.getLastVersion(lastId.intValue());
            id = lastId.intValue();
        }
        else {
            id = 0;
        }

        try {

            if (lastVersion == null) {
                lastVersion = this.initialize_current_1_2(++id);
            } else {
                if (lastVersion.equals("0.1")) {
                    lastVersion = this.update_0_2(++id);
                }
                if (lastVersion.equals("0.2")) {
                    lastVersion = this.update_1_0(++id);
                }
                if (lastVersion.equals("1.0")) {
                    lastVersion = this.update_1_1(++id);
                }
                if (lastVersion.equals("1.1")) {
                    lastVersion = this.update_1_2(++id);
                }
            }

            if (!CURRENT_VERSION.equals(lastVersion)) {
                throw new DAOException("fancymail schema has unknown version ["+lastVersion+"]");
            }

            log.info("Schema version is now ["+lastVersion+"] and up to date.");
        }
        catch (Throwable e) {
            this.schemaEngine.rollback();
            throw e;
        }
    }

    public void provideDatasource(DataSource dataSource) {

        try (Connection conn = dataSource.getConnection()) {

            this.schemaEngine.setConnection(conn);
            this.checkSchema();
        } catch (SQLException e) {
            throw new DAOException("Error checking fancymail DB schema",e);
        }
    }

}
