package org.clazzes.fancymail.server.service.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import org.clazzes.fancymail.server.api.SMSDTO;
import org.clazzes.fancymail.server.api.SMSDestinationDTO;
import org.clazzes.fancymail.server.api.SMSSenderDTO;
import org.clazzes.fancymail.server.dao.SMSDAO;
import org.clazzes.fancymail.server.dao.SMSSenderDAO;
import org.clazzes.fancymail.server.entities.EMail;
import org.clazzes.fancymail.server.entities.SMS;
import org.clazzes.fancymail.server.entities.SMSRecipient;
import org.clazzes.fancymail.server.entities.SMSSender;
import org.clazzes.fancymail.server.service.SMSService;
import org.clazzes.fancymail.sms.ISMS;
import org.clazzes.fancymail.sms.SMSException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SMSServiceImpl implements SMSService {

    private static final Logger log = LoggerFactory.getLogger(SMSServiceImpl.class);

    private SMSSenderDAO smsSenderDAO;
    public void setSmsSenderDAO(SMSSenderDAO smsSenderDAO) {
        this.smsSenderDAO = smsSenderDAO;
    }

    private SMSDAO smsDAO;
    public void setSmsDAO(SMSDAO smsDAO) {
        this.smsDAO = smsDAO;
    }

    private int maxBulkSize;
    public void setMaxBulkSize(int maxBulkSize) {
        this.maxBulkSize = maxBulkSize;
    }

    private int maxRetries;
    public void setMaxRetries(int maxRetries) {
        this.maxRetries = maxRetries;
    }

    private int staleTimeoutMinutes;
    public void setStaleTimeoutMinutes(int staleTimeoutMinutes) {
        this.staleTimeoutMinutes = staleTimeoutMinutes;
    }

    public SMSSender getSenderBySourceNumber(String sourceNumber) {
        return this.smsSenderDAO.getBySourceNumber(sourceNumber);
    }

    public List<SMSSender> getAllSMSSenders() {
        return this.smsSenderDAO.getAll();
    }

    public SMSSender insertSender(SMSSenderDTO sender) {
        SMSSender entity = new SMSSender();

        entity.setPersonalName(sender.getPersonalName());
        entity.setSourceNumber(sender.getSourceNumber());

        return this.smsSenderDAO.save(entity);
    }

    public SMSSender updateSender(SMSSenderDTO sender) {
        SMSSender entity = this.smsSenderDAO.getBySourceNumber(sender.getSourceNumber());
        entity.setPersonalName(sender.getPersonalName());
        this.smsSenderDAO.update(entity);
        return entity;
    }

    public void updateSenders(Collection<SMSSenderDTO> senders) {
        for (SMSSenderDTO sender:senders) {
            this.updateSender(sender);
        }
    }

    public SMS insertSMS(SMSDTO sms) {
        SMS entity = new SMS();
        entity.setSender(this.getSenderBySourceNumber(sms.getSender()));
        entity.setCreated(new Date());
        entity.setText(sms.getText());
        entity.setStatus(SMS.SMS_STATUS_TOSEND);

        List<SMSRecipient> recipients = new ArrayList<SMSRecipient>();
        for (SMSDestinationDTO destinationDTO : sms.getDestinations()) {
            SMSRecipient recipient = new SMSRecipient();
            recipient.setDestinationNumber(destinationDTO.getDestNumber());
            recipient.setPersonalName(destinationDTO.getPersonalName());

            recipients.add(recipient);
        }

        entity.setRecipients(recipients);
        return this.smsDAO.save(entity);
    }

    public Collection<? extends ISMS> getSMSesToSend() throws SMSException {

        List<SMS> ret;

        if (this.staleTimeoutMinutes > 0) {

            ret = this.smsDAO.getAllByStatusForUpdate(EMail.MAIL_STATUS_TOSEND,this.maxBulkSize);

            if (ret == null || ret.size() <= 0) {
                ret = this.smsDAO.getAllByStatusForUpdate(EMail.MAIL_STATUS_SENDERROR,this.maxBulkSize);
            }

            Date now = new Date();

            Date watermark = new Date(now.getTime()-this.staleTimeoutMinutes*60000L);

            if (ret == null || ret.size() <= 0) {
                ret = this.smsDAO.getStaleSending(watermark,this.maxBulkSize);
            }

            ret.stream().forEach(sms -> sms.setSendingState(now));

            this.smsDAO.updateBatch(ret);
        }
        else {
            ret = this.smsDAO.getAllByStatus(EMail.MAIL_STATUS_TOSEND,this.maxBulkSize);

            if (ret == null || ret.size() <= 0) {
                ret = this.smsDAO.getAllByStatus(EMail.MAIL_STATUS_SENDERROR,this.maxBulkSize);
            }
        }

        if (ret.size() > 0) {
          log.info(ret
                 .stream()
                 .map(n -> String.valueOf(n))
                 .collect(Collectors.joining("\n  ", "SMSes to process are [", "]")));
        } else if (log.isDebugEnabled()) {
            log.debug("Found no SMSes to process");
        }

        return ret;
    }

    public void reportSMSTransmissionAttempt(ISMS iSms) {
        SMS sms = (SMS)iSms;

        if (sms.getStatus() == SMS.SMS_STATUS_SENDERROR &&
                sms.getErrorCount() > this.maxRetries) {
            sms.setStatus(SMS.SMS_STATUS_FAILED);
        }

        this.smsDAO.update(sms);
    }

    @Override
    public int deleteOutdatedSMSs(Date watermark) {

        List<Long> ids = this.smsDAO.getOutdatedIds(watermark);

        this.smsDAO.deleteByIds(ids);

        return ids.size();
    }

    public List<SMS> getAllSMSInRange(Date from, Date to) {
        return this.smsDAO.getAllInRange(from, to);
    }

    public List<SMS> getAllSMSInRangeForSender(String sender, Date from, Date to) {
        return this.smsDAO.getAllForSender(sender, from, to);
    }

    public List<SMS> getAllSMSInRangeForRecipient(String recipient, Date from, Date to) {
        return this.smsDAO.getAllForRecipient(recipient, from, to);
    }

    public List<SMS> getAllSMSInRangeForSenderRecipient(String sender, String recipient, Date from, Date to) {
        return this.smsDAO.getAllForSenderRecipient(sender, recipient, from, to);
    }
}
