/***********************************************************
 * $Id$
 * 
 * FancyMail standalone OSGi server of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.fancymail.server.service.impl;

import java.util.Calendar;
import java.util.Date;
import java.util.Timer;
import java.util.TimerTask;

import org.clazzes.fancymail.server.service.EMailService;
import org.clazzes.fancymail.server.service.SMSService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This bean deletes outdated e-mails at the specified time of day.
 */
public class GarbageCollector {

    private static final Logger log = LoggerFactory.getLogger(GarbageCollector.class); 
    
    // java.util.Timer is sufficient, because spanning a separate thread is
    // not necessary for a daily job.  
    private Timer timer;
    private int minutesOfDay;
    // string representation of minutesOfDay in the form hh:mm
    private String timeOfDay;
    private int monthsToKeep;
    private EMailService mailService;
    private SMSService smsService;
    
    private class CleanupTask extends TimerTask {

        @Override
        public void run() {
            
            Calendar c = Calendar.getInstance();
            c.set(Calendar.HOUR_OF_DAY,0);
            c.set(Calendar.MINUTE,0);
            c.set(Calendar.SECOND,0);
            c.add(Calendar.MONTH,-GarbageCollector.this.monthsToKeep);
            
            Date watermark = c.getTime();
            
            if (log.isDebugEnabled())
                log.debug("Garbage collecting mails generated before  ["+watermark+"]...");
            
            try {
                int n = GarbageCollector.this.mailService.deleteOutdatedEMails(watermark);
                
                if (n == 1) {
                    log.info("Successfully garbage collected [1] mail generated before  ["+watermark+"].");
                }
                else if (n > 1) {
                    log.info("Successfully garbage collected ["+n+"] mails generated before  ["+watermark+"].");
                }
                else {
                    if (log.isDebugEnabled()) {
                        log.debug("No mails found, which have been generated before ["+watermark+"].");
                    }
                }
            }
            catch (Throwable e) {
                log.error("Error garbage collecting mails generated before  ["+watermark+"]",e);
            }
            
            if (log.isDebugEnabled())
                log.debug("Garbage collecting SMSes generated before  ["+watermark+"]...");
            
            try {
                int n = GarbageCollector.this.smsService.deleteOutdatedSMSs(watermark);
                
                if (n == 1) {
                    log.info("Successfully garbage collected [1] SMS generated before  ["+watermark+"].");
                }
                else if (n > 1) {
                    log.info("Successfully garbage collected ["+n+"] SMSes generated before  ["+watermark+"].");
                }
                else {
                    if (log.isDebugEnabled()) {
                        log.debug("No SMSes found, which have been generated before ["+watermark+"].");
                    }
                }
            }
            catch (Throwable e) {
                log.error("Error garbage collecting mails generated before  ["+watermark+"]",e);
            }

            GarbageCollector.this.scheduleCleanupTask();
        }
    }
    
    public GarbageCollector() {
        this.timeOfDay = "0:00";
        this.monthsToKeep = 36;
    }
    
    private void scheduleCleanupTask() {
        
        Calendar c = Calendar.getInstance();
        c.set(Calendar.HOUR_OF_DAY,this.minutesOfDay/60);
        c.set(Calendar.MINUTE,this.minutesOfDay%60);
        c.set(Calendar.SECOND,0);

        // if we are past the given time of day, start tomorrow.
        if (c.getTimeInMillis() <= System.currentTimeMillis())
            c.add(Calendar.DAY_OF_YEAR,1);
            
        Date date = c.getTime();
        
        log.info("Scheduling next garbage collector run at ["+date+"].");
        
        this.timer.schedule(new CleanupTask(),date);
    }
    
    public void start() {
        
        this.timer = new Timer();
       
        this.scheduleCleanupTask();
    }
    
    public void stop() {
        
        if (this.timer == null) return;
        
        this.timer.cancel();
        this.timer = null;
    }
    
    /**
     * @return the time of a day where to run the job in the format
     *          <code>hh:mm</code> or <code>h:mm"</code>.
     */
    public String getTimeOfDay() {
        return this.timeOfDay;
    }

    /**
     * @param timeOfDay the time of a day where to run the job in the format
     *          <code>hh:mm</code> or <code>h:mm"</code> to set.
     */
    public void setTimeOfDay(String timeOfDay) {
        
        String[] parts = timeOfDay.split(":",2);
        
        if (parts.length != 2)
            throw new IllegalArgumentException("Invalid time of day ["+timeOfDay+"] sepcified.");
        
        int hh = Integer.parseInt(parts[0]);
        int mm = Integer.parseInt(parts[1]);
        
        if (hh<0 || hh>23 || mm<0 || mm > 59)
            throw new IllegalArgumentException("Invalid time of day ["+timeOfDay+"] sepcified.");
            
        this.minutesOfDay = hh*60 + mm;
        this.timeOfDay = timeOfDay;
    }

    /**
     * @return the number of month to keep e-mails in the database.
     */
    public int getMonthsToKeep() {
        return this.monthsToKeep;
    }

    /**
     * @param monthsToKeep the number of month to keep e-mails in the database to set.
     */
    public void setMonthsToKeep(int monthsToKeep) {
        this.monthsToKeep = monthsToKeep;
    }

    /**
     * @return the mailService
     */
    public EMailService getMailService() {
        return this.mailService;
    }

    /**
     * @param mailService the mailService to set
     */
    public void setMailService(EMailService mailService) {
        this.mailService = mailService;
    }

    public void setSmsService(SMSService smsService) {
        this.smsService = smsService;
    }
    
}
