/***********************************************************
 * $Id$
 * 
 * FancyMail standalone OSGi server of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.fancymail.server.service.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import javax.activation.DataSource;

import org.clazzes.fancymail.server.api.EMailAddressDTO;
import org.clazzes.fancymail.server.api.EMailDTO;
import org.clazzes.fancymail.server.api.EMailReportDTO;
import org.clazzes.fancymail.server.api.EMailSenderDTO;
import org.clazzes.fancymail.server.api.FancyMailServerService;
import org.clazzes.fancymail.server.entities.EMail;
import org.clazzes.fancymail.server.entities.EMailRecipient;
import org.clazzes.fancymail.server.entities.EMailSender;
import org.clazzes.fancymail.server.service.EMailService;

/**
 * The implementation of the external fancy mail server API.
 */
public class FancyMailServerServiceImpl implements FancyMailServerService {

    private EMailService mailService;
    
    private static EMailSenderDTO buildSender(EMailSender sender) {
        
        EMailSenderDTO ret = new EMailSenderDTO();
        ret.setAddress(sender.getAddress());
        ret.setPersonalName(sender.getPersonalName());
        ret.setReplyTo(sender.getReplyTo());
        return ret;
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.api.FancyMailServerService#getSenderAddress(java.lang.String)
     */
    @Override
    public EMailSenderDTO getSenderAddress(String emailAddr) {

        EMailSender sender = this.mailService.getSenderByAddr(emailAddr);
        
        if (sender == null) return null;
        
        return buildSender(sender);
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.api.FancyMailServerService#getAllSenders()
     */
    @Override
    public List<EMailSenderDTO> getAllSenders() {
        
        List<EMailSender> senders = this.mailService.getAllEMailSenders();
        
        List<EMailSenderDTO> ret = new ArrayList<EMailSenderDTO>(senders.size());
        
        for (EMailSender sender : senders) {
            ret.add(buildSender(sender));
        }
         
        return ret;
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.api.FancyMailServerService#insertSender(org.clazzes.fancymail.server.api.EMailSenderDTO)
     */
    @Override
    public void insertSender(EMailSenderDTO sender) {
        
        this.mailService.insertSender(sender);
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.api.FancyMailServerService#updateSender(org.clazzes.fancymail.server.api.EMailSenderDTO)
     */
    @Override
    public void updateSender(EMailSenderDTO sender) {
        
        this.mailService.updateSender(sender);
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.api.FancyMailServerService#updateSenders(java.util.Collection)
     */
    @Override
    public void updateSenders(Collection<EMailSenderDTO> senders) {
        
        this.mailService.updateSenders(senders);
    }

    private static EMailAddressDTO buildAddress(EMailRecipient r) {
        
        EMailAddressDTO ret = new EMailAddressDTO();
        ret.setAddress(r.getAddress());
        ret.setPersonalName(r.getPersonalName());
        return ret;
    }
    
    private static EMailReportDTO buildReport(EMail email) {
        
        EMailReportDTO ret = new EMailReportDTO();
        ret.setSender(email.getSender().getAddress());
        ret.setSubject(email.getSubject());
        ret.setBody(email.getBody());
        ret.setCreated(new Date(email.getCreated().getTime()));
        if (email.getSentAt() != null)
            ret.setSentAt(new Date(email.getSentAt().getTime()));
        ret.setErrorCount(email.getErrorCount());
        ret.setLastErrorText(email.getLastErrorText());
        ret.setStatus(email.getTransmissionReport());
        
        List<EMailAddressDTO> to = null;
        List<EMailAddressDTO> cc = null;
        List<EMailAddressDTO> bcc = null;
        
        Collection<EMailRecipient> rec = email.getRecipients();
        
        for (EMailRecipient r : rec) {
         
            switch(r.getEntitlement()) {
            case EMailRecipient.RECIPIENT_TO:
                if (to == null) to = new ArrayList<EMailAddressDTO>();
                to.add(buildAddress(r));
                break;
            case EMailRecipient.RECIPIENT_CC:
                if (cc == null) cc = new ArrayList<EMailAddressDTO>();
                cc.add(buildAddress(r));
                break;
            case EMailRecipient.RECIPIENT_BCC:
                if (bcc == null) bcc = new ArrayList<EMailAddressDTO>();
                bcc.add(buildAddress(r));
                break;              
            }
        }
        
        ret.setTo(to);
        ret.setCc(cc);
        ret.setBcc(bcc);
        
        return ret;
    }
    
    private static List<EMailReportDTO> buildReportList(List<EMail> emails) {

        List<EMailReportDTO> ret = new ArrayList<EMailReportDTO>();
        
        for (EMail email:emails)
            ret.add(buildReport(email));
        
        return ret;
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.api.FancyMailServerService#queueMail(org.clazzes.fancymail.server.api.EMailDTO)
     */
    @Override
    public Date queueMail(EMailDTO email) {
        
        EMail mail = this.mailService.insertEMail(email);
        
        return mail.getCreated();
    }
    
    @Override
    public Date queueMailWithAttachments(EMailDTO email,
            List<DataSource> attachments) {
        
        EMail mail = this.mailService.insertEMailWithAttachments(email, attachments);
        
        return mail.getCreated();
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.api.FancyMailServerService#getAllEMails(java.util.Date, java.util.Date, java.lang.String, java.lang.String)
     */
    @Override
    public List<EMailReportDTO> getAllEMails(Date from, Date to,
            String senderAddr, String recipientAddr) {

        if (senderAddr == null) {
        
            if (recipientAddr == null)
                return buildReportList(this.mailService.getAllEMailsInRange(from,to));
            else
                return buildReportList(this.mailService.getAllEMailsInRangeForRecipient(recipientAddr,from,to));
        }
        else {
            
            if (recipientAddr == null)
                return buildReportList(this.mailService.getAllEMailsInRangeForSender(senderAddr,from,to));
            else
                return buildReportList(this.mailService.getAllEMailsInRangeForSenderRecipient(senderAddr,recipientAddr,from,to));

        }
    }
    /**
     * @return the mailService
     */
    public EMailService getMailService() {
        return this.mailService;
    }

    /**
     * @param mailService the mailService to set
     */
    public void setMailService(EMailService mailService) {
        this.mailService = mailService;
    }

}
