/***********************************************************
 * $Id$
 *
 * FancyMail standalone OSGi server of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.fancymail.server.service.impl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import javax.activation.DataSource;

import org.clazzes.fancymail.mail.EMailException;
import org.clazzes.fancymail.mail.IEMail;
import org.clazzes.fancymail.server.api.EMailAddressDTO;
import org.clazzes.fancymail.server.api.EMailDTO;
import org.clazzes.fancymail.server.api.EMailSenderDTO;
import org.clazzes.fancymail.server.dao.EMailAttachmentDAO;
import org.clazzes.fancymail.server.dao.EMailDAO;
import org.clazzes.fancymail.server.dao.EMailSenderDAO;
import org.clazzes.fancymail.server.entities.EMail;
import org.clazzes.fancymail.server.entities.EMailAttachment;
import org.clazzes.fancymail.server.entities.EMailRecipient;
import org.clazzes.fancymail.server.entities.EMailSender;
import org.clazzes.fancymail.server.service.EMailService;
import org.clazzes.util.aop.DAOException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An implementation of the internal email service interface.
 */
public class EMailServiceImpl implements EMailService {

    private final Logger log = LoggerFactory.getLogger(EMailServiceImpl.class);

    private EMailSenderDAO mailSenderDAO;
    private EMailAttachmentDAO mailAttachmentDAO;
    private EMailDAO mailDAO;
    private int maxBulkSize;
    private int maxRetries;
    private int staleTimeoutMinutes;

    public EMailServiceImpl() {
        this.maxBulkSize = 10;
        this.maxRetries = 16;
        this.staleTimeoutMinutes = 15;
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.service.EMailService#insertSender(org.clazzes.fancymail.server.api.EMailSenderDTO)
     */
    @Override
    public EMailSender insertSender(EMailSenderDTO sender) {

        EMailSender entity = new EMailSender();

        entity.setAddress(sender.getAddress());
        entity.setPersonalName(sender.getPersonalName());
        entity.setReplyTo(sender.getReplyTo());

        return this.mailSenderDAO.save(entity);
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.service.EMailService#updateSender(org.clazzes.fancymail.server.api.EMailSenderDTO)
     */
    @Override
    public EMailSender updateSender(EMailSenderDTO sender) {

        EMailSender entity = this.mailSenderDAO.getByAddress(sender.getAddress());
        entity.setPersonalName(sender.getPersonalName());
        entity.setReplyTo(sender.getReplyTo());
        this.mailSenderDAO.update(entity);
        return entity;
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.service.EMailService#updateSenders(java.util.Collection)
     */
    @Override
    public void updateSenders(Collection<EMailSenderDTO> senders) {

        for (EMailSenderDTO sender:senders)
            this.updateSender(sender);
    }

    @Override
    public EMailSender getSenderByAddr(String emailAddr) {

        return this.mailSenderDAO.getByAddress(emailAddr);
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.service.EMailService#getAllEMailSenders()
     */
    @Override
    public List<EMailSender> getAllEMailSenders() {

        return this.mailSenderDAO.getAll();
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.sending.IEMailFactory#getMailsToSend()
     */
    @Override
    public Collection<EMail> getMailsToSend() throws EMailException {

        List<EMail> ret;

        if (this.staleTimeoutMinutes >0) {

            ret = this.mailDAO.getAllByStatusForUpdate(EMail.MAIL_STATUS_TOSEND,this.maxBulkSize,this.mailAttachmentDAO);

            if (ret == null || ret.size() <= 0) {
                ret = this.mailDAO.getAllByStatusForUpdate(EMail.MAIL_STATUS_SENDERROR,this.maxBulkSize,this.mailAttachmentDAO);
            }

            Date now = new Date();

            if (ret == null || ret.size() <= 0) {

                Date watermark = new Date(now.getTime()-this.staleTimeoutMinutes*60000L);

                ret = this.mailDAO.getStaleSending(
                            watermark,
                            this.maxBulkSize,this.mailAttachmentDAO);

                if (ret.size() > 0) {
                    log.warn("Found [{}] stale e-mail before [{}] to reprocess.",
                             ret.size(),watermark);
                }
            }

            ret.stream().forEach(mail -> mail.setSendingState(now));

            this.mailDAO.updateBatch(ret);
        }
        else {

            ret = this.mailDAO.getAllByStatus(EMail.MAIL_STATUS_TOSEND,this.maxBulkSize,this.mailAttachmentDAO);

            if (ret == null || ret.size() <= 0) {
                ret = this.mailDAO.getAllByStatus(EMail.MAIL_STATUS_SENDERROR,this.maxBulkSize,this.mailAttachmentDAO);
            }
        }

        if (ret.size() > 0) {
            log.info(ret
                 .stream()
                 .map(n -> String.valueOf(n))
                 .collect(Collectors.joining("\n  ", "Emails to process are [", "]")));
        } else if (log.isDebugEnabled()) {
            log.debug("Found no Emails to process");
        }

        return ret;
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.sending.IEMailReportConsumer#reportMailTransmissionAttempt(org.clazzes.fancymail.mail.IEMail)
     */
    @Override
    public void reportMailTransmissionAttempt(IEMail eMail) {

        EMail mail = (EMail)eMail;

        if (mail.getStatus() == EMail.MAIL_STATUS_SENDERROR &&
                mail.getErrorCount() > this.maxRetries)
            mail.setStatus(EMail.MAIL_STATUS_FAILED);

        this.mailDAO.update(mail);
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.service.EMailService#insertEMail(org.clazzes.fancymail.server.entities.EMail)
     */
    @Override
    public EMail insertEMail(EMailDTO email) {

        EMail entity = new EMail();
        entity.setSender(this.getSenderByAddr(email.getSender()));
        entity.setCreated(new Date());
        entity.setSubject(email.getSubject());
        entity.setBody(email.getBody());
        entity.setStatus(EMail.MAIL_STATUS_TOSEND);

        List<EMailRecipient> recipients = new ArrayList<EMailRecipient>();

        if (email.getTo() != null) {

            for (EMailAddressDTO addr : email.getTo()) {

                EMailRecipient recipient = new EMailRecipient();
                recipient.setEntitlement(EMailRecipient.RECIPIENT_TO);
                recipient.setAddress(addr.getAddress());
                recipient.setPersonalName(addr.getPersonalName());
                recipient.setEmailId(entity.getId());
                recipients.add(recipient);
            }
        }

        if (email.getCc() != null) {

            for (EMailAddressDTO addr : email.getCc()) {

                EMailRecipient recipient = new EMailRecipient();
                recipient.setEntitlement(EMailRecipient.RECIPIENT_CC);
                recipient.setAddress(addr.getAddress());
                recipient.setPersonalName(addr.getPersonalName());
                recipient.setEmailId(entity.getId());
                recipients.add(recipient);
            }
        }

       if (email.getBcc() != null) {

            for (EMailAddressDTO addr : email.getBcc()) {

                EMailRecipient recipient = new EMailRecipient();
                recipient.setEntitlement(EMailRecipient.RECIPIENT_BCC);
                recipient.setAddress(addr.getAddress());
                recipient.setPersonalName(addr.getPersonalName());
                recipient.setEmailId(entity.getId());
                recipients.add(recipient);
            }
        }

       entity.setRecipients(recipients);
       return this.mailDAO.save(entity);
    }

    @Override
    public EMail insertEMailWithAttachments(EMailDTO email,
            List<DataSource> attachments) {

        EMail mail = this.insertEMail(email);

        if (attachments != null) {

            for (DataSource ds : attachments) {

                ByteArrayOutputStream bos = new ByteArrayOutputStream();

                try (InputStream is = ds.getInputStream()) {

                    byte[] buf = new byte[16384];
                    int n;

                    while ((n=is.read(buf)) > 0) {
                        bos.write(buf,0,n);
                    }

                } catch (IOException e) {
                    throw new DAOException("Unable to copy mail attachment contents",e);
                }

                EMailAttachment att = new EMailAttachment();

                att.setContent(bos.toByteArray());

                att.setMail(mail);
                att.setPrettyName(ds.getName());
                att.setMimeType(ds.getContentType());

                this.mailAttachmentDAO.save(att);
            }
        }

        return mail;
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.service.EMailService#deleteOutdatedEMails(java.util.Date)
     */
    @Override
    public int deleteOutdatedEMails(Date watermark) {

        List<Long> ids = this.mailDAO.getOutdatedIds(watermark);

        this.mailAttachmentDAO.deleteAllForEMails(ids);
        this.mailDAO.deleteByIds(ids);

        return ids.size();
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.service.EMailService#getAllEMailsInRange(java.util.Date, java.util.Date)
     */
    @Override
    public List<EMail> getAllEMailsInRange(Date from, Date to) {

        return this.mailDAO.getAllInRange(from,to);
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.service.EMailService#getAllEMailsInRangeForSender(java.lang.String, java.util.Date, java.util.Date)
     */
    @Override
    public List<EMail> getAllEMailsInRangeForSender(String sender, Date from,
            Date to) {

        return this.mailDAO.getAllForSender(sender,from,to);
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.service.EMailService#getAllEMailsInRangeForRecipient(java.lang.String, java.util.Date, java.util.Date)
     */
    @Override
    public List<EMail> getAllEMailsInRangeForRecipient(String sender,
            Date from, Date to) {

        return this.mailDAO.getAllForRecipient(sender,from,to);
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.service.EMailService#getAllEMailsInRangeForSenderRecipient(java.lang.String, java.lang.String, java.util.Date, java.util.Date)
     */
    @Override
    public List<EMail> getAllEMailsInRangeForSenderRecipient(String senderAddr,
            String recipientAddr, Date from, Date to) {
        return this.mailDAO.getAllForSenderRecipient(senderAddr,recipientAddr,from,to);
    }

    /**
     * @return the mailDAO
     */
    public EMailDAO getMailDAO() {
        return this.mailDAO;
    }

    /**
     * @param mailDAO the mailDAO to set
     */
    public void setMailDAO(EMailDAO mailDAO) {
        this.mailDAO = mailDAO;
    }

    /**
     * @return the mailSenderDAO
     */
    public EMailSenderDAO getMailSenderDAO() {
        return this.mailSenderDAO;
    }


    /**
     * @param mailSenderDAO the mailSenderDAO to set
     */
    public void setMailSenderDAO(EMailSenderDAO mailSenderDAO) {
        this.mailSenderDAO = mailSenderDAO;
    }


    public EMailAttachmentDAO getMailAttachmentDAO() {
        return this.mailAttachmentDAO;
    }

    public void setMailAttachmentDAO(EMailAttachmentDAO mailAttachmentDAO) {
        this.mailAttachmentDAO = mailAttachmentDAO;
    }

    /**
     * @return the maximal number of emails to pass to the sender thread at once.
     */
    public int getMaxBulkSize() {
        return this.maxBulkSize;
    }

    /**
     * @param maxBulkSize the maximal number of emails to pass to the sender thread
     *                    at once to set.
     */
    public void setMaxBulkSize(int maxBulkSize) {
        this.maxBulkSize = maxBulkSize;
    }

    /**
     * @return the maximal number of retries for mails which failed to be delivered.
     */
    public int getMaxRetries() {
        return this.maxRetries;
    }

    /**
     * @param maxRetries the maximal number of retries to set
     */
    public void setMaxRetries(int maxRetries) {
        this.maxRetries = maxRetries;
    }

    public int getStaleTimeoutMinutes() {
        return this.staleTimeoutMinutes;
    }

    public void setStaleTimeoutMinutes(int staleTimeoutMinutes) {
        this.staleTimeoutMinutes = staleTimeoutMinutes;
    }

}
