/***********************************************************
 * $Id$
 * 
 * FancyMail standalone OSGi server of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.fancymail.server.service.impl;

import java.io.UnsupportedEncodingException;

import javax.activation.DataHandler;
import javax.mail.Address;
import javax.mail.Message.RecipientType;
import javax.mail.MessagingException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.util.ByteArrayDataSource;

import org.clazzes.fancymail.mail.EMailException;
import org.clazzes.fancymail.mail.IEMail;
import org.clazzes.fancymail.mail.IEMail2MimeMessageResolver;
import org.clazzes.fancymail.server.entities.EMail;
import org.clazzes.fancymail.server.entities.EMailAttachment;
import org.clazzes.fancymail.server.entities.EMailRecipient;

/**
 * This implementation fills in our DB entities into a javax.mail.MimeMessage 
 */
public class EMail2MimeMessageResolverImpl implements
        IEMail2MimeMessageResolver {

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.mail.IEMail2MimeMessageResolver#fillMimeMessage(javax.mail.internet.MimeMessage, org.clazzes.fancymail.mail.IEMail)
     */
    @Override
    public void fillMimeMessage(MimeMessage mimeMessage, IEMail withEMail)
            throws EMailException, MessagingException {
        
        if (!(withEMail instanceof EMail))
            throw new EMailException("EMail2MimeMessageResolverImpl only understand entites of type ["+EMail.class+"].");
        
        EMail eMail = (EMail)withEMail;
        
        mimeMessage.setSubject(eMail.getSubject(),"UTF-8");
        try {
            
            mimeMessage.setSender(new InternetAddress(
                    eMail.getSender().getAddress(),
                    eMail.getSender().getPersonalName()));
            mimeMessage.setFrom(new InternetAddress(
                    eMail.getSender().getAddress(),
                    eMail.getSender().getPersonalName()));
             
        } catch (UnsupportedEncodingException e) {
            throw new EMailException("Unsupported ecnoding in e-mail sender ["+eMail.getSender()+"].");
        }
        
        if (eMail.getSender().getReplyTo() != null && eMail.getSender().getReplyTo().length() > 0)
            mimeMessage.setReplyTo(
                    new Address[]{new InternetAddress(eMail.getSender().getReplyTo())});
      
        for (EMailRecipient r : eMail.getRecipients()) {
            
            try {
                
                RecipientType type;
                
                switch(r.getEntitlement()) {
                
                case EMailRecipient.RECIPIENT_TO: type = RecipientType.TO; break;
                case EMailRecipient.RECIPIENT_CC: type = RecipientType.CC; break;
                case EMailRecipient.RECIPIENT_BCC: type = RecipientType.BCC; break;
                default: continue;
                }
                
                mimeMessage.addRecipient(type,new InternetAddress(
                        r.getAddress(),
                        r.getPersonalName()));
                  
            } catch (UnsupportedEncodingException e) {
                throw new EMailException("Unsupported ecnoding in e-mail recipient ["+r+"].");
            }
        }
        
        mimeMessage.setSentDate(eMail.getCreated());
        mimeMessage.setSubject(eMail.getSubject(),"UTF-8");
        
        if (eMail.getAttachments() == null) {
            mimeMessage.setText(eMail.getBody(),"UTF-8");
        }
        else {
            
            MimeMultipart multi = new MimeMultipart();
            
            MimeBodyPart body = new MimeBodyPart();
            
            body.setText(eMail.getBody(),"UTF-8");
            
            multi.addBodyPart(body);
            
            for (EMailAttachment att : eMail.getAttachments()) {
                
                MimeBodyPart part = new MimeBodyPart();
                
                part.setFileName(att.getPrettyName());
                ByteArrayDataSource ds = new ByteArrayDataSource(att.getContent(),att.getMimeType());
                ds.setName(att.getPrettyName());
                DataHandler dh = new DataHandler(ds);
                part.setDataHandler(dh);
                
                multi.addBodyPart(part);
            }
            
            mimeMessage.setContent(multi);
        }

        // trying to make sure every header field is set, hoping to get nice From header after all:
        mimeMessage.saveChanges();

    }

}
