/***********************************************************
 * $Id$
 *
 * FancyMail, a Clazzes.org project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.fancymail.server.entities;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.clazzes.fancymail.mail.IEMail;

/**
 * A persisted email to send.
 */
public class EMail implements IEMail, Serializable {

    private static final long serialVersionUID = -3208549379134568740L;

    public static final int MAIL_STATUS_UNKNOWN = 0;
    public static final int MAIL_STATUS_TOSEND = 1;
    public static final int MAIL_STATUS_SENT = 2;
    public static final int MAIL_STATUS_SENDERROR = 3;
    // maximal number of retries are exceeded.
    public static final int MAIL_STATUS_FAILED = 4;
    // currently being processed.
    public static final int MAIL_STATUS_SENDING = 99;

    private Long id;
    private String subject;
    private int status;
    private int errorCount;
    private String lastErrorText;
    private Date sentAt;
    private String body;
    private EMailSender sender;
    private Date created;
    private Date sending;
    private String lastErrorException;

    private List<EMailRecipient> recipients;

    private List<EMailAttachment> attachments;

    public EMail() {
        this.status = MAIL_STATUS_UNKNOWN;
        this.lastErrorText = "";
    }

    @Override
    public void setSent(String successDesc) {

        this.setStatus(MAIL_STATUS_SENT);
        this.setLastErrorText(successDesc);
        this.setSentAt(new Date());
        this.setSending(null);
        this.setLastErrorException(null);
    }

    @Override
    public void setUnsent(String errorDesc) {

        this.setErrorCount(this.getErrorCount()+1);
        this.setStatus(MAIL_STATUS_SENDERROR);
        this.setLastErrorText(errorDesc);
        this.setSending(null);
        this.setLastErrorException(null);
    }

    @Override
    public void setUnsent(String errorDesc, Exception e) {

        this.setErrorCount(this.getErrorCount()+1);
        this.setStatus(MAIL_STATUS_SENDERROR);
        this.setLastErrorText(errorDesc);
        this.setSending(null);
        this.setLastErrorException(e.getClass().getName());
    }

    public void setSendingState(Date now) {

        this.setStatus(MAIL_STATUS_SENDING);
        this.setLastErrorText("Sending...");
        this.setSending(now);
        this.setLastErrorException(null);
    }

    @Override
    public boolean isSent() {

        return this.getStatus() == MAIL_STATUS_SENT;
    }

    @Override
    public String getTransmissionReport() {

        switch (this.getStatus())
        {
        case MAIL_STATUS_SENT:
            return "Sent OK";
        case MAIL_STATUS_SENDERROR:
            return "NOT SENT! Error: " + this.getLastErrorText();
        case MAIL_STATUS_FAILED:
            return "NOT SENT! Maximal number of retries exceeded.";
        case MAIL_STATUS_SENDING:
            return "Currently being sent...";
        default:
            return "Not yet transmitted.";
        }
    }

    /**
     * @return the id
     */
    public Long getId() {
        return this.id;
    }

    /**
     * @param id the id to set
     */
    public void setId(Long id) {
        this.id = id;
    }

    /**
     * @return the list of recipients
     */
    public List<EMailRecipient> getRecipients() {
        return this.recipients;
    }

    /**
     * @param recipients the list recipients to set
     */
    public void setRecipients(List<EMailRecipient> recipients) {
        this.recipients = recipients;
    }

    /**
     * @param recipient A recipient to add.
     */
    public void addRecipient(EMailRecipient recipient) {

        if (this.recipients == null) {
            this.recipients = new ArrayList<EMailRecipient>();
        }

        this.recipients.add(recipient);
    }

    /**
     * @return The list of attachments, if they have been fetched from the DB-
     */
    public List<EMailAttachment> getAttachments() {
        return this.attachments;
    }

    /**
     * @param attachments The list of attachments to set.
     */
    public void setAttachments(List<EMailAttachment> attachments) {
        this.attachments = attachments;
    }

    /**
     * @param attachment The attachment to add to the list of attachments.
     */
    public void addAttachment(EMailAttachment attachment) {

        if (this.attachments == null) {
            this.attachments = new ArrayList<EMailAttachment>();
        }

        this.attachments.add(attachment);
    }

    /**
     * @return the status
     * @see EMail#MAIL_STATUS_SENDERROR
     * @see EMail#MAIL_STATUS_SENT
     * @see EMail#MAIL_STATUS_UNKNOWN
     * @see EMail#MAIL_STATUS_TOSEND
     * @see EMail#MAIL_STATUS_FAILED
     * @see EMail#MAIL_STATUS_SENDING
     */
    public int getStatus() {
        return this.status;
    }

    /**
     * @param status the status to set
     * @see EMail#MAIL_STATUS_SENDERROR
     * @see EMail#MAIL_STATUS_SENT
     * @see EMail#MAIL_STATUS_UNKNOWN
     * @see EMail#MAIL_STATUS_TOSEND
     * @see EMail#MAIL_STATUS_FAILED
     * @see EMail#MAIL_STATUS_SENDING
     */
    public void setStatus(int status) {
        this.status = status;
    }

    /**
     * @return the errorCount
     */
    public int getErrorCount() {
        return this.errorCount;
    }

    /**
     * @param errorCount the errorCount to set
     */
    public void setErrorCount(int errorCount) {
        this.errorCount = errorCount;
    }

    /**
     * @return the lastErrorText
     */
    public String getLastErrorText() {
        return this.lastErrorText;
    }

    /**
     * @param lastErrorText the lastErrorText to set
     */
    public void setLastErrorText(String lastErrorText) {
        this.lastErrorText = lastErrorText;
    }

    /**
     * @return the sentAt
     */
    public Date getSentAt() {
        return this.sentAt;
    }

    /**
     * @param sentAt the sentAt to set
     */
    public void setSentAt(Date sentAt) {
        this.sentAt = sentAt;
    }

    /**
     * @return the body
     */
    public String getBody() {
        return this.body;
    }

    /**
     * @param body the body to set
     */
    public void setBody(String body) {
        this.body = body;
    }

    /**
     * @return the sender
     */
    public EMailSender getSender() {
        return this.sender;
    }

    /**
     * @param sender the sender to set
     */
    public void setSender(EMailSender sender) {
        this.sender = sender;
    }

    /**
     * @return the created
     */
    public Date getCreated() {
        return this.created;
    }

    /**
     * @param created the created to set
     */
    public void setCreated(Date created) {
        this.created = created;
    }

    @Override
    public String getSubject() {

        return this.subject;
    }

    /**
     * @param subject the subject to set
     */
    public void setSubject(String subject) {
        this.subject = subject;
    }

    public Date getSending() {
        return this.sending;
    }

    public void setSending(Date sending) {
        this.sending = sending;
    }

    public String getLastErrorException() {
        return this.lastErrorException;
    }

    public void setLastErrorException(String lastErrorException) {
        this.lastErrorException = lastErrorException;
    }

    @Override
    public String toString() {
        return "EMail [id=" + this.id +
               ", created=" + this.created +
               ", subject=" + this.subject +
               ", errorCount=" + this.errorCount +
               ", recipients=" + this.recipients + "]";
    }

}
