package org.clazzes.fancymail.server.dao.jdbc;

import java.io.Serializable;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Map;

import org.clazzes.fancymail.server.dao.SMSSenderDAO;
import org.clazzes.fancymail.server.entities.SMSSender;
import org.clazzes.util.aop.ThreadLocalManager;
import org.clazzes.util.sql.criteria.SQLCondition;
import org.clazzes.util.sql.criteria.SQLValue;
import org.clazzes.util.sql.dao.AbstrIdDAO;
import org.clazzes.util.sql.dao.StatementPreparer;
import org.clazzes.util.sql.helper.JDBCHelper;

public class JdbcSMSSenderDAO extends AbstrIdDAO<SMSSender> implements SMSSenderDAO {

    public JdbcSMSSenderDAO() {
        super(SMSSender.class,"id","SMS_SENDER",new String[]{
            "ID",
            "SOURCE_NUMBER",
            "PERSONAL_NAME"
        });
    }
    
    @Override
    public SMSSender get(Serializable id) {
        
        Map<Long,SMSSender> senderCache =
                ThreadLocalManager.getBoundResource(SMSSenderCacheInterceptor.SENDER_CACHE_KEY);
        
        if (senderCache != null) {
            SMSSender ret = senderCache.get(id);
            if (ret != null) {
                return ret;
            }
        }

        return super.get(id);
    }

    public SMSSender getBySourceNumber(final String sourceNumber) {
        SQLCondition cond = SQLCondition.eq(
                SQLValue.tableColumn(this.getTableName(),"SOURCE_NUMBER"),
                SQLValue.INSERT_VALUE);
        
        return this.getUniqueWithCondition(cond,new StatementPreparer() {
            
            @Override
            public void fillInsertValues(PreparedStatement statement)
                    throws SQLException {
               statement.setString(1, sourceNumber);
            }
        });
    }

    protected SMSSender fillDtoFromResultSet(ResultSet rs) throws SQLException {
        SMSSender ret = new SMSSender();
        
        ret.setId(JDBCHelper.getLong(rs,1));
        ret.setSourceNumber(rs.getString(2));
        ret.setPersonalName(rs.getString(3));
        
        Map<Long,SMSSender> senderCache =
                ThreadLocalManager.getBoundResource(SMSSenderCacheInterceptor.SENDER_CACHE_KEY);
        
        if (senderCache != null) {
            senderCache.put(ret.getId(),ret);
        }
        
        return ret;
    }

    protected void fillPreparedStatementFromDto(PreparedStatement statement, SMSSender dto) throws SQLException {
        JDBCHelper.setLong(statement,1,dto.getId());
        statement.setString(2, dto.getSourceNumber());
        statement.setString(3, dto.getPersonalName());
    }
}
