package org.clazzes.fancymail.server.dao.jdbc;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.clazzes.fancymail.server.dao.SMSDAO;
import org.clazzes.fancymail.server.dao.SMSDestinationDAO;
import org.clazzes.fancymail.server.dao.SMSSenderDAO;
import org.clazzes.fancymail.server.entities.EMail;
import org.clazzes.fancymail.server.entities.SMS;
import org.clazzes.fancymail.server.entities.SMSRecipient;
import org.clazzes.fancymail.server.entities.SMSSender;
import org.clazzes.util.aop.jdbc.JdbcPreparedStatementAction;
import org.clazzes.util.sql.criteria.SQLCondition;
import org.clazzes.util.sql.criteria.SQLValue;
import org.clazzes.util.sql.dao.AbstrIdDAO;
import org.clazzes.util.sql.dao.StatementPreparer;
import org.clazzes.util.sql.helper.JDBCHelper;

public class JdbcSMSDAO extends AbstrIdDAO<SMS> implements SMSDAO {

    private SMSDestinationDAO destinationDAO;
    public void setDestinationDAO(SMSDestinationDAO destinationDAO) {
        this.destinationDAO = destinationDAO;
    }



    private SMSSenderDAO senderDAO;
    public void setSenderDAO(SMSSenderDAO senderDAO) {
        this.senderDAO = senderDAO;
    }

    public JdbcSMSDAO() {
        super(SMS.class,"id","SMS",new String[]{
            "ID",
            "SENDER_ID",
            "CREATED",
            "SENT",
            "TEXT",
            "STATUS",
            "ERROR_COUNT",
            "LAST_ERROR_TEXT",
            "SENDING",
            "LAST_ERROR_EXCEPTION"
        });
    }

    public List<SMS> getAllByStatus(final int status, final int maxCount) {

        SQLCondition cond = SQLCondition.eq(SQLValue.tableColumn(this.getTableName(),"STATUS"),SQLValue.INSERT_VALUE);

        final StatementPreparer preparer = new StatementPreparer() {

            @Override
            public void fillInsertValues(PreparedStatement statement)
                    throws SQLException {
                statement.setInt(1,status);
                statement.setMaxRows(maxCount);
            }
        };

        return this.getSMSList(cond,preparer);
    }

    @Override
    public List<SMS> getAllByStatusForUpdate(int status, int maxCount) {

        SQLCondition cond = SQLCondition.eq(SQLValue.tableColumn(this.getTableName(),"STATUS"),SQLValue.INSERT_VALUE);

        final StatementPreparer preparer = new StatementPreparer() {

            @Override
            public void fillInsertValues(PreparedStatement statement)
                    throws SQLException {
                statement.setInt(1,status);
                statement.setMaxRows(maxCount);
            }
        };

        return this.getSMSListForUpdate(cond,preparer);
    }

    @Override
    public List<SMS> getStaleSending(Date before, int maxCount) {

        SQLCondition cond =
            SQLCondition.and(
                SQLCondition.eq(SQLValue.tableColumn(this.getTableName(),"STATUS"),SQLValue.integerValue(EMail.MAIL_STATUS_SENDING)),
                SQLCondition.ltEquals(SQLValue.tableColumn(this.getTableName(),"SENDING"),SQLValue.INSERT_VALUE));

        final StatementPreparer preparer = new StatementPreparer() {

            @Override
            public void fillInsertValues(PreparedStatement statement)
                    throws SQLException {
                statement.setTimestamp(1,new java.sql.Timestamp(before.getTime()));
                statement.setMaxRows(maxCount);
            }
        };

        return this.getSMSListForUpdate(cond,preparer);
    }

    protected List<SMS> getSMSList(SQLCondition condition, StatementPreparer preparer) {

        List<SMS> sms = this.getListWithCondition(condition,preparer);

        this.destinationDAO.fetchAllForSMS(sms,condition,preparer);
        return sms;
    }

    protected List<SMS> getSMSListForUpdate(SQLCondition condition, StatementPreparer preparer) {

        List<SMS> sms = this.getListWithConditionForUpdate(condition,preparer);

        this.destinationDAO.fetchAllForSMS(sms,condition,preparer);
        return sms;
    }

    public SMS save(SMS dto) {

        SMS ret = super.save(dto);

        if (ret.getRecipients() != null) {

            for (SMSRecipient r : ret.getRecipients()) {
                r.setSmsId(ret.getId());
            }

            this.destinationDAO.saveBatch(ret.getRecipients());
        }

        return ret;
    }

    @Override
    protected SMS fillDtoFromResultSet(ResultSet rs) throws SQLException {
        SMS ret = new SMS();

        ret.setId(JDBCHelper.getLong(rs,1));
        ret.setSender(this.senderDAO.get(JDBCHelper.getLong(rs,2)));
        ret.setCreated(rs.getTimestamp(3));
        ret.setSentAt(rs.getTimestamp(4));
        ret.setText(rs.getString(5));
        ret.setStatus(rs.getInt(6));
        ret.setErrorCount(rs.getInt(7));
        ret.setLastErrorText(rs.getString(8));
        ret.setSending(rs.getTimestamp(9));
        ret.setLastErrorException(rs.getString(10));

        return ret;
    }

    @Override
    protected void fillPreparedStatementFromDto(PreparedStatement statement,
            SMS dto) throws SQLException {

        JDBCHelper.setLong(statement,1,dto.getId());
        JDBCHelper.setLong(statement,2,dto.getSender() == null ? null : dto.getSender().getId());
        statement.setTimestamp(3,dto.getCreated() == null ? null : new Timestamp(dto.getCreated().getTime()));
        statement.setTimestamp(4,dto.getSentAt() == null ? null : new Timestamp(dto.getSentAt().getTime()));
        statement.setString(5,dto.getText());
        statement.setInt(6,dto.getStatus());
        statement.setInt(7,dto.getErrorCount());
        statement.setString(8,dto.getLastErrorText());
        statement.setTimestamp(9,dto.getSending() == null ? null : new Timestamp(dto.getSending().getTime()));
        statement.setString(10,dto.getLastErrorException());
    }

    protected SQLValue createdColumn() {
        return SQLValue.tableColumn(this.getTableName(),"CREATED");
    }

    protected SQLCondition createdFrom() {
        return SQLCondition.gtEquals(this.createdColumn(),SQLValue.INSERT_VALUE);
    }

    protected SQLCondition createdTo() {
        return SQLCondition.ltEquals(this.createdColumn(),SQLValue.INSERT_VALUE);
    }

    public List<SMS> getAllInRange(final Date from, final Date to) {
        SQLCondition cond;

        if (from == null) {

            if (to == null) {
                cond = null;
            } else {
                cond = this.createdTo();
            }
        }
        else {

            if (to == null) {
                cond = this.createdFrom();
            } else {
                cond = SQLCondition.and(this.createdFrom(),this.createdTo());
            }
        }

        StatementPreparer preparer = new StatementPreparer() {

            @Override
            public void fillInsertValues(PreparedStatement statement)
                    throws SQLException {
                int i = 0;
                if (from != null) {
                    statement.setDate(++i,new java.sql.Date(from.getTime()));
                }
                if (to != null) {
                    statement.setDate(++i,new java.sql.Date(to.getTime()));
                }
            }
        };

        return this.getSMSList(cond,preparer);
    }

    public List<SMS> getAllForRecipient(final String recipient, final Date from, final Date to) {
        SQLCondition cond = SQLCondition.eq(SQLValue.tableColumn("SMS_RECIPIENT","DEST_NUMBER"),SQLValue.INSERT_VALUE);

        if (from == null) {
            if (to != null) {
                cond = SQLCondition.and(cond,this.createdTo());
            }
        } else {
            if (to == null) {
                cond = SQLCondition.and(cond,this.createdFrom());
            }
            else {
                cond = SQLCondition.and(cond,this.createdFrom(),this.createdTo());
            }
        }

        final StatementPreparer preparer = new StatementPreparer() {
             @Override
            public void fillInsertValues(PreparedStatement statement) throws SQLException {
                int i = 0;
                statement.setString(++i,recipient);
                if (from != null) {
                    statement.setDate(++i,new java.sql.Date(from.getTime()));
                }
                if (to != null) {
                    statement.setDate(++i,new java.sql.Date(to.getTime()));
                }
            }
        };

        String sql = this.getGenerator().innerJoin(
                this.getTableName(),
                "SMS_RECIPIENT",
                SQLCondition.eq(
                        SQLValue.tableColumn("SMS_RECIPIENT","SMS_ID"),
                        SQLValue.tableColumn(this.getTableName(),"ID")),
                cond,
                SQLValue.columnList(this.getTableName(),this.getColumnNames()));

        List<SMS> ret = this.performWithPreparedStatement(sql,new JdbcPreparedStatementAction<List<SMS>>() {

            @Override
            public List<SMS> perform(PreparedStatement statement)
                    throws Exception {

                preparer.fillInsertValues(statement);

                ResultSet rs = statement.executeQuery();

                List<SMS> r = new ArrayList<SMS>();

                while (rs.next()) {
                    r.add(JdbcSMSDAO.this.fillDtoFromResultSet(rs));
                }

                return r;
            }

        });

        this.destinationDAO.fetchAllForSMS(ret, cond, preparer);

        return ret;
    }

    public List<SMS> getAllForSender(final String sender, final Date from, final Date to) {
        final SMSSender senderDTO = this.senderDAO.getBySourceNumber(sender);

        if (senderDTO == null) {
            return null;
        }

        SQLCondition cond = SQLCondition.eq(SQLValue.tableColumn(this.getTableName(),"SENDER_ID"),SQLValue.INSERT_VALUE);

        if (from == null) {
            if (to != null) {
                cond = SQLCondition.and(cond,this.createdTo());
            }
        } else {
            if (to == null) {
                cond = SQLCondition.and(cond,this.createdFrom());
            }
            else {
                cond = SQLCondition.and(cond,this.createdFrom(),this.createdTo());
            }
        }

        final StatementPreparer preparer = new StatementPreparer() {

            @Override
            public void fillInsertValues(PreparedStatement statement) throws SQLException {
                int i = 0;
                JDBCHelper.setLong(statement,++i,senderDTO.getId());
                if (from != null) {
                    statement.setDate(++i,new java.sql.Date(from.getTime()));
                }
                if (to != null) {
                    statement.setDate(++i,new java.sql.Date(to.getTime()));
                }
            }
        };

        return this.getSMSList(cond,preparer);
    }

    public List<SMS> getAllForSenderRecipient(final String sender, final String recipient, final Date from, final Date to) {
        final SMSSender senderDTO = this.senderDAO.getBySourceNumber(sender);

        if (senderDTO == null) {
            return null;
        }

        SQLCondition cond;
        SQLCondition cond1 = SQLCondition.eq(SQLValue.tableColumn(this.getTableName(),"SENDER_ID"),SQLValue.INSERT_VALUE);
        SQLCondition cond2 = SQLCondition.eq(SQLValue.tableColumn("SMS_RECIPIENT","DEST_NUMBER"),SQLValue.INSERT_VALUE);

        if (from == null) {
            if (to == null) {
                cond = SQLCondition.and(cond1,cond2);
            }
            else {
                cond = SQLCondition.and(cond1,cond2,this.createdTo());
            }
        } else {
            if (to == null) {
                cond = SQLCondition.and(cond1,cond2,this.createdFrom());
            }
            else {
                cond = SQLCondition.and(cond1,cond2,this.createdFrom(),this.createdTo());
            }
        }

        final StatementPreparer preparer = new StatementPreparer() {

            @Override
            public void fillInsertValues(PreparedStatement statement) throws SQLException {
                int i = 0;
                JDBCHelper.setLong(statement,++i,senderDTO.getId());
                statement.setString(++i,recipient);
                if (from != null) {
                    statement.setDate(++i,new java.sql.Date(from.getTime()));
                }
                if (to != null) {
                    statement.setDate(++i,new java.sql.Date(to.getTime()));
                }
            }
        };

        String sql = this.getGenerator().innerJoin(
                this.getTableName(),
                "SMS_RECIPIENT",
                SQLCondition.eq(
                        SQLValue.tableColumn("SMS_RECIPIENT","SMS_ID"),
                        SQLValue.tableColumn(this.getTableName(),"ID")),
                cond,
                SQLValue.columnList(this.getTableName(),this.getColumnNames()));

        List<SMS> ret = this.performWithPreparedStatement(sql,new JdbcPreparedStatementAction<List<SMS>>() {

            @Override
            public List<SMS> perform(PreparedStatement statement)
                    throws Exception {

                preparer.fillInsertValues(statement);

                ResultSet rs = statement.executeQuery();

                List<SMS> r = new ArrayList<SMS>();

                while (rs.next()) {
                    r.add(JdbcSMSDAO.this.fillDtoFromResultSet(rs));
                }

                return r;
            }

        });

        this.destinationDAO.fetchAllForSMS(ret,cond,preparer);

        return ret;
    }

    @Override
    public List<Long> getOutdatedIds(final Date to) {

        String sql = this.getGenerator().select(this.getTableName(),
                new SQLValue[]{SQLValue.tableColumn(this.getTableName(),"ID")},this.createdTo());

        List<Long> ids =
        this.performWithPreparedStatement(sql,new JdbcPreparedStatementAction<List<Long>>() {

            @Override
            public List<Long> perform(PreparedStatement statement) throws Exception {

                statement.setDate(1,new java.sql.Date(to.getTime()));

                ResultSet rs = statement.executeQuery();

                List<Long> r = new ArrayList<Long>();

                while (rs.next()) {
                    r.add(JDBCHelper.getLong(rs,1));
                }
                return r;
            }
        });

        return ids;
    }

    @Override
    public void deleteByIds(List<Long> ids) {

        this.destinationDAO.deleteAllForSMSs(ids);
        this.deleteBatch(ids);
    }

}
