/***********************************************************
 * $Id$
 * 
 * FancyMail standalone OSGi server of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.fancymail.server.dao.jdbc;

import java.io.Serializable;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Map;

import org.clazzes.fancymail.server.dao.EMailSenderDAO;
import org.clazzes.fancymail.server.entities.EMailSender;
import org.clazzes.util.aop.ThreadLocalManager;
import org.clazzes.util.sql.criteria.SQLCondition;
import org.clazzes.util.sql.criteria.SQLValue;
import org.clazzes.util.sql.dao.AbstrIdDAO;
import org.clazzes.util.sql.dao.StatementPreparer;
import org.clazzes.util.sql.helper.JDBCHelper;

/**
 * DAO implementation for {@link EMailSender} entities.
 */
public class JdbcEMailSenderDAO extends AbstrIdDAO<EMailSender> implements
        EMailSenderDAO {

    public JdbcEMailSenderDAO() {
        super(EMailSender.class,"id","EMAIL_SENDER",new String[]{
            "ID",
            "ADDRESS",
            "PERSONAL_NAME",
            "REPLY_TO"
        });
    }

    /* (non-Javadoc)
     * @see org.clazzes.fancymail.server.dao.EMailSenderDAO#getByAddress(java.lang.String)
     */
    @Override
    public EMailSender getByAddress(final String address) {
        
        SQLCondition cond = SQLCondition.eq(
                SQLValue.tableColumn(this.getTableName(),"ADDRESS"),
                SQLValue.INSERT_VALUE);
        
        return this.getUniqueWithCondition(cond,new StatementPreparer() {
            
            @Override
            public void fillInsertValues(PreparedStatement statement)
                    throws SQLException {
               statement.setString(1,address);
            }
        });
    }

    @Override
    public EMailSender get(Serializable id) {
        
        Map<Long,EMailSender> senderCache =
                ThreadLocalManager.getBoundResource(EMailSenderCacheInterceptor.SENDER_CACHE_KEY);

        if (senderCache != null) {
            
            EMailSender ret = senderCache.get(id);
            if (ret != null) {
                return ret;
            }
        }
        
        return super.get(id);
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sql.dao.AbstrBasicDAO#fillDtoFromResultSet(java.sql.ResultSet)
     */
    @Override
    protected EMailSender fillDtoFromResultSet(ResultSet rs)
            throws SQLException {
        
        EMailSender ret = new EMailSender();
        
        ret.setId(JDBCHelper.getLong(rs,1));
        ret.setAddress(rs.getString(2));
        ret.setPersonalName(rs.getString(3));
        ret.setReplyTo(rs.getString(4));
           
        Map<Long,EMailSender> senderCache =
                ThreadLocalManager.getBoundResource(EMailSenderCacheInterceptor.SENDER_CACHE_KEY);

        if (senderCache != null) {
            senderCache.put(ret.getId(),ret);
        }
        
        return ret;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sql.dao.AbstrBasicDAO#fillPreparedStatementFromDto(java.sql.PreparedStatement, java.lang.Object)
     */
    @Override
    protected void fillPreparedStatementFromDto(PreparedStatement statement,
            EMailSender dto) throws SQLException {
        
        JDBCHelper.setLong(statement,1,dto.getId());
        statement.setString(2,dto.getAddress());
        statement.setString(3,dto.getPersonalName());
        statement.setString(4,dto.getReplyTo());
    }

}
